import { z } from 'zod';
import { type APIClient } from '../api.ts';
import { type CreateWebhookRequest, type ListWebhooksRequest, type UpdateWebhookRequest, type Webhook, type WebhookApiOptions } from './types.ts';
export declare const WebhookResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        id: z.ZodString;
        created_at: z.ZodString;
        updated_at: z.ZodString;
        created_by: z.ZodString;
        name: z.ZodString;
        description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
        url: z.ZodOptional<z.ZodString>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const WebhooksListResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodArray<z.ZodObject<{
        id: z.ZodString;
        created_at: z.ZodString;
        updated_at: z.ZodString;
        created_by: z.ZodString;
        name: z.ZodString;
        description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
        url: z.ZodOptional<z.ZodString>;
    }, z.core.$strip>>;
}, z.core.$strip>], "success">;
export declare const DeleteWebhookResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
}, z.core.$strip>], "success">;
/**
 * Create a new webhook.
 *
 * Creates a webhook with the specified name and optional description.
 *
 * @param client - The API client instance
 * @param params - Webhook creation parameters
 * @param options - Optional API options (e.g., orgId)
 * @returns The created webhook
 * @throws {WebhookError} If the API request fails
 *
 * @example
 * ```typescript
 * const webhook = await createWebhook(client, {
 *   name: 'github-events',
 *   description: 'Receives GitHub webhook events',
 * });
 * console.log(`Created webhook: ${webhook.id}`);
 * ```
 */
export declare function createWebhook(client: APIClient, params: CreateWebhookRequest, options?: WebhookApiOptions): Promise<Webhook>;
/**
 * Get a webhook by ID.
 *
 * Retrieves the webhook details.
 *
 * @param client - The API client instance
 * @param webhookId - The webhook ID (prefixed with wh_)
 * @param options - Optional API options (e.g., orgId)
 * @returns The webhook details
 * @throws {WebhookNotFoundError} If the webhook does not exist
 * @throws {WebhookError} If the API request fails
 *
 * @example
 * ```typescript
 * const webhook = await getWebhook(client, 'wh_abc123');
 * console.log(`Webhook: ${webhook.name}`);
 * ```
 */
export declare function getWebhook(client: APIClient, webhookId: string, options?: WebhookApiOptions): Promise<Webhook>;
/**
 * List all webhooks with optional pagination.
 *
 * @param client - The API client instance
 * @param params - Optional pagination parameters
 * @param options - Optional API options (e.g., orgId)
 * @returns Object containing the list of webhooks
 * @throws {WebhookError} If the API request fails
 *
 * @example
 * ```typescript
 * // List first 10 webhooks
 * const { webhooks } = await listWebhooks(client, { limit: 10 });
 * console.log(`Found ${webhooks.length} webhooks`);
 *
 * // Paginate through all webhooks
 * const { webhooks: page2 } = await listWebhooks(client, { limit: 10, offset: 10 });
 * ```
 */
export declare function listWebhooks(client: APIClient, params?: ListWebhooksRequest, options?: WebhookApiOptions): Promise<{
    webhooks: Webhook[];
}>;
/**
 * Update an existing webhook.
 *
 * Updates the webhook name and/or description.
 *
 * @param client - The API client instance
 * @param webhookId - The webhook ID (prefixed with wh_)
 * @param params - Update parameters
 * @param options - Optional API options (e.g., orgId)
 * @returns The updated webhook
 * @throws {WebhookNotFoundError} If the webhook does not exist
 * @throws {WebhookError} If the API request fails
 *
 * @example
 * ```typescript
 * const webhook = await updateWebhook(client, 'wh_abc123', {
 *   name: 'github-events-v2',
 *   description: 'Updated description',
 * });
 * ```
 */
export declare function updateWebhook(client: APIClient, webhookId: string, params: UpdateWebhookRequest, options?: WebhookApiOptions): Promise<Webhook>;
/**
 * Delete a webhook.
 *
 * Permanently deletes a webhook and all its destinations, receipts, and deliveries.
 * This action cannot be undone.
 *
 * @param client - The API client instance
 * @param webhookId - The webhook ID (prefixed with wh_)
 * @param options - Optional API options (e.g., orgId)
 * @throws {WebhookNotFoundError} If the webhook does not exist
 * @throws {WebhookError} If the API request fails
 *
 * @example
 * ```typescript
 * await deleteWebhook(client, 'wh_abc123');
 * console.log('Webhook deleted');
 * ```
 */
export declare function deleteWebhook(client: APIClient, webhookId: string, options?: WebhookApiOptions): Promise<void>;
//# sourceMappingURL=webhooks.d.ts.map