import { z } from 'zod';
/**
 * Webhook destination type schema.
 *
 * Currently only 'url' destinations are supported.
 *
 * @example
 * ```typescript
 * const destType = WebhookDestinationTypeSchema.parse('url');
 * ```
 */
export declare const WebhookDestinationTypeSchema: z.ZodEnum<{
    url: "url";
}>;
/**
 * Webhook destination type.
 */
export type WebhookDestinationType = z.infer<typeof WebhookDestinationTypeSchema>;
/**
 * Webhook delivery status schema.
 *
 * - `pending`: Delivery is queued and waiting to be sent.
 * - `success`: Delivery was completed successfully.
 * - `failed`: Delivery failed after all retry attempts.
 *
 * @example
 * ```typescript
 * const status = WebhookDeliveryStatusSchema.parse('success');
 * ```
 */
export declare const WebhookDeliveryStatusSchema: z.ZodEnum<{
    success: "success";
    pending: "pending";
    failed: "failed";
}>;
/**
 * Webhook delivery status type.
 */
export type WebhookDeliveryStatus = z.infer<typeof WebhookDeliveryStatusSchema>;
/**
 * Webhook schema representing a webhook endpoint.
 *
 * @example
 * ```typescript
 * const webhook = await getWebhook(client, 'wh_abc123');
 * console.log(`Webhook: ${webhook.name} (${webhook.id})`);
 * ```
 */
export declare const WebhookSchema: z.ZodObject<{
    id: z.ZodString;
    created_at: z.ZodString;
    updated_at: z.ZodString;
    created_by: z.ZodString;
    name: z.ZodString;
    description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    url: z.ZodOptional<z.ZodString>;
}, z.core.$strip>;
/**
 * Webhook type.
 */
export type Webhook = z.infer<typeof WebhookSchema>;
/**
 * Webhook destination schema representing a delivery target for webhook events.
 *
 * Destinations define where incoming webhook payloads are forwarded to.
 *
 * @example
 * ```typescript
 * const destinations = await listWebhookDestinations(client, 'wh_abc123');
 * for (const dest of destinations) {
 *   console.log(`Destination ${dest.id}: type=${dest.type}`);
 * }
 * ```
 */
export declare const WebhookDestinationSchema: z.ZodObject<{
    id: z.ZodString;
    created_at: z.ZodString;
    updated_at: z.ZodString;
    created_by: z.ZodString;
    webhook_id: z.ZodString;
    type: z.ZodEnum<{
        url: "url";
    }>;
    config: z.ZodRecord<z.ZodString, z.ZodUnknown>;
}, z.core.$strip>;
/**
 * Webhook destination type.
 */
export type WebhookDestination = z.infer<typeof WebhookDestinationSchema>;
/**
 * Webhook receipt schema representing an incoming webhook payload that was received.
 *
 * Receipts capture the raw payload and headers of each incoming webhook request.
 *
 * @example
 * ```typescript
 * const receipt = await getWebhookReceipt(client, 'wh_abc123', 'whrc_def456');
 * console.log(`Receipt ${receipt.id}: received at ${receipt.date}`);
 * ```
 */
export declare const WebhookReceiptSchema: z.ZodObject<{
    id: z.ZodString;
    date: z.ZodString;
    webhook_id: z.ZodString;
    headers: z.ZodRecord<z.ZodString, z.ZodUnknown>;
    payload: z.ZodUnknown;
}, z.core.$strip>;
/**
 * Webhook receipt type.
 */
export type WebhookReceipt = z.infer<typeof WebhookReceiptSchema>;
/**
 * Webhook delivery schema representing a delivery attempt to a destination.
 *
 * Deliveries track the status and result of forwarding a received webhook
 * payload to a configured destination.
 *
 * @example
 * ```typescript
 * const deliveries = await listWebhookDeliveries(client, 'wh_abc123');
 * for (const delivery of deliveries) {
 *   console.log(`Delivery ${delivery.id}: ${delivery.status}`);
 * }
 * ```
 */
export declare const WebhookDeliverySchema: z.ZodObject<{
    id: z.ZodString;
    date: z.ZodString;
    webhook_id: z.ZodString;
    webhook_destination_id: z.ZodString;
    webhook_receipt_id: z.ZodString;
    status: z.ZodEnum<{
        success: "success";
        pending: "pending";
        failed: "failed";
    }>;
    retries: z.ZodNumber;
    error: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    response: z.ZodOptional<z.ZodNullable<z.ZodRecord<z.ZodString, z.ZodUnknown>>>;
}, z.core.$strip>;
/**
 * Webhook delivery type.
 */
export type WebhookDelivery = z.infer<typeof WebhookDeliverySchema>;
/**
 * Common options for webhook API calls.
 *
 * Used to pass organization context when calling from CLI or other
 * contexts where the org is not implicit in the authentication token.
 */
export interface WebhookApiOptions {
    /**
     * Organization ID for the request.
     * Required when using user authentication (CLI) instead of SDK key.
     */
    orgId?: string;
}
/**
 * Request schema for creating a new webhook.
 *
 * @example
 * ```typescript
 * const request: CreateWebhookRequest = {
 *   name: 'github-events',
 *   description: 'Receives GitHub webhook events',
 * };
 * ```
 */
export declare const CreateWebhookRequestSchema: z.ZodObject<{
    name: z.ZodString;
    description: z.ZodOptional<z.ZodString>;
}, z.core.$strip>;
/** Request type for creating a webhook. */
export type CreateWebhookRequest = z.infer<typeof CreateWebhookRequestSchema>;
/**
 * Request schema for updating an existing webhook.
 *
 * @example
 * ```typescript
 * const request: UpdateWebhookRequest = {
 *   name: 'github-events-v2',
 *   description: 'Updated description',
 * };
 * ```
 */
export declare const UpdateWebhookRequestSchema: z.ZodObject<{
    name: z.ZodString;
    description: z.ZodOptional<z.ZodString>;
}, z.core.$strip>;
/** Request type for updating a webhook. */
export type UpdateWebhookRequest = z.infer<typeof UpdateWebhookRequestSchema>;
/**
 * Request schema for creating a webhook destination.
 *
 * @example
 * ```typescript
 * const request: CreateWebhookDestinationRequest = {
 *   type: 'url',
 *   config: { url: 'https://api.example.com/webhook' },
 * };
 * ```
 */
export declare const CreateWebhookDestinationRequestSchema: z.ZodObject<{
    type: z.ZodEnum<{
        url: "url";
    }>;
    config: z.ZodRecord<z.ZodString, z.ZodUnknown>;
}, z.core.$strip>;
/** Request type for creating a webhook destination. */
export type CreateWebhookDestinationRequest = z.infer<typeof CreateWebhookDestinationRequestSchema>;
/**
 * Request schema for updating a webhook destination.
 *
 * @example
 * ```typescript
 * const request: UpdateWebhookDestinationRequest = {
 *   config: { url: 'https://api.example.com/webhook/v2' },
 * };
 * ```
 */
export declare const UpdateWebhookDestinationRequestSchema: z.ZodObject<{
    config: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodUnknown>>;
}, z.core.$strip>;
/** Request type for updating a webhook destination. */
export type UpdateWebhookDestinationRequest = z.infer<typeof UpdateWebhookDestinationRequestSchema>;
export declare const PaginationRequestSchema: z.ZodObject<{
    limit: z.ZodOptional<z.ZodNumber>;
    offset: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
export type PaginationRequest = z.infer<typeof PaginationRequestSchema>;
export declare const ListWebhooksRequestSchema: z.ZodObject<{
    limit: z.ZodOptional<z.ZodNumber>;
    offset: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
/** Request type for listing webhooks. */
export type ListWebhooksRequest = z.infer<typeof ListWebhooksRequestSchema>;
export declare const ListWebhookReceiptsRequestSchema: z.ZodObject<{
    limit: z.ZodOptional<z.ZodNumber>;
    offset: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
/** Request type for listing webhook receipts. */
export type ListWebhookReceiptsRequest = z.infer<typeof ListWebhookReceiptsRequestSchema>;
export declare const ListWebhookDeliveriesRequestSchema: z.ZodObject<{
    limit: z.ZodOptional<z.ZodNumber>;
    offset: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
/** Request type for listing webhook deliveries. */
export type ListWebhookDeliveriesRequest = z.infer<typeof ListWebhookDeliveriesRequestSchema>;
//# sourceMappingURL=types.d.ts.map