import { z } from 'zod';
import { type APIClient } from '../api.ts';
import { type AnalyticsOptions, type OrgAnalytics, type QueueAnalytics, type SSEStatsEvent, type StreamAnalyticsOptions, type TimeSeriesData } from './types.ts';
export declare const OrgAnalyticsResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        analytics: z.ZodObject<{
            org_id: z.ZodString;
            period: z.ZodObject<{
                start: z.ZodString;
                end: z.ZodString;
                granularity: z.ZodOptional<z.ZodEnum<{
                    minute: "minute";
                    hour: "hour";
                    day: "day";
                }>>;
            }, z.core.$strip>;
            summary: z.ZodObject<{
                total_queues: z.ZodNumber;
                total_messages_published: z.ZodNumber;
                total_messages_delivered: z.ZodNumber;
                total_messages_acknowledged: z.ZodNumber;
                total_dlq_messages: z.ZodNumber;
                total_bytes_published: z.ZodNumber;
                avg_latency_ms: z.ZodNumber;
                p95_latency_ms: z.ZodNumber;
                error_rate_percent: z.ZodNumber;
            }, z.core.$strip>;
            queues: z.ZodArray<z.ZodObject<{
                id: z.ZodString;
                name: z.ZodString;
                queue_type: z.ZodString;
                messages_published: z.ZodNumber;
                messages_delivered: z.ZodNumber;
                messages_acknowledged: z.ZodNumber;
                backlog: z.ZodNumber;
                dlq_count: z.ZodNumber;
                avg_latency_ms: z.ZodNumber;
                error_rate_percent: z.ZodNumber;
            }, z.core.$strip>>;
        }, z.core.$strip>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const QueueAnalyticsResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        analytics: z.ZodObject<{
            queue_id: z.ZodString;
            queue_name: z.ZodString;
            queue_type: z.ZodString;
            period: z.ZodObject<{
                start: z.ZodString;
                end: z.ZodString;
                granularity: z.ZodOptional<z.ZodEnum<{
                    minute: "minute";
                    hour: "hour";
                    day: "day";
                }>>;
            }, z.core.$strip>;
            current: z.ZodObject<{
                backlog: z.ZodNumber;
                dlq_count: z.ZodNumber;
                messages_in_flight: z.ZodNumber;
                active_consumers: z.ZodNumber;
                oldest_message_age_seconds: z.ZodOptional<z.ZodNullable<z.ZodNumber>>;
            }, z.core.$strip>;
            period_stats: z.ZodObject<{
                messages_published: z.ZodNumber;
                messages_delivered: z.ZodNumber;
                messages_acknowledged: z.ZodNumber;
                messages_failed: z.ZodNumber;
                messages_replayed: z.ZodNumber;
                bytes_published: z.ZodNumber;
                delivery_attempts: z.ZodNumber;
                retry_count: z.ZodNumber;
            }, z.core.$strip>;
            latency: z.ZodObject<{
                avg_ms: z.ZodNumber;
                p50_ms: z.ZodOptional<z.ZodNumber>;
                p95_ms: z.ZodOptional<z.ZodNumber>;
                p99_ms: z.ZodOptional<z.ZodNumber>;
                max_ms: z.ZodOptional<z.ZodNumber>;
            }, z.core.$strip>;
            consumer_latency: z.ZodObject<{
                avg_ms: z.ZodNumber;
                p50_ms: z.ZodOptional<z.ZodNumber>;
                p95_ms: z.ZodOptional<z.ZodNumber>;
                p99_ms: z.ZodOptional<z.ZodNumber>;
                max_ms: z.ZodOptional<z.ZodNumber>;
            }, z.core.$strip>;
            destinations: z.ZodOptional<z.ZodArray<z.ZodObject<{
                id: z.ZodString;
                type: z.ZodString;
                url: z.ZodString;
                success_count: z.ZodNumber;
                failure_count: z.ZodNumber;
                avg_response_time_ms: z.ZodOptional<z.ZodNumber>;
                last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
                last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            }, z.core.$strip>>>;
        }, z.core.$strip>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const TimeSeriesResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        timeseries: z.ZodObject<{
            queue_id: z.ZodString;
            queue_name: z.ZodString;
            period: z.ZodObject<{
                start: z.ZodString;
                end: z.ZodString;
                granularity: z.ZodOptional<z.ZodEnum<{
                    minute: "minute";
                    hour: "hour";
                    day: "day";
                }>>;
            }, z.core.$strip>;
            series: z.ZodArray<z.ZodObject<{
                timestamp: z.ZodString;
                throughput: z.ZodNumber;
                delivery_rate: z.ZodNumber;
                ack_rate: z.ZodNumber;
                error_rate: z.ZodNumber;
                avg_latency_ms: z.ZodNumber;
                p95_latency_ms: z.ZodOptional<z.ZodNumber>;
                backlog: z.ZodOptional<z.ZodNumber>;
                messages_in_flight: z.ZodOptional<z.ZodNumber>;
            }, z.core.$strip>>;
        }, z.core.$strip>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
/**
 * Get org-level analytics for all queues.
 *
 * Returns aggregated statistics across all queues in the organization.
 *
 * @param client - The API client instance
 * @param options - Analytics options (time range, filters)
 * @returns Org-level analytics summary
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const analytics = await getOrgAnalytics(client, {
 *   start: '2026-01-14T00:00:00Z',
 *   end: '2026-01-15T00:00:00Z',
 * });
 * console.log(`Total queues: ${analytics.summary.total_queues}`);
 * console.log(`Messages published: ${analytics.summary.total_messages_published}`);
 * ```
 */
export declare function getOrgAnalytics(client: APIClient, options?: AnalyticsOptions): Promise<OrgAnalytics>;
/**
 * Get detailed analytics for a specific queue.
 *
 * Returns current state, period statistics, latency metrics, and destination stats.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @param options - Analytics options (time range, filters)
 * @returns Queue analytics
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const analytics = await getQueueAnalytics(client, 'order-processing', {
 *   start: '2026-01-14T00:00:00Z',
 * });
 * console.log(`Backlog: ${analytics.current.backlog}`);
 * console.log(`P95 latency: ${analytics.latency.p95_ms}ms`);
 * ```
 */
export declare function getQueueAnalytics(client: APIClient, name: string, options?: AnalyticsOptions): Promise<QueueAnalytics>;
/**
 * Get time series analytics data for a queue.
 *
 * Returns time-bucketed metrics for visualization in charts and graphs.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @param options - Analytics options (time range, granularity)
 * @returns Time series data
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const timeseries = await getQueueTimeSeries(client, 'order-processing', {
 *   granularity: 'hour',
 *   start: '2026-01-14T00:00:00Z',
 * });
 * for (const point of timeseries.series) {
 *   console.log(`${point.timestamp}: ${point.throughput} msg/h`);
 * }
 * ```
 */
export declare function getQueueTimeSeries(client: APIClient, name: string, options?: AnalyticsOptions): Promise<TimeSeriesData>;
/**
 * Stream real-time analytics for all queues via SSE.
 *
 * Returns an async iterator that yields stats events at the specified interval.
 * The connection stays open until the iterator is closed or an error occurs.
 *
 * @param client - The API client instance
 * @param options - Stream options (interval, orgId)
 * @returns Async iterator of SSE stats events
 *
 * @example
 * ```typescript
 * const stream = streamOrgAnalytics(client, { interval: 5 });
 * for await (const event of stream) {
 *   console.log(`Backlog: ${event.backlog}, Throughput: ${event.throughput_1m}/min`);
 * }
 * ```
 */
export declare function streamOrgAnalytics(client: APIClient, options?: StreamAnalyticsOptions): AsyncGenerator<SSEStatsEvent, void, unknown>;
/**
 * Stream real-time analytics for a specific queue via SSE.
 *
 * Returns an async iterator that yields stats events at the specified interval.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @param options - Stream options (interval, orgId)
 * @returns Async iterator of SSE stats events
 *
 * @example
 * ```typescript
 * const stream = streamQueueAnalytics(client, 'order-processing', { interval: 5 });
 * for await (const event of stream) {
 *   console.log(`Backlog: ${event.backlog}, In-flight: ${event.messages_in_flight}`);
 * }
 * ```
 */
export declare function streamQueueAnalytics(client: APIClient, name: string, options?: StreamAnalyticsOptions): AsyncGenerator<SSEStatsEvent, void, unknown>;
//# sourceMappingURL=analytics.d.ts.map