import { success, createParseMethods, SCHEMA_KIND } from "../base.js";
/**
 * Schema for optional values (T | undefined).
 * Accepts undefined or the wrapped schema's type.
 *
 * @template T - The wrapped schema type
 *
 * @example
 * ```typescript
 * const schema = s.optional(s.string());
 * schema.parse('hello'); // 'hello'
 * schema.parse(undefined); // undefined
 * schema.parse(123); // throws ValidationError
 * ```
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any
export class OptionalSchema {
    [SCHEMA_KIND] = 'OptionalSchema';
    schema;
    description;
    '~standard' = {
        version: 1,
        vendor: 'agentuity',
        validate: (value) => {
            if (value === undefined) {
                return success(undefined);
            }
            return this.schema['~standard'].validate(value);
        },
        types: undefined,
    };
    // Type-safe parse methods for this instance
    parseMethods = createParseMethods();
    constructor(schema) {
        this.schema = schema;
    }
    describe(description) {
        this.description = description;
        return this;
    }
    optional() {
        return this; // Already optional
    }
    nullable() {
        // Import here to avoid circular dependency
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const { nullable } = require('./nullable.js');
        return nullable(this);
    }
    parse = this.parseMethods.parse;
    safeParse = this.parseMethods.safeParse;
}
/**
 * Make a schema optional (T | undefined).
 *
 * @param schema - The schema to make optional
 *
 * @example
 * ```typescript
 * const userSchema = s.object({
 *   name: s.string(),
 *   nickname: s.optional(s.string())
 * });
 * ```
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any
export function optional(schema) {
    return new OptionalSchema(schema);
}
//# sourceMappingURL=optional.js.map