import { success, createParseMethods, SCHEMA_KIND } from "../base.js";
import { optional } from "../utils/optional.js";
import { nullable } from "../utils/nullable.js";
const parseMethods = createParseMethods();
/**
 * Schema that accepts any value with type-safe unknown.
 * Returns the value as-is without validation.
 * Use this when you want to accept any value but force type checking at usage site.
 *
 * @example
 * ```typescript
 * const schema = s.unknown();
 * const value = schema.parse(123); // unknown
 * const value2 = schema.parse('hello'); // unknown
 * const value3 = schema.parse(null); // unknown
 *
 * // Forces type narrowing
 * if (typeof value === 'string') {
 *   console.log(value.toUpperCase());
 * }
 * ```
 */
export class UnknownSchema {
    [SCHEMA_KIND] = 'UnknownSchema';
    description;
    '~standard' = {
        version: 1,
        vendor: 'agentuity',
        validate: (value) => success(value),
        types: undefined,
    };
    describe(description) {
        this.description = description;
        return this;
    }
    optional() {
        return optional(this);
    }
    nullable() {
        return nullable(this);
    }
    parse = parseMethods.parse;
    safeParse = parseMethods.safeParse;
}
/**
 * Create an unknown schema that accepts any value.
 *
 * @example
 * ```typescript
 * const schema = s.unknown();
 * const value = schema.parse(anything); // Type is unknown
 * ```
 */
export function unknown() {
    return new UnknownSchema();
}
//# sourceMappingURL=unknown.js.map