import { type WebRTCClientCallbacks, type TrackSource } from '@agentuity/frontend';
import type { WebRTCConnectionState, DataChannelConfig, DataChannelState, ConnectionQualitySummary, RecordingHandle, RecordingOptions } from '@agentuity/core';
export type { WebRTCClientCallbacks, DataChannelConfig, DataChannelState, ConnectionQualitySummary, };
export type { WebRTCConnectionState };
/**
 * Options for useWebRTCCall hook
 */
export interface UseWebRTCCallOptions {
    /** Room ID to join */
    roomId: string;
    /** WebSocket signaling URL (e.g., '/call/signal' or full URL) */
    signalUrl: string;
    /** Whether this peer is "polite" in perfect negotiation */
    polite?: boolean;
    /** ICE servers configuration */
    iceServers?: RTCIceServer[];
    /**
     * Media source configuration.
     * - `false`: Data-only mode (no media)
     * - `MediaStreamConstraints`: Use getUserMedia with these constraints
     * - `TrackSource`: Use a custom track source
     * Default: { video: true, audio: true }
     */
    media?: MediaStreamConstraints | TrackSource | false;
    /**
     * Data channels to create when connection is established.
     * Only the offerer (late joiner) creates channels; the answerer receives them.
     */
    dataChannels?: DataChannelConfig[];
    /**
     * Whether to auto-reconnect on WebSocket/ICE failures (default: true)
     */
    autoReconnect?: boolean;
    /**
     * Maximum reconnection attempts before giving up (default: 5)
     */
    maxReconnectAttempts?: number;
    /**
     * Connection timeout in ms for connecting/negotiating (default: 30000)
     */
    connectionTimeout?: number;
    /**
     * ICE gathering timeout in ms (default: 10000)
     */
    iceGatheringTimeout?: number;
    /** Whether to auto-connect on mount (default: true) */
    autoConnect?: boolean;
    /**
     * Optional callbacks for WebRTC events.
     * These are called in addition to the hook's internal state management.
     */
    callbacks?: Partial<WebRTCClientCallbacks>;
}
/**
 * Return type for useWebRTCCall hook
 */
export interface UseWebRTCCallResult {
    /** Ref to attach to local video element */
    localVideoRef: React.RefObject<HTMLVideoElement | null>;
    /** Current connection state */
    state: WebRTCConnectionState;
    /** Current error if any */
    error: Error | null;
    /** Local peer ID assigned by server */
    peerId: string | null;
    /** Remote peer IDs */
    remotePeerIds: string[];
    /** Remote streams keyed by peer ID */
    remoteStreams: Map<string, MediaStream>;
    /** Whether audio is muted */
    isAudioMuted: boolean;
    /** Whether video is muted */
    isVideoMuted: boolean;
    /** Whether this is a data-only connection (no media) */
    isDataOnly: boolean;
    /** Whether screen sharing is active */
    isScreenSharing: boolean;
    /** Manually start the connection (if autoConnect is false) */
    connect: () => void;
    /** End the call */
    hangup: () => void;
    /** Mute or unmute audio */
    muteAudio: (muted: boolean) => void;
    /** Mute or unmute video */
    muteVideo: (muted: boolean) => void;
    /** Start screen sharing */
    startScreenShare: (options?: DisplayMediaStreamOptions) => Promise<void>;
    /** Stop screen sharing */
    stopScreenShare: () => Promise<void>;
    /** Create a new data channel to all peers */
    createDataChannel: (config: DataChannelConfig) => Map<string, RTCDataChannel>;
    /** Get all open data channel labels */
    getDataChannelLabels: () => string[];
    /** Get the state of a data channel for a specific peer */
    getDataChannelState: (peerId: string, label: string) => DataChannelState | null;
    /** Send a string message to all peers */
    sendString: (label: string, data: string) => boolean;
    /** Send a string message to a specific peer */
    sendStringTo: (peerId: string, label: string, data: string) => boolean;
    /** Send binary data to all peers */
    sendBinary: (label: string, data: ArrayBuffer | Uint8Array) => boolean;
    /** Send binary data to a specific peer */
    sendBinaryTo: (peerId: string, label: string, data: ArrayBuffer | Uint8Array) => boolean;
    /** Send JSON data to all peers */
    sendJSON: (label: string, data: unknown) => boolean;
    /** Send JSON data to a specific peer */
    sendJSONTo: (peerId: string, label: string, data: unknown) => boolean;
    /** Close a specific data channel on all peers */
    closeDataChannel: (label: string) => boolean;
    /** Get quality summary for a peer */
    getQualitySummary: (peerId: string) => Promise<ConnectionQualitySummary | null>;
    /** Get quality summaries for all peers */
    getAllQualitySummaries: () => Promise<Map<string, ConnectionQualitySummary>>;
    /** Start recording a stream */
    startRecording: (streamId: string, options?: RecordingOptions) => RecordingHandle | null;
    /** Check if a stream is being recorded */
    isRecording: (streamId: string) => boolean;
    /** Stop all recordings */
    stopAllRecordings: () => Promise<Map<string, Blob>>;
}
/**
 * React hook for WebRTC peer-to-peer audio/video/data calls.
 *
 * Supports multi-peer mesh networking, screen sharing, recording, and stats.
 *
 * @example
 * ```tsx
 * function VideoCall({ roomId }: { roomId: string }) {
 *   const {
 *     localVideoRef,
 *     state,
 *     remotePeerIds,
 *     remoteStreams,
 *     hangup,
 *     muteAudio,
 *     isAudioMuted,
 *     startScreenShare,
 *   } = useWebRTCCall({
 *     roomId,
 *     signalUrl: '/call/signal',
 *     callbacks: {
 *       onStateChange: (from, to, reason) => console.log(`${from} → ${to}`, reason),
 *       onRemoteStream: (peerId, stream) => console.log(`Got stream from ${peerId}`),
 *     },
 *   });
 *
 *   return (
 *     <div>
 *       <video ref={localVideoRef} autoPlay muted playsInline />
 *       {remotePeerIds.map((peerId) => (
 *         <RemoteVideo key={peerId} stream={remoteStreams.get(peerId)} />
 *       ))}
 *       <p>State: {state}</p>
 *       <button onClick={() => muteAudio(!isAudioMuted)}>
 *         {isAudioMuted ? 'Unmute' : 'Mute'}
 *       </button>
 *       <button onClick={hangup}>Hang Up</button>
 *     </div>
 *   );
 * }
 * ```
 */
export declare function useWebRTCCall(options: UseWebRTCCallOptions): UseWebRTCCallResult;
//# sourceMappingURL=webrtc.d.ts.map