import { jsx as _jsx } from "react/jsx-runtime";
import { useState, useEffect, useLayoutEffect, useMemo, useCallback } from 'react';
import { createContext, useContext } from 'react';
import { defaultBaseUrl } from '@agentuity/frontend';
import { setGlobalBaseUrl, setGlobalAuthHeader } from './client';
export const AgentuityContext = createContext(null);
export const AgentuityProvider = ({ baseUrl, authHeader: authHeaderProp, children, }) => {
    const [authHeader, setAuthHeaderState] = useState(authHeaderProp ?? null);
    const [authLoading, setAuthLoadingState] = useState(false);
    const resolvedBaseUrl = baseUrl || defaultBaseUrl;
    // Memoize setter functions to prevent unnecessary re-renders
    const setAuthHeader = useCallback((token) => {
        setAuthHeaderState(token);
    }, []);
    const setAuthLoading = useCallback((loading) => {
        setAuthLoadingState(loading);
    }, []);
    // Set global baseUrl for RPC clients
    useEffect(() => {
        setGlobalBaseUrl(resolvedBaseUrl);
    }, [resolvedBaseUrl]);
    // Sync authHeader to global state for RPC clients
    useEffect(() => {
        setGlobalAuthHeader(authHeader);
    }, [authHeader]);
    // Sync authHeader prop changes to state synchronously
    // useLayoutEffect ensures the state is updated before child effects run,
    // preventing race conditions where useAPI makes requests before auth is set (issue #732)
    useLayoutEffect(() => {
        if (authHeaderProp !== undefined) {
            setAuthHeaderState(authHeaderProp);
        }
    }, [authHeaderProp]);
    // Memoize context value to prevent unnecessary re-renders
    const contextValue = useMemo(() => ({
        baseUrl: resolvedBaseUrl,
        authHeader,
        setAuthHeader,
        authLoading,
        setAuthLoading,
    }), [resolvedBaseUrl, authHeader, setAuthHeader, authLoading, setAuthLoading]);
    return _jsx(AgentuityContext.Provider, { value: contextValue, children: children });
};
/**
 * Hook to access Agentuity context (non-auth properties only).
 * For authentication state, use useAuth() instead.
 *
 * @throws Error if used outside of AgentuityProvider
 */
export function useAgentuity() {
    const context = useContext(AgentuityContext);
    if (!context || !context.baseUrl) {
        throw new Error('useAgentuity must be used within AgentuityProvider');
    }
    return {
        baseUrl: context.baseUrl,
    };
}
/**
 * Low-level hook for Agentuity's transport auth.
 *
 * This hook exposes the Authorization header and loading state used by
 * Agentuity's API clients (useAPI, useWebsocket, etc.).
 *
 * @example
 * ```tsx
 * import { useAuth } from '@agentuity/react';
 * const { authHeader, isAuthenticated } = useAuth();
 * ```
 *
 * @throws Error if used outside of AgentuityProvider
 */
export function useAuth() {
    const context = useContext(AgentuityContext);
    if (!context || !context.baseUrl) {
        throw new Error('useAuth must be used within AgentuityProvider');
    }
    // Convenience property: authenticated = has token and not loading
    const isAuthenticated = !context.authLoading && !!context.authHeader;
    return {
        authHeader: context.authHeader,
        setAuthHeader: context.setAuthHeader,
        authLoading: context.authLoading,
        setAuthLoading: context.setAuthLoading,
        isAuthenticated,
    };
}
//# sourceMappingURL=context.js.map