/**
 * Global registry for PostgreSQL clients and pools.
 *
 * This module provides a way to track all active PostgreSQL clients and pools
 * so they can be gracefully shut down together (e.g., on process exit).
 *
 * The runtime can use `shutdownAll()` to close all registered clients/pools
 * during graceful shutdown.
 *
 * When @agentuity/runtime is available, this module automatically registers
 * a shutdown hook so all postgres clients/pools are closed during graceful shutdown.
 */
/**
 * Common interface for registrable PostgreSQL connections.
 * Both PostgresClient and PostgresPool implement this interface.
 */
export interface Registrable {
    /**
     * Whether the connection is shutting down.
     */
    readonly shuttingDown: boolean;
    /**
     * Signal that the connection is shutting down.
     */
    shutdown(): void;
    /**
     * Close the connection.
     */
    close(): Promise<void>;
}
/**
 * Registers a client or pool in the global registry.
 * Called automatically when a client or pool is created.
 *
 * @param connection - The client or pool to register
 * @internal
 */
export declare function registerClient(connection: Registrable): void;
/**
 * Unregisters a client or pool from the global registry.
 * Called automatically when a client or pool is closed.
 *
 * @param connection - The client or pool to unregister
 * @internal
 */
export declare function unregisterClient(connection: Registrable): void;
/**
 * Returns the number of registered clients and pools.
 * Useful for debugging and testing.
 */
export declare function getClientCount(): number;
/**
 * Returns all registered clients and pools.
 * Useful for debugging and monitoring.
 */
export declare function getClients(): ReadonlySet<Registrable>;
/**
 * Shuts down all registered PostgreSQL clients and pools gracefully.
 *
 * This function:
 * 1. Signals shutdown to all clients/pools (prevents reconnection)
 * 2. Closes all clients/pools in parallel
 * 3. Clears the registry
 *
 * This is intended to be called by the runtime during graceful shutdown.
 *
 * @param timeoutMs - Optional timeout in milliseconds. If provided, the function
 *                    will resolve after the timeout even if some connections haven't
 *                    finished closing. Default: no timeout.
 * @returns A promise that resolves when all connections are closed (or timeout)
 *
 * @example
 * ```typescript
 * import { shutdownAll } from '@agentuity/postgres';
 *
 * process.on('SIGTERM', async () => {
 *   await shutdownAll(5000); // 5 second timeout
 *   process.exit(0);
 * });
 * ```
 */
export declare function shutdownAll(timeoutMs?: number): Promise<void>;
/**
 * Checks if there are any active (non-shutdown) clients or pools.
 * Useful for health checks.
 */
export declare function hasActiveClients(): boolean;
//# sourceMappingURL=registry.d.ts.map