/**
 * Creates hooks for logging agent completion metrics.
 *
 * Tracks the start of each LLM call (via chat.params) and logs
 * agent name, model, and duration when the response arrives (via chat.message).
 */
export function createCompletionHooks(ctx, _config) {
    const startTimes = new Map();
    return {
        onParams(input) {
            // OpenCode passes agent and model as structured objects (not plain strings)
            // in the chat.params hook. Normalize to string here so template literals
            // don't produce '[object Object]' in the completion log.
            const inp = input;
            if (!inp.sessionID)
                return;
            const agentStr = typeof inp.agent === 'string'
                ? inp.agent
                : (inp.agent
                    ?.displayName ??
                    inp.agent?.name ??
                    inp.agent?.id ??
                    String(inp.agent ?? 'unknown'));
            const modelStr = typeof inp.model === 'string'
                ? inp.model
                : (inp.model?.id ??
                    inp.model?.name ??
                    String(inp.model ?? 'unknown'));
            startTimes.set(inp.sessionID, {
                startedAt: Date.now(),
                agent: agentStr,
                model: modelStr,
            });
        },
        onMessage(input) {
            const inp = input;
            if (!inp.sessionID)
                return;
            const start = startTimes.get(inp.sessionID);
            if (!start)
                return;
            const durationMs = Date.now() - start.startedAt;
            const durationSec = (durationMs / 1000).toFixed(1);
            const logLine = `Completion: agent=${start.agent} model=${start.model} duration=${durationSec}s`;
            // Send to the OpenCode log service (never use console.* in a TUI context —
            // raw stdout writes bypass the terminal renderer and corrupt the display)
            ctx.client.app.log({
                body: {
                    service: 'agentuity-coder',
                    level: 'debug',
                    message: logLine,
                },
            });
            // Clean up after logging
            startTimes.delete(inp.sessionID);
        },
    };
}
//# sourceMappingURL=completion.js.map