import { SQL as BunSQL, type SQL as BunSQLClient, type SQLOptions } from 'bun';
import { type BunSQLDatabase } from 'drizzle-orm/bun-sql';
import type { DrizzleConfig } from 'drizzle-orm';
import { type CallablePostgresClient, type PostgresConfig } from '@agentuity/postgres';
import type { PostgresDrizzleConfig, PostgresDrizzle, PostgresDrizzlePg } from './types.ts';
/**
 * Resolves the PostgreSQL client configuration from Drizzle config options.
 *
 * URL priority chain: `connection.url` > `url` > `connectionString` > `process.env.DATABASE_URL`
 *
 * @internal Exported for testing — not part of the public package API.
 */
export declare function resolvePostgresClientConfig<TSchema extends Record<string, unknown> = Record<string, never>>(config?: PostgresDrizzleConfig<TSchema>): PostgresConfig;
/**
 * Creates a dynamic SQL proxy that always delegates to the PostgresClient's
 * current raw connection. This ensures that after automatic reconnection,
 * Drizzle ORM uses the fresh connection instead of a stale reference.
 *
 * The proxy also wraps `unsafe()` calls with the client's retry logic,
 * providing automatic retry on transient connection errors.
 *
 * @internal Exported for testing — not part of the public package API.
 */
export declare function createResilientSQLProxy(client: CallablePostgresClient): InstanceType<typeof BunSQL>;
type DrizzleConnectionConfig = string | ({
    url?: string;
} & SQLOptions);
declare function _drizzle<TSchema extends Record<string, unknown> = Record<string, never>, TClient extends BunSQLClient = BunSQLClient>(...params: [TClient | string] | [TClient | string, DrizzleConfig<TSchema>] | [DrizzleConfig<TSchema> & ({
    connection: DrizzleConnectionConfig;
} | {
    client: TClient;
})]): BunSQLDatabase<TSchema> & {
    $client: TClient;
};
declare namespace _drizzle {
    var mock: <TSchema extends Record<string, unknown> = Record<string, never>>(config?: DrizzleConfig<TSchema>) => BunSQLDatabase<TSchema> & {
        $client: "$client is not available on drizzle.mock()";
    };
}
export declare const drizzle: typeof _drizzle & {
    mock: typeof _drizzle.mock;
};
/**
 * Creates a Drizzle ORM instance with a resilient PostgreSQL connection.
 *
 * This function combines the power of Drizzle ORM with @agentuity/postgres's
 * automatic reconnection capabilities. The underlying connection will
 * automatically reconnect with exponential backoff if the connection is lost.
 *
 * @template TSchema - The Drizzle schema type for type-safe queries
 * @param config - Configuration options for the database connection
 * @returns An object containing the Drizzle instance, underlying client, and close function
 *
 * @example
 * ```typescript
 * import { createPostgresDrizzle } from '@agentuity/drizzle';
 * import * as schema from './schema';
 *
 * // Basic usage with DATABASE_URL (defaults to pg driver with resilient pool)
 * const { db, client, close } = createPostgresDrizzle({ schema });
 *
 * // Query with type safety
 * const users = await db.select().from(schema.users);
 *
 * // Access connection stats from the resilient pool
 * console.log(client.stats);
 *
 * // Clean up when done
 * await close();
 * ```
 *
 * @example
 * ```typescript
 * // With custom connection and reconnection configuration
 * const { db, client, close } = createPostgresDrizzle({
 *   connectionString: 'postgres://user:pass@localhost:5432/mydb',
 *   schema,
 *   logger: true,
 *   reconnect: {
 *     maxAttempts: 5,
 *     initialDelayMs: 100,
 *   },
 *   onReconnected: () => console.log('Database reconnected'),
 * });
 * ```
 *
 * @example
 * ```typescript
 * // Opt-in to Bun's native SQL driver for maximum performance
 * const { db, client, close } = createPostgresDrizzle({
 *   url: process.env.DATABASE_URL,
 *   schema,
 *   driver: 'bun-sql',
 * });
 * ```
 */
/**
 * Creates a Drizzle ORM instance using Bun's native SQL driver.
 */
export declare function createPostgresDrizzle<TSchema extends Record<string, unknown> = Record<string, never>>(config: PostgresDrizzleConfig<TSchema> & {
    driver: 'bun-sql';
}): PostgresDrizzle<TSchema>;
/**
 * Creates a Drizzle ORM instance using the pg (node-postgres) driver (default).
 */
export declare function createPostgresDrizzle<TSchema extends Record<string, unknown> = Record<string, never>>(config?: PostgresDrizzleConfig<TSchema>): PostgresDrizzlePg<TSchema>;
export {};
//# sourceMappingURL=postgres.d.ts.map