import { buildUrl, toServiceException } from "./_util.js";
import { StructuredError } from "../error.js";
import { safeStringify } from "../json.js";
const TASK_API_VERSION = '2026-02-24';
const TaskIdRequiredError = StructuredError('TaskIdRequiredError', 'Task ID is required and must be a non-empty string');
const TaskTitleRequiredError = StructuredError('TaskTitleRequiredError', 'Task title is required and must be a non-empty string');
const TaskStorageResponseError = StructuredError('TaskStorageResponseError')();
export class TaskStorageService {
    #adapter;
    #baseUrl;
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    async create(params) {
        if (!params?.title || typeof params.title !== 'string' || params.title.trim().length === 0) {
            throw new TaskTitleRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(params),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.create',
                attributes: {
                    type: params.type,
                    priority: params.priority ?? 'none',
                    status: params.status ?? 'open',
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async get(id) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.get',
                attributes: { id },
            },
        });
        if (res.response.status === 404) {
            return null;
        }
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async list(params) {
        const queryParams = new URLSearchParams();
        if (params?.status)
            queryParams.set('status', params.status);
        if (params?.type)
            queryParams.set('type', params.type);
        if (params?.priority)
            queryParams.set('priority', params.priority);
        if (params?.assigned_id)
            queryParams.set('assigned_id', params.assigned_id);
        if (params?.parent_id)
            queryParams.set('parent_id', params.parent_id);
        if (params?.sort)
            queryParams.set('sort', params.sort);
        if (params?.order)
            queryParams.set('order', params.order);
        if (params?.limit !== undefined)
            queryParams.set('limit', String(params.limit));
        if (params?.offset !== undefined)
            queryParams.set('offset', String(params.offset));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.list',
                attributes: {
                    ...(params?.status ? { status: params.status } : {}),
                    ...(params?.type ? { type: params.type } : {}),
                    ...(params?.priority ? { priority: params.priority } : {}),
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async update(id, params) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        if (params.title !== undefined &&
            (typeof params.title !== 'string' || params.title.trim().length === 0)) {
            throw new TaskTitleRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'PATCH',
            body: safeStringify(params),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.update',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('PATCH', url, res.response);
    }
    async close(id) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.close',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async changelog(id, params) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const queryParams = new URLSearchParams();
        if (params?.limit !== undefined)
            queryParams.set('limit', String(params.limit));
        if (params?.offset !== undefined)
            queryParams.set('offset', String(params.offset));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task-changelog/${TASK_API_VERSION}/${encodeURIComponent(id)}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.changelog',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
}
//# sourceMappingURL=task.js.map