/**
 * Hub URL resolution for Coder CLI commands.
 *
 * Resolution priority:
 *   1. --hub-url flag (explicit per-command override)
 *   2. AGENTUITY_CODER_HUB_URL env var
 *   3. AGENTUITY_DEVMODE_URL env var (dev tunnel URL)
 */
/**
 * Resolve the Hub HTTP base URL for REST API calls.
 * Converts ws:// URLs to http:// automatically.
 *
 * @param flagUrl  Optional --hub-url flag value
 * @returns HTTP base URL (e.g. "http://localhost:3500") or null if Hub is unreachable
 */
export async function resolveHubUrl(flagUrl) {
    // 1. Explicit flag
    if (flagUrl)
        return normalizeToHttp(flagUrl);
    // 2. Env var (explicit)
    const envUrl = process.env.AGENTUITY_CODER_HUB_URL;
    if (envUrl)
        return normalizeToHttp(envUrl);
    // 3. Dev mode URL (tunnel)
    const devUrl = process.env.AGENTUITY_DEVMODE_URL;
    if (devUrl)
        return normalizeToHttp(devUrl);
    return null;
}
/**
 * Resolve the Hub WebSocket URL for Pi extension connections.
 * Converts http:// URLs to ws:// automatically and ensures /api/ws path.
 *
 * @param flagUrl  Optional --hub-url flag value
 * @returns WebSocket URL (e.g. "ws://127.0.0.1:3500/api/ws") or null
 */
export async function resolveHubWsUrl(flagUrl) {
    const httpUrl = await resolveHubUrl(flagUrl);
    if (!httpUrl)
        return null;
    return normalizeToWs(httpUrl);
}
/**
 * Convert any URL form to an HTTP base URL (strip paths, convert ws->http).
 */
function normalizeToHttp(url) {
    let normalized = url.trim().replace(/\/+$/, '');
    // ws:// -> http://
    if (normalized.startsWith('ws://'))
        normalized = 'http://' + normalized.slice(5);
    else if (normalized.startsWith('wss://'))
        normalized = 'https://' + normalized.slice(6);
    // Strip /api/ws or /api/hub/* paths to get base URL
    normalized = normalized.replace(/\/api\/ws\b.*$/, '');
    normalized = normalized.replace(/\/api\/hub\b.*$/, '');
    return normalized.replace(/\/+$/, '');
}
/**
 * Convert an HTTP base URL to a WebSocket URL with /api/ws path.
 */
function normalizeToWs(httpUrl) {
    let wsUrl = httpUrl;
    if (wsUrl.startsWith('http://'))
        wsUrl = 'ws://' + wsUrl.slice(7);
    else if (wsUrl.startsWith('https://'))
        wsUrl = 'wss://' + wsUrl.slice(8);
    if (!wsUrl.includes('/api/ws')) {
        wsUrl = wsUrl.replace(/\/?$/, '/api/ws');
    }
    return wsUrl;
}
//# sourceMappingURL=hub-url.js.map