import * as acornLoose from 'acorn-loose';
import { dirname, relative, join, basename, resolve } from 'node:path';
import { parse as parseCronExpression } from '@datasert/cronjs-parser';
import { generate } from 'astring';
import { createLogger } from '@agentuity/server';
import * as ts from 'typescript';
import { StructuredError } from '@agentuity/core';
import { existsSync, mkdirSync, statSync } from 'node:fs';
import JSON5 from 'json5';
import { formatSchemaCode } from './format-schema';
import { toForwardSlash } from '../../utils/normalize-path';
import { computeApiMountPath, joinMountAndRoute, extractRelativeApiPath, } from './vite/api-mount-path';
const logger = createLogger((process.env.AGENTUITY_LOG_LEVEL || 'info'));
function parseObjectExpressionToMap(expr) {
    const result = new Map();
    for (const prop of expr.properties) {
        switch (prop.value.type) {
            case 'Literal': {
                const value = prop.value;
                result.set(prop.key.name, String(value.value));
                break;
            }
            default: {
                console.warn('AST value type %s of metadata key: %s not supported', prop.value.type, prop.key.name);
            }
        }
    }
    return result;
}
function createObjectPropertyNode(key, value) {
    return {
        type: 'Property',
        kind: 'init',
        key: {
            type: 'Identifier',
            name: key,
        },
        value: {
            type: 'Literal',
            value,
        },
    };
}
function createNewMetadataNode() {
    return {
        type: 'Property',
        kind: 'init',
        key: {
            type: 'Identifier',
            name: 'metadata',
        },
        value: {
            type: 'ObjectExpression',
            properties: [],
        },
    };
}
function hash(...val) {
    const hasher = new Bun.CryptoHasher('sha256');
    val.map((val) => hasher.update(val));
    return hasher.digest().toHex();
}
function hashSHA1(...val) {
    const hasher = new Bun.CryptoHasher('sha1');
    val.map((val) => hasher.update(val));
    return hasher.digest().toHex();
}
export function getDevmodeDeploymentId(projectId, endpointId) {
    return `devmode_${hashSHA1(projectId, endpointId)}`;
}
// getAgentId generates the deployment-specific agent ID (becomes database PK agent.id)
// This ID changes with each deployment and uses the agentid_ prefix
// Hash includes deploymentId so it's unique per deployment
function getAgentId(projectId, deploymentId, filename, version) {
    return `agentid_${hashSHA1(projectId, deploymentId, filename, version)}`;
}
function getEvalId(projectId, deploymentId, filename, name, version) {
    return `evalid_${hashSHA1(projectId, deploymentId, filename, name, version)}`;
}
function generateRouteId(projectId, deploymentId, type, method, filename, path, version) {
    return `route_${hashSHA1(projectId, deploymentId, type, method, filename, path, version)}`;
}
// generateStableAgentId generates the stable identifier (becomes database agent.identifier)
// This uses the agent_ prefix and is the same across all deployments
// Hash only includes projectId + name, no deploymentId
function generateStableAgentId(projectId, name) {
    return `agent_${hashSHA1(projectId, name)}`.substring(0, 64);
}
function generateStableEvalId(projectId, agentId, name) {
    return `eval_${hashSHA1(projectId, agentId, name)}`.substring(0, 64);
}
/**
 * Type guard to check if an AST node is an ObjectExpression
 */
function isObjectExpression(node) {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    return typeof node === 'object' && node !== null && node.type === 'ObjectExpression';
}
/**
 * Extract schema code from createAgent call arguments
 * Returns input and output schema code as strings
 */
function extractSchemaCode(callargexp) {
    let schemaObj;
    // Find the schema property
    for (const prop of callargexp.properties) {
        if (prop.key.type === 'Identifier' && prop.key.name === 'schema') {
            if (prop.value.type === 'ObjectExpression') {
                schemaObj = prop.value;
                break;
            }
        }
    }
    if (!schemaObj) {
        return {};
    }
    let inputSchemaCode;
    let outputSchemaCode;
    // Extract input and output schema code
    for (const prop of schemaObj.properties) {
        if (prop.key.type === 'Identifier') {
            if (prop.key.name === 'input' && prop.value) {
                // Generate source code from AST node and format it
                inputSchemaCode = formatSchemaCode(generate(prop.value));
            }
            else if (prop.key.name === 'output' && prop.value) {
                // Generate source code from AST node and format it
                outputSchemaCode = formatSchemaCode(generate(prop.value));
            }
        }
    }
    return { inputSchemaCode, outputSchemaCode };
}
const MetadataError = StructuredError('MetatadataNameMissingError')();
function augmentAgentMetadataNode(projectId, id, rel, version, ast, propvalue, filename, inputSchemaCode, outputSchemaCode) {
    const metadata = parseObjectExpressionToMap(propvalue);
    if (!metadata.has('name')) {
        const location = ast.loc?.start?.line ? ` on line ${ast.loc.start.line}` : '';
        throw new MetadataError({
            filename,
            line: ast.loc?.start?.line,
            message: `missing required metadata.name in ${filename}${location}. This Agent should have a unique and human readable name for this project.`,
        });
    }
    const name = metadata.get('name');
    const descriptionNode = propvalue.properties.find((x) => x.key.name === 'description')?.value;
    const descriptionValue = descriptionNode ? descriptionNode.value : '';
    const description = typeof descriptionValue === 'string' ? descriptionValue : '';
    const agentId = generateStableAgentId(projectId, name);
    metadata.set('version', version);
    metadata.set('filename', rel);
    metadata.set('id', id);
    metadata.set('agentId', agentId);
    metadata.set('description', description);
    if (inputSchemaCode) {
        metadata.set('inputSchemaCode', inputSchemaCode);
    }
    if (outputSchemaCode) {
        metadata.set('outputSchemaCode', outputSchemaCode);
    }
    propvalue.properties.push(createObjectPropertyNode('id', id), createObjectPropertyNode('agentId', agentId), createObjectPropertyNode('version', version), createObjectPropertyNode('filename', rel), createObjectPropertyNode('description', description));
    if (inputSchemaCode) {
        propvalue.properties.push(createObjectPropertyNode('inputSchemaCode', inputSchemaCode));
    }
    if (outputSchemaCode) {
        propvalue.properties.push(createObjectPropertyNode('outputSchemaCode', outputSchemaCode));
    }
    const newsource = generate(ast);
    // Evals imports are now handled in registry.generated.ts
    return [newsource, metadata];
}
function createAgentMetadataNode(id, name, rel, version, ast, callargexp, _filename, projectId, inputSchemaCode, outputSchemaCode) {
    const newmetadata = createNewMetadataNode();
    const agentId = generateStableAgentId(projectId, name);
    const md = new Map();
    md.set('id', id);
    md.set('agentId', agentId);
    md.set('version', version);
    md.set('name', name);
    md.set('filename', rel);
    if (inputSchemaCode) {
        md.set('inputSchemaCode', inputSchemaCode);
    }
    if (outputSchemaCode) {
        md.set('outputSchemaCode', outputSchemaCode);
    }
    for (const [key, value] of md) {
        newmetadata.value.properties.push(createObjectPropertyNode(key, value));
    }
    callargexp.properties.push(newmetadata);
    const newsource = generate(ast);
    // Evals imports are now handled in registry.generated.ts
    return [newsource, md];
}
const DuplicateNameError = StructuredError('DuplicateNameError')();
function injectEvalMetadata(configObj, evalId, stableEvalId, version, filename, agentId) {
    // Create metadata object with eval IDs and version
    const properties = [
        createObjectPropertyNode('id', evalId),
        createObjectPropertyNode('evalId', stableEvalId),
        createObjectPropertyNode('version', version),
        createObjectPropertyNode('filename', filename),
    ];
    // Add agentId if available
    if (agentId) {
        properties.push(createObjectPropertyNode('agentId', agentId));
    }
    const metadataObj = {
        type: 'Property',
        kind: 'init',
        key: {
            type: 'Identifier',
            name: 'metadata',
        },
        value: {
            type: 'ObjectExpression',
            properties,
        },
    };
    // Add metadata to the config object
    configObj.properties.push(metadataObj);
}
function findAgentVariableAndImport(ast) {
    // First, find what variable is being used in agent.createEval() calls
    let agentVarName;
    for (const node of ast.body) {
        if (node.type === 'ExportNamedDeclaration') {
            const exportDecl = node;
            if (exportDecl.declaration?.type === 'VariableDeclaration') {
                const variableDeclaration = exportDecl.declaration;
                for (const vardecl of variableDeclaration.declarations) {
                    if (vardecl.type === 'VariableDeclarator' &&
                        vardecl.init?.type === 'CallExpression') {
                        const call = vardecl.init;
                        if (call.callee.type === 'MemberExpression') {
                            const memberExpr = call.callee;
                            const object = memberExpr.object;
                            const property = memberExpr.property;
                            if (object.type === 'Identifier' &&
                                property.type === 'Identifier' &&
                                property.name === 'createEval') {
                                agentVarName = object.name;
                                break;
                            }
                        }
                    }
                }
                if (agentVarName)
                    break;
            }
        }
    }
    if (!agentVarName)
        return undefined;
    // Now find the import for this variable
    for (const node of ast.body) {
        if (node.type === 'ImportDeclaration') {
            const importDecl = node;
            // Find default import specifier that matches our variable
            for (const spec of importDecl.specifiers) {
                if (spec.type === 'ImportDefaultSpecifier' && spec.local.name === agentVarName) {
                    const importPath = importDecl.source.value;
                    if (typeof importPath === 'string') {
                        return { varName: agentVarName, importPath };
                    }
                }
            }
        }
    }
    return undefined;
}
export async function parseEvalMetadata(rootDir, filename, contents, projectId, deploymentId, agentId, agentMetadata) {
    const logLevel = (process.env.AGENTUITY_LOG_LEVEL || 'info');
    const logger = createLogger(logLevel);
    logger.trace(`Parsing evals from ${filename}`);
    // Quick string search optimization - skip AST parsing if no createEval call
    if (!contents.includes('createEval')) {
        logger.trace(`Skipping ${filename}: no createEval found`);
        return [contents, []];
    }
    const ast = acornLoose.parse(contents, {
        locations: true,
        ecmaVersion: 'latest',
        sourceType: 'module',
    });
    const rel = toForwardSlash(relative(rootDir, filename));
    const version = hash(contents);
    const evals = [];
    // Try to find the corresponding agent to get the agentId
    let resolvedAgentId = agentId;
    if (!resolvedAgentId && agentMetadata) {
        const agentInfo = findAgentVariableAndImport(ast);
        if (agentInfo) {
            logger.trace(`[EVAL METADATA] Found agent variable '${agentInfo.varName}' imported from '${agentInfo.importPath}'`);
            // Resolve the import path to actual file path
            let resolvedPath = agentInfo.importPath;
            if (resolvedPath.startsWith('./') || resolvedPath.startsWith('../')) {
                // Convert relative path to match the format in agentMetadata
                const baseDir = dirname(filename);
                resolvedPath = join(baseDir, resolvedPath);
                // Normalize and ensure .ts extension
                if (!resolvedPath.endsWith('.ts')) {
                    resolvedPath += '.ts';
                }
            }
            // Find the agent metadata from the passed agentMetadata map
            for (const [agentFile, metadata] of agentMetadata) {
                // Check if this agent file matches the resolved import path
                if (agentFile.includes(basename(resolvedPath)) && metadata.has('agentId')) {
                    resolvedAgentId = metadata.get('agentId');
                    logger.trace(`[EVAL METADATA] Resolved agentId from agent metadata: ${resolvedAgentId} (file: ${agentFile})`);
                    break;
                }
            }
            if (!resolvedAgentId) {
                logger.warn(`[EVAL METADATA] Could not find agent metadata for import path: ${resolvedPath}`);
            }
        }
    }
    // Find all exported agent.createEval() calls
    for (const body of ast.body) {
        let variableDeclaration;
        // Only process exported VariableDeclarations
        if (body.type === 'ExportNamedDeclaration') {
            const exportDecl = body;
            if (exportDecl.declaration?.type === 'VariableDeclaration') {
                variableDeclaration = exportDecl.declaration;
            }
        }
        if (variableDeclaration) {
            for (const vardecl of variableDeclaration.declarations) {
                if (vardecl.type === 'VariableDeclarator' && vardecl.init?.type === 'CallExpression') {
                    const call = vardecl.init;
                    if (call.callee.type === 'MemberExpression') {
                        const memberExpr = call.callee;
                        const object = memberExpr.object;
                        const property = memberExpr.property;
                        if (object.type === 'Identifier' &&
                            object.name === 'agent' &&
                            property.type === 'Identifier' &&
                            property.name === 'createEval') {
                            // Found agent.createEval() call
                            // New signature: agent.createEval(name, { description?, handler })
                            if (call.arguments.length >= 2) {
                                const firstArg = call.arguments[0];
                                const secondArg = call.arguments[1];
                                let evalName;
                                let evalDescription;
                                let configObj;
                                // First argument should be a string literal (the name)
                                if (firstArg.type === 'Literal' &&
                                    typeof firstArg.value === 'string') {
                                    evalName = firstArg.value;
                                }
                                else {
                                    throw new MetadataError({
                                        filename,
                                        line: body.loc?.start?.line,
                                        message: 'agent.createEval() first argument must be a string literal name.',
                                    });
                                }
                                // Second argument should be the config object
                                if (secondArg.type === 'ObjectExpression') {
                                    configObj = secondArg;
                                    // Extract description from config object
                                    for (const prop of configObj.properties) {
                                        if (prop.key.type === 'Identifier' &&
                                            prop.key.name === 'description') {
                                            if (prop.value.type === 'Literal') {
                                                const literalValue = prop.value.value;
                                                evalDescription =
                                                    typeof literalValue === 'string' ? literalValue : undefined;
                                            }
                                        }
                                    }
                                }
                                const finalName = evalName;
                                logger.trace(`Found eval: ${finalName}${evalDescription ? ` - ${evalDescription}` : ''}`);
                                const evalId = getEvalId(projectId, deploymentId, rel, finalName, version);
                                // Generate stable evalId using resolved agentId
                                const effectiveAgentId = resolvedAgentId || '';
                                const stableEvalId = generateStableEvalId(projectId, effectiveAgentId, finalName);
                                // Inject eval metadata into the AST (same pattern as agents)
                                if (configObj) {
                                    injectEvalMetadata(configObj, evalId, stableEvalId, version, rel, resolvedAgentId);
                                }
                                evals.push({
                                    filename: rel,
                                    id: evalId,
                                    version,
                                    name: finalName,
                                    evalId: stableEvalId,
                                    description: evalDescription,
                                });
                            }
                        }
                    }
                }
            }
        }
    }
    // Check for duplicate eval names in the same file
    // This prevents hash collisions when projectId/deploymentId are empty
    const seenNames = new Map();
    for (const evalItem of evals) {
        const count = seenNames.get(evalItem.name) || 0;
        seenNames.set(evalItem.name, count + 1);
    }
    const duplicates = [];
    for (const [name, count] of seenNames.entries()) {
        if (count > 1) {
            duplicates.push(name);
        }
    }
    if (duplicates.length > 0) {
        throw new DuplicateNameError({
            filename,
            message: `Duplicate eval names found in ${rel}: ${duplicates.join(', ')}. ` +
                'Eval names must be unique within the same file to prevent ID collisions.',
        });
    }
    const newsource = generate(ast);
    logger.trace(`Parsed ${evals.length} eval(s) from ${filename}`);
    return [newsource, evals];
}
const InvalidExportError = StructuredError('InvalidExportError')();
export async function parseAgentMetadata(rootDir, filename, contents, projectId, deploymentId) {
    // Quick string search optimization - skip AST parsing if no createAgent call
    if (!contents.includes('createAgent')) {
        return undefined;
    }
    const ast = acornLoose.parse(contents, {
        locations: true,
        ecmaVersion: 'latest',
        sourceType: 'module',
    });
    let exportName;
    const rel = toForwardSlash(relative(rootDir, filename));
    let name; // Will be set from createAgent identifier
    const version = hash(contents);
    const id = getAgentId(projectId, deploymentId, rel, version);
    let result;
    let schemaCodeExtracted = false;
    for (const body of ast.body) {
        if (body.type === 'ExportDefaultDeclaration') {
            if (body.declaration?.type === 'CallExpression') {
                const call = body.declaration;
                if (call.callee.name === 'createAgent') {
                    // Enforce new API: createAgent('name', {config})
                    if (call.arguments.length < 2) {
                        throw new Error(`createAgent requires 2 arguments: createAgent('name', config) in ${filename}`);
                    }
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any
                    const nameArg = call.arguments[0];
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any
                    const configArg = call.arguments[1];
                    if (!nameArg || nameArg.type !== 'Literal' || typeof nameArg.value !== 'string') {
                        throw new Error(`createAgent first argument must be a string literal in ${filename}`);
                    }
                    if (!isObjectExpression(configArg)) {
                        throw new Error(`createAgent second argument must be a config object in ${filename}`);
                    }
                    // Extract agent identifier from createAgent first argument
                    name = nameArg.value;
                    const callargexp = configArg;
                    // Extract schema code before processing metadata
                    let inputSchemaCode;
                    let outputSchemaCode;
                    if (!schemaCodeExtracted) {
                        const schemaCode = extractSchemaCode(callargexp);
                        inputSchemaCode = schemaCode.inputSchemaCode;
                        outputSchemaCode = schemaCode.outputSchemaCode;
                        schemaCodeExtracted = true;
                    }
                    for (const prop of callargexp.properties) {
                        if (prop.key.type === 'Identifier' && prop.key.name === 'metadata') {
                            result = augmentAgentMetadataNode(projectId, id, rel, version, ast, prop.value, filename, inputSchemaCode, outputSchemaCode);
                            break;
                        }
                    }
                    if (!result && name) {
                        result = createAgentMetadataNode(id, name, rel, version, ast, callargexp, filename, projectId, inputSchemaCode, outputSchemaCode);
                    }
                    break;
                }
            }
        }
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        if (!result && body.declaration?.type === 'Identifier') {
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            const identifier = body.declaration;
            exportName = identifier.name;
            break;
        }
    }
    // If no default export or createAgent found, skip this file (it's not an agent)
    if (!result && !exportName) {
        return undefined;
    }
    if (!result) {
        for (const body of ast.body) {
            if (body.type === 'VariableDeclaration') {
                for (const vardecl of body.declarations) {
                    if (vardecl.type === 'VariableDeclarator' && vardecl.id.type === 'Identifier') {
                        const identifier = vardecl.id;
                        if (identifier.name === exportName) {
                            if (vardecl.init?.type === 'CallExpression') {
                                const call = vardecl.init;
                                if (call.callee.name === 'createAgent') {
                                    // Enforce new API: createAgent('name', {config})
                                    if (call.arguments.length < 2) {
                                        throw new Error(`createAgent requires 2 arguments: createAgent('name', config) in ${filename}`);
                                    }
                                    // eslint-disable-next-line @typescript-eslint/no-explicit-any
                                    const nameArg = call.arguments[0];
                                    // eslint-disable-next-line @typescript-eslint/no-explicit-any
                                    const configArg = call.arguments[1];
                                    if (!nameArg ||
                                        nameArg.type !== 'Literal' ||
                                        typeof nameArg.value !== 'string') {
                                        throw new Error(`createAgent first argument must be a string literal in ${filename}`);
                                    }
                                    if (!isObjectExpression(configArg)) {
                                        throw new Error(`createAgent second argument must be a config object in ${filename}`);
                                    }
                                    // Extract agent identifier from createAgent first argument
                                    name = nameArg.value;
                                    const callargexp = configArg;
                                    // Extract schema code before processing metadata
                                    let inputSchemaCode;
                                    let outputSchemaCode;
                                    if (!schemaCodeExtracted) {
                                        const schemaCode = extractSchemaCode(callargexp);
                                        inputSchemaCode = schemaCode.inputSchemaCode;
                                        outputSchemaCode = schemaCode.outputSchemaCode;
                                        schemaCodeExtracted = true;
                                    }
                                    for (const prop of callargexp.properties) {
                                        if (prop.key.type === 'Identifier' && prop.key.name === 'metadata') {
                                            result = augmentAgentMetadataNode(projectId, id, rel, version, ast, prop.value, filename, inputSchemaCode, outputSchemaCode);
                                            break;
                                        }
                                    }
                                    if (!result && name) {
                                        result = createAgentMetadataNode(id, name, rel, version, ast, callargexp, filename, projectId, inputSchemaCode, outputSchemaCode);
                                    }
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    // If no createAgent found after checking all declarations, skip this file
    if (!result) {
        return undefined;
    }
    // Parse evals from eval.ts file in the same directory
    const logLevel = (process.env.AGENTUITY_LOG_LEVEL || 'info');
    const logger = createLogger(logLevel);
    const agentDir = dirname(filename);
    const evalsPath = join(agentDir, 'eval.ts');
    logger.trace(`Checking for evals file at ${evalsPath}`);
    const evalsFile = Bun.file(evalsPath);
    if (await evalsFile.exists()) {
        logger.trace(`Found evals file at ${evalsPath}, parsing...`);
        const evalsSource = await evalsFile.text();
        const transpiler = new Bun.Transpiler({ loader: 'ts', target: 'bun' });
        const evalsContents = transpiler.transformSync(evalsSource);
        const agentId = result[1].get('agentId') || '';
        const [, evals] = await parseEvalMetadata(rootDir, evalsPath, evalsContents, projectId, deploymentId, agentId, new Map() // Empty map since we already have agentId
        );
        if (evals.length > 0) {
            logger.trace(`Adding ${evals.length} eval(s) to agent metadata for ${name}`);
            result[1].set('evals', JSON.stringify(evals));
        }
        else {
            logger.trace(`No evals found in ${evalsPath}`);
        }
    }
    else {
        logger.trace(`No evals file found at ${evalsPath}`);
    }
    return result;
}
const InvalidCreateRouterError = StructuredError('InvalidCreateRouterError')();
const InvalidRouterConfigError = StructuredError('InvalidRouterConfigError')();
const SchemaNotExportedError = StructuredError('SchemaNotExportedError')();
/**
 * Build a set of exported identifiers from the top-level program.
 * Handles both `export const X = ...` and `export { X }` patterns.
 */
function buildExportedIdentifierSet(program) {
    const exported = new Set();
    for (const node of program.body) {
        if (node.type === 'ExportNamedDeclaration') {
            const exp = node;
            // Handle `export const X = ...` or `export function X() { ... }` or `export class X { ... }`
            if (exp.declaration) {
                if (exp.declaration.type === 'VariableDeclaration') {
                    const decl = exp.declaration;
                    for (const d of decl.declarations) {
                        if (d.id.type === 'Identifier') {
                            const id = d.id;
                            exported.add(id.name);
                        }
                    }
                }
                else if (exp.declaration.type === 'FunctionDeclaration') {
                    const funcDecl = exp.declaration;
                    if (funcDecl.id?.name) {
                        exported.add(funcDecl.id.name);
                    }
                }
                else if (exp.declaration.type === 'ClassDeclaration') {
                    const classDecl = exp.declaration;
                    if (classDecl.id?.name) {
                        exported.add(classDecl.id.name);
                    }
                }
            }
            // Handle `export { X }` or `export { X as Y }`
            if (exp.specifiers && Array.isArray(exp.specifiers)) {
                for (const spec of exp.specifiers) {
                    // For `export { X }`, local.name is the variable name in this file
                    if (spec.local?.name) {
                        exported.add(spec.local.name);
                    }
                }
            }
        }
    }
    return exported;
}
/**
 * Validate that schema variables used in validators are either imported or exported.
 * Throws SchemaNotExportedError if a locally-defined schema is not exported.
 */
function validateSchemaExports(schemaVariable, kind, importedNames, exportedNames, filename, method, path) {
    if (!schemaVariable)
        return;
    // If the schema is imported from another file, it's already exported from its source
    if (importedNames.has(schemaVariable))
        return;
    // If the schema is defined locally, it must be exported
    if (!exportedNames.has(schemaVariable)) {
        const routeDesc = method && path ? ` for route "${method.toUpperCase()} ${path}"` : '';
        throw new SchemaNotExportedError({
            filename,
            schemaName: schemaVariable,
            kind,
            method,
            path,
            message: `Schema "${schemaVariable}" used as the ${kind} validator${routeDesc} in ${filename} is not exported.\n\n` +
                `Agentuity generates a route registry that imports schema types by name, so the schema must be exported.\n\n` +
                `To fix this, add "export" to the schema declaration:\n\n` +
                `  export const ${schemaVariable} = s.object({ ... });\n`,
        });
    }
}
/**
 * Scan route handler arguments for validator middleware and extract schema information.
 *
 * Accumulates schema info across ALL validator arguments in the route handler,
 * supporting common patterns like combining param + JSON body validation:
 *
 * ```ts
 * router.patch('/:id',
 *   zValidator('param', paramSchema),  // detected, no schema extracted (non-json)
 *   zValidator('json', bodySchema),    // detected, inputSchemaVariable = 'bodySchema'
 *   async (c) => { ... }
 * );
 * ```
 *
 * **Schema merge strategy — first match wins:**
 * When multiple validators provide the same schema field (e.g., two `inputSchemaVariable`
 * providers), the first one encountered is kept. This is intentional because:
 *
 * 1. Primary validators (e.g., `validator({ input, output })`, `agent.validator()`) are
 *    conventionally listed before supplementary validators (param, query, header, cookie).
 * 2. For `zValidator`, only `'json'` targets extract schemas — other targets (param, query,
 *    header, cookie) return no schema variables, so ordering rarely matters in practice.
 * 3. Duplicate json validators on the same route is uncommon; when it occurs, a warning
 *    is logged to help developers catch unintentional conflicts.
 *
 * Supported validator patterns:
 * - `validator({ input, output, stream })` — Agentuity object-style
 * - `validator('json', callback)` — Hono callback-style
 * - `zValidator('json', schema)` — Zod validator (only 'json' target extracts schemas)
 * - `agent.validator()` / `agent.validator({ input, output })` — Agent validators
 *
 * @param args - The arguments array from a route handler call expression (e.g., `router.post(path, ...args)`)
 * @returns Accumulated validator info with merged schemas from all validators found
 */
function hasValidatorCall(args) {
    if (!args || args.length === 0)
        return { hasValidator: false };
    const result = { hasValidator: false };
    // Helper: merge a schema field using first-match-wins strategy, warn on conflict.
    // When a field is already set and a different value is encountered, the first value
    // is kept and a warning is emitted to help developers catch unintentional duplicates.
    const mergeField = (field, value) => {
        if (!value)
            return;
        if (result[field] && result[field] !== value) {
            const label = field === 'inputSchemaVariable' ? 'inputSchema' : 'outputSchema';
            logger.warn('Multiple validators provide %s: using "%s", ignoring "%s"', label, result[field], value);
        }
        else if (!result[field]) {
            result[field] = value;
        }
    };
    for (const arg of args) {
        if (!arg || typeof arg !== 'object')
            continue;
        const node = arg;
        // Check if this is a CallExpression with callee named 'validator'
        if (node.type === 'CallExpression') {
            const callExpr = node;
            // Check for standalone validator({ input, output }) or Hono validator('json', callback)
            if (callExpr.callee.type === 'Identifier') {
                const identifier = callExpr.callee;
                if (identifier.name === 'validator') {
                    // Try to extract schema variables from validator({ input, output, stream })
                    const schemas = extractValidatorSchemas(callExpr);
                    // If we found schemas from object-style validator, merge them
                    if (schemas.inputSchemaVariable ||
                        schemas.outputSchemaVariable ||
                        schemas.stream !== undefined) {
                        result.hasValidator = true;
                        mergeField('inputSchemaVariable', schemas.inputSchemaVariable);
                        mergeField('outputSchemaVariable', schemas.outputSchemaVariable);
                        if (schemas.stream !== undefined && result.stream === undefined) {
                            result.stream = schemas.stream;
                        }
                        continue;
                    }
                    // Try Hono validator('json', callback) pattern
                    const honoSchemas = extractHonoValidatorSchema(callExpr);
                    result.hasValidator = true;
                    mergeField('inputSchemaVariable', honoSchemas.inputSchemaVariable);
                    continue;
                }
                // Check for zValidator('json', schema)
                if (identifier.name === 'zValidator') {
                    const schemas = extractZValidatorSchema(callExpr);
                    result.hasValidator = true;
                    mergeField('inputSchemaVariable', schemas.inputSchemaVariable);
                    continue;
                }
            }
            // Check for agent.validator()
            if (callExpr.callee.type === 'MemberExpression') {
                const member = callExpr.callee;
                if (member.property && member.property.name === 'validator') {
                    // Extract agent variable name (the object before .validator())
                    const agentVariable = member.object.type === 'Identifier'
                        ? member.object.name
                        : undefined;
                    // Also check for schema overrides: agent.validator({ input, output })
                    const schemas = extractValidatorSchemas(callExpr);
                    result.hasValidator = true;
                    if (agentVariable && !result.agentVariable) {
                        result.agentVariable = agentVariable;
                    }
                    mergeField('inputSchemaVariable', schemas.inputSchemaVariable);
                    mergeField('outputSchemaVariable', schemas.outputSchemaVariable);
                    if (schemas.stream !== undefined && result.stream === undefined) {
                        result.stream = schemas.stream;
                    }
                    continue;
                }
            }
        }
    }
    return result;
}
/**
 * Extract schema variable names and stream flag from validator() call arguments
 * Example: validator({ input: myInputSchema, output: myOutputSchema, stream: true })
 */
function extractValidatorSchemas(callExpr) {
    const result = {};
    // Check if validator has arguments
    if (!callExpr.arguments || callExpr.arguments.length === 0) {
        return result;
    }
    // First argument should be an object expression
    const firstArg = callExpr.arguments[0];
    if (!firstArg || firstArg.type !== 'ObjectExpression') {
        return result;
    }
    const objExpr = firstArg;
    for (const prop of objExpr.properties) {
        // Extract key name defensively - could be Identifier or Literal
        let keyName;
        const propKey = prop.key;
        if (propKey.type === 'Identifier') {
            keyName = propKey.name;
        }
        else if (propKey.type === 'Literal') {
            keyName = String(propKey.value);
        }
        if (!keyName)
            continue;
        if ((keyName === 'input' || keyName === 'output') && prop.value.type === 'Identifier') {
            const valueName = prop.value.name;
            if (keyName === 'input') {
                result.inputSchemaVariable = valueName;
            }
            else {
                result.outputSchemaVariable = valueName;
            }
        }
        // Extract stream flag - can be Literal, Identifier, or UnaryExpression (!0 or !1)
        if (keyName === 'stream') {
            if (prop.value.type === 'Literal') {
                const literal = prop.value;
                if (typeof literal.value === 'boolean') {
                    result.stream = literal.value;
                }
            }
            else if (prop.value.type === 'Identifier') {
                const identifier = prop.value;
                // Handle stream: true or stream: false as identifiers
                if (identifier.name === 'true') {
                    result.stream = true;
                }
                else if (identifier.name === 'false') {
                    result.stream = false;
                }
            }
            else if (prop.value.type === 'UnaryExpression') {
                // Handle !0 (true) or !1 (false) - acorn-loose transpiles booleans this way
                const unary = prop.value;
                if (unary.argument?.type === 'Literal') {
                    const literal = unary.argument;
                    // Numeric literal: !0 = true, !1 = false
                    if (typeof literal.value === 'number') {
                        if (unary.operator === '!') {
                            result.stream = literal.value === 0;
                        }
                    }
                    else if (typeof literal.value === 'boolean') {
                        result.stream = unary.operator === '!' ? !literal.value : literal.value;
                    }
                }
                // Handle true/false as identifiers
                if (unary.argument?.type === 'Identifier') {
                    const identifier = unary.argument;
                    if (identifier.name === 'true') {
                        result.stream = unary.operator === '!' ? false : true;
                    }
                    else if (identifier.name === 'false') {
                        result.stream = unary.operator === '!' ? true : false;
                    }
                }
            }
        }
    }
    return result;
}
/**
 * Extract schema from zValidator() call arguments
 * Example: zValidator('json', mySchema) or zValidator('json', z.object({...}))
 * Returns the schema as inputSchemaVariable since zValidator is for request body validation
 * Only extracts schemas for 'json' target, not 'query', 'param', 'header', or 'cookie'
 */
function extractZValidatorSchema(callExpr) {
    const result = {};
    // zValidator requires at least 2 arguments: zValidator(target, schema)
    if (!callExpr.arguments || callExpr.arguments.length < 2) {
        return result;
    }
    // First argument should be 'json' literal
    const targetArg = callExpr.arguments[0];
    if (targetArg.type === 'Literal') {
        const targetValue = targetArg.value;
        // Only extract schemas for JSON body validation
        if (typeof targetValue === 'string' && targetValue !== 'json') {
            return result;
        }
    }
    else {
        // If first arg is not a literal, we can't determine the target, skip
        return result;
    }
    // Second argument is the schema
    const schemaArg = callExpr.arguments[1];
    // If it's an identifier (variable reference), extract the name
    if (schemaArg.type === 'Identifier') {
        result.inputSchemaVariable = schemaArg.name;
    }
    // If it's inline schema (CallExpression like z.object({...})), we detect but don't extract yet
    // TODO: Extract inline schema code
    return result;
}
/**
 * Extract output schema from SSE options object.
 * Example: sse({ output: MySchema }, handler)
 *
 * @param callExpr - The SSE CallExpression AST node
 * @returns Object with outputSchemaVariable if found
 */
function extractSSEOutputSchema(callExpr) {
    const result = {};
    // sse() can be called as:
    // 1. sse(handler) - no schema
    // 2. sse({ output: schema }, handler) - with schema
    if (!callExpr.arguments || callExpr.arguments.length === 0) {
        return result;
    }
    // Check if first argument is an options object with 'output' property
    const firstArg = callExpr.arguments[0];
    if (firstArg.type !== 'ObjectExpression') {
        // First argument is handler function, no options
        return result;
    }
    const objExpr = firstArg;
    for (const prop of objExpr.properties) {
        // Skip SpreadElement entries (e.g., { ...obj }) which don't have key/value
        if (prop.type !== 'Property') {
            continue;
        }
        // Extract key name - could be Identifier or Literal
        let keyName;
        const propKey = prop.key;
        if (propKey.type === 'Identifier') {
            keyName = propKey.name;
        }
        else if (propKey.type === 'Literal') {
            keyName = String(propKey.value);
        }
        if (!keyName)
            continue;
        // Look for the 'output' property
        if (keyName === 'output' && prop.value.type === 'Identifier') {
            result.outputSchemaVariable = prop.value.name;
            break;
        }
    }
    return result;
}
/**
 * Extract schema from Hono validator('json', callback) pattern
 * Example: validator('json', (value, c) => { const result = mySchema['~standard'].validate(value); ... })
 * Searches the callback function body for schema.validate() or schema['~standard'].validate() calls
 */
function extractHonoValidatorSchema(callExpr) {
    const result = {};
    // Hono validator requires at least 2 arguments: validator(target, callback)
    if (!callExpr.arguments || callExpr.arguments.length < 2) {
        return result;
    }
    // First argument should be 'json' literal (only extract for JSON validation)
    const targetArg = callExpr.arguments[0];
    if (targetArg.type === 'Literal') {
        const targetValue = targetArg.value;
        if (typeof targetValue === 'string' && targetValue !== 'json') {
            return result;
        }
    }
    else {
        return result;
    }
    // Second argument should be a function (arrow or regular)
    const callbackArg = callExpr.arguments[1];
    if (callbackArg.type !== 'ArrowFunctionExpression' &&
        callbackArg.type !== 'FunctionExpression') {
        return result;
    }
    // Get the function body
    const funcExpr = callbackArg;
    if (!funcExpr.body) {
        return result;
    }
    // Search the function body for schema.validate() or schema['~standard'].validate() calls
    const schemaVar = findSchemaValidateCall(funcExpr.body);
    if (schemaVar) {
        result.inputSchemaVariable = schemaVar;
    }
    return result;
}
/**
 * Recursively search AST for schema.validate() or schema['~standard'].validate() calls
 * Returns the schema variable name if found
 */
function findSchemaValidateCall(node) {
    if (!node || typeof node !== 'object')
        return undefined;
    // Check if this is a CallExpression with .validate()
    if (node.type === 'CallExpression') {
        const callExpr = node;
        // Check for schema['~standard'].validate(value) pattern
        // AST: CallExpression -> MemberExpression(validate) -> MemberExpression(['~standard']) -> Identifier(schema)
        if (callExpr.callee.type === 'MemberExpression') {
            const member = callExpr.callee;
            const propName = member.property.type === 'Identifier'
                ? member.property.name
                : undefined;
            if (propName === 'validate') {
                // Check if the object is schema['~standard'] or just schema
                if (member.object.type === 'MemberExpression') {
                    // schema['~standard'].validate() pattern
                    const innerMember = member.object;
                    if (innerMember.object.type === 'Identifier') {
                        return innerMember.object.name;
                    }
                }
                else if (member.object.type === 'Identifier') {
                    // schema.validate() pattern
                    return member.object.name;
                }
            }
        }
    }
    // Recursively search child nodes
    for (const key of Object.keys(node)) {
        const value = node[key];
        if (Array.isArray(value)) {
            for (const item of value) {
                if (item && typeof item === 'object') {
                    const found = findSchemaValidateCall(item);
                    if (found)
                        return found;
                }
            }
        }
        else if (value && typeof value === 'object') {
            const found = findSchemaValidateCall(value);
            if (found)
                return found;
        }
    }
    return undefined;
}
/**
 * Resolve an import path to an actual file on disk.
 * Tries the path as-is, then with common extensions.
 * Returns null for non-relative (package) imports or if no file is found.
 */
function resolveImportPath(fromDir, importPath) {
    // If it's a package import (not relative), skip
    if (!importPath.startsWith('.') && !importPath.startsWith('/')) {
        return null;
    }
    const basePath = resolve(fromDir, importPath);
    const extensions = ['.ts', '.tsx', '/index.ts', '/index.tsx'];
    // Try exact path first (might already have extension)
    if (existsSync(basePath)) {
        try {
            const stat = statSync(basePath);
            if (stat.isFile())
                return basePath;
        }
        catch {
            // ignore stat errors
        }
    }
    // Try with extensions
    for (const ext of extensions) {
        const candidate = basePath + ext;
        if (existsSync(candidate)) {
            return candidate;
        }
    }
    return null;
}
export async function parseRoute(rootDir, filename, projectId, deploymentId, visitedFiles, mountedSubrouters) {
    // Track visited files to prevent infinite recursion
    const visited = visitedFiles ?? new Set();
    const resolvedFilename = resolve(filename);
    if (visited.has(resolvedFilename)) {
        return []; // Already parsed this file, avoid infinite loop
    }
    visited.add(resolvedFilename);
    const rawContents = await Bun.file(filename).text();
    const version = hash(rawContents);
    // Transpile TypeScript to JavaScript so acorn-loose can parse it properly
    const transpiler = new Bun.Transpiler({ loader: 'ts', target: 'bun' });
    const contents = transpiler.transformSync(rawContents);
    const ast = acornLoose.parse(contents, {
        locations: true,
        ecmaVersion: 'latest',
        sourceType: 'module',
    });
    let exportName;
    let variableName;
    // Extract import statements to map variable names to their import sources
    const importMap = new Map(); // Maps variable name to import path (for backwards compat)
    const importInfoMap = new Map(); // Maps variable name to full import info
    for (const body of ast.body) {
        if (body.type === 'ImportDeclaration') {
            const importDecl = body;
            const importPath = importDecl.source?.value;
            if (importPath && importDecl.specifiers) {
                for (const spec of importDecl.specifiers) {
                    if (spec.type === 'ImportDefaultSpecifier' && spec.local?.name) {
                        // import hello from '@agent/hello'
                        importMap.set(spec.local.name, importPath);
                        importInfoMap.set(spec.local.name, {
                            modulePath: importPath,
                            importedName: 'default',
                            importKind: 'default',
                        });
                    }
                    else if (spec.type === 'ImportSpecifier' && spec.local?.name) {
                        // import { hello } from './shared' or import { hello as h } from './shared'
                        const importedName = spec.imported?.name ?? spec.local.name;
                        importMap.set(spec.local.name, importPath);
                        importInfoMap.set(spec.local.name, {
                            modulePath: importPath,
                            importedName,
                            importKind: 'named',
                        });
                    }
                }
            }
        }
    }
    // Build set of imported names for schema export validation
    const importedNames = new Set(importMap.keys());
    // Build set of exported identifiers for schema export validation
    const exportedNames = buildExportedIdentifierSet(ast);
    // Scan for exported schemas (for WebSocket/SSE routes)
    let exportedInputSchemaName;
    let exportedOutputSchemaName;
    for (const body of ast.body) {
        if (body.type === 'ExportNamedDeclaration') {
            const exportDecl = body;
            if (exportDecl.declaration?.type === 'VariableDeclaration') {
                const varDecl = exportDecl.declaration;
                for (const d of varDecl.declarations) {
                    if (d.id.type === 'Identifier') {
                        const name = d.id.name;
                        if (name === 'inputSchema') {
                            exportedInputSchemaName = name;
                        }
                        else if (name === 'outputSchema') {
                            exportedOutputSchemaName = name;
                        }
                    }
                }
            }
        }
    }
    for (const body of ast.body) {
        if (body.type === 'ExportDefaultDeclaration') {
            const identifier = body.declaration;
            exportName = identifier.name;
            break;
        }
    }
    if (!exportName) {
        throw new InvalidExportError({
            filename,
            message: `could not find default export for ${filename} using ${rootDir}`,
        });
    }
    for (const body of ast.body) {
        if (body.type === 'VariableDeclaration') {
            for (const vardecl of body.declarations) {
                if (vardecl.type === 'VariableDeclarator' && vardecl.id.type === 'Identifier') {
                    const identifier = vardecl.id;
                    if (identifier.name === exportName) {
                        if (vardecl.init?.type === 'CallExpression') {
                            const call = vardecl.init;
                            // Support both createRouter() and new Hono()
                            if (call.callee.name === 'createRouter') {
                                variableName = identifier.name;
                                break;
                            }
                        }
                        else if (vardecl.init?.type === 'NewExpression') {
                            const newExpr = vardecl.init;
                            // Support new Hono() pattern
                            if (newExpr.callee.name === 'Hono') {
                                variableName = identifier.name;
                                break;
                            }
                        }
                    }
                }
            }
        }
    }
    if (!variableName) {
        throw new InvalidCreateRouterError({
            filename,
            message: `error parsing: ${filename}. could not find an proper createRouter or new Hono() defined in this file`,
        });
    }
    const rel = toForwardSlash(relative(rootDir, filename));
    // Compute the API mount path using the shared helper
    // This ensures consistency between route type generation (here) and runtime mounting (entry-generator.ts)
    // Examples:
    //   src/api/index.ts           -> basePath = '/api'
    //   src/api/sessions.ts        -> basePath = '/api/sessions'
    //   src/api/auth/route.ts      -> basePath = '/api/auth'
    //   src/api/users/profile/route.ts -> basePath = '/api/users/profile'
    const srcDir = join(rootDir, 'src');
    const relativeApiPath = extractRelativeApiPath(filename, srcDir);
    const basePath = computeApiMountPath(relativeApiPath);
    const routes = [];
    try {
        for (const body of ast.body) {
            if (body.type === 'ExpressionStatement') {
                const statement = body;
                // Validate that the expression is a call expression (e.g. function call)
                if (statement.expression.type !== 'CallExpression') {
                    continue;
                }
                const callee = statement.expression.callee;
                // Validate that the callee is a member expression (e.g. object.method())
                // This handles cases like 'console.log()' or 'router.get()'
                // direct function calls like 'myFunc()' have type 'Identifier' and will be skipped
                if (callee.type !== 'MemberExpression') {
                    continue;
                }
                if (callee.object.type === 'Identifier' && statement.expression.arguments?.length > 0) {
                    const identifier = callee.object;
                    if (identifier.name === variableName) {
                        let method = callee.property.name;
                        let type = 'api';
                        const action = statement.expression.arguments[0];
                        let suffix = '';
                        let config;
                        // Capture SSE call expression for output schema extraction
                        let sseCallExpr;
                        // Supported HTTP methods that can be represented in BuildMetadata
                        const SUPPORTED_HTTP_METHODS = ['get', 'post', 'put', 'delete', 'patch'];
                        const isSupportedHttpMethod = (m) => SUPPORTED_HTTP_METHODS.includes(m.toLowerCase());
                        switch (method) {
                            case 'use':
                            case 'onError':
                            case 'notFound':
                            case 'basePath':
                            case 'mount': {
                                // Skip Hono middleware, lifecycle handlers, and configuration methods - they don't represent API routes
                                continue;
                            }
                            case 'route': {
                                // router.route(mountPath, subRouterVariable)
                                // Follow the import to discover sub-router routes
                                const mountPathArg = statement.expression.arguments[0];
                                const subRouterArg = statement.expression.arguments[1];
                                // First arg must be a string literal (the mount path)
                                if (!mountPathArg || mountPathArg.type !== 'Literal') {
                                    continue;
                                }
                                // Second arg must be an identifier (the sub-router variable)
                                if (!subRouterArg ||
                                    subRouterArg.type !== 'Identifier') {
                                    continue;
                                }
                                const mountPath = String(mountPathArg.value);
                                const subRouterName = subRouterArg.name;
                                // Look up import path
                                const subRouterImportPath = importMap.get(subRouterName);
                                if (!subRouterImportPath) {
                                    continue; // Can't resolve, skip
                                }
                                // Resolve to actual file path
                                const resolvedFile = resolveImportPath(dirname(filename), subRouterImportPath);
                                if (!resolvedFile || visited.has(resolve(resolvedFile))) {
                                    continue;
                                }
                                try {
                                    // Parse sub-router's routes
                                    const subRoutes = await parseRoute(rootDir, resolvedFile, projectId, deploymentId, visited, mountedSubrouters);
                                    // Track this file as a mounted sub-router
                                    if (mountedSubrouters) {
                                        mountedSubrouters.add(resolve(resolvedFile));
                                    }
                                    // Compute the sub-router's own basePath so we can strip it
                                    const subSrcDir = join(rootDir, 'src');
                                    const subRelativeApiPath = extractRelativeApiPath(resolvedFile, subSrcDir);
                                    const subBasePath = computeApiMountPath(subRelativeApiPath);
                                    // The combined mount point for sub-routes
                                    const combinedBase = joinMountAndRoute(basePath, mountPath);
                                    for (const subRoute of subRoutes) {
                                        // Strip the sub-router's own basePath from the route path
                                        let routeSuffix = subRoute.path;
                                        if (routeSuffix.startsWith(subBasePath)) {
                                            routeSuffix = routeSuffix.slice(subBasePath.length) || '/';
                                        }
                                        const fullPath = joinMountAndRoute(combinedBase, routeSuffix);
                                        const id = generateRouteId(projectId, deploymentId, subRoute.type, subRoute.method, rel, fullPath, subRoute.version);
                                        routes.push({
                                            ...subRoute,
                                            id,
                                            path: fullPath,
                                            filename: rel, // Keep parent file as the filename since routes are mounted here
                                        });
                                    }
                                }
                                catch {
                                    // Sub-router parse failure - skip silently (could be a non-route file)
                                }
                                continue;
                            }
                            case 'on': {
                                // router.on(method | method[], path, handler)
                                // First arg is method(s), second arg is path
                                const methodArg = statement.expression.arguments[0];
                                const pathArg = statement.expression.arguments[1];
                                // Extract methods from first argument
                                const methods = [];
                                if (methodArg && methodArg.type === 'Literal') {
                                    // Single method: router.on('GET', '/path', handler)
                                    const raw = String(methodArg.value || '').toLowerCase();
                                    if (isSupportedHttpMethod(raw)) {
                                        methods.push(raw);
                                    }
                                }
                                else if (methodArg && methodArg.type === 'ArrayExpression') {
                                    // Array of methods: router.on(['GET', 'POST'], '/path', handler)
                                    const arr = methodArg;
                                    for (const el of arr.elements) {
                                        if (!el || el.type !== 'Literal')
                                            continue;
                                        const raw = String(el.value || '').toLowerCase();
                                        if (isSupportedHttpMethod(raw)) {
                                            methods.push(raw);
                                        }
                                    }
                                }
                                // Skip if no supported methods or path is not a literal
                                if (methods.length === 0 ||
                                    !pathArg ||
                                    pathArg.type !== 'Literal') {
                                    continue;
                                }
                                const pathSuffix = String(pathArg.value);
                                // Create a route entry for each method
                                for (const httpMethod of methods) {
                                    const thepath = joinMountAndRoute(basePath, pathSuffix);
                                    const id = generateRouteId(projectId, deploymentId, 'api', httpMethod, rel, thepath, version);
                                    // Check if this route uses validator middleware
                                    const validatorInfo = hasValidatorCall(statement.expression.arguments);
                                    const routeConfig = {};
                                    if (validatorInfo.hasValidator) {
                                        routeConfig.hasValidator = true;
                                        if (validatorInfo.agentVariable) {
                                            routeConfig.agentVariable = validatorInfo.agentVariable;
                                            const agentImportPath = importMap.get(validatorInfo.agentVariable);
                                            if (agentImportPath) {
                                                routeConfig.agentImportPath = agentImportPath;
                                            }
                                        }
                                        // Validate that schema variables are exported (if defined locally)
                                        validateSchemaExports(validatorInfo.inputSchemaVariable, 'input', importedNames, exportedNames, rel, httpMethod, thepath);
                                        validateSchemaExports(validatorInfo.outputSchemaVariable, 'output', importedNames, exportedNames, rel, httpMethod, thepath);
                                        if (validatorInfo.inputSchemaVariable) {
                                            routeConfig.inputSchemaVariable =
                                                validatorInfo.inputSchemaVariable;
                                            // Track where the schema is imported from (if imported)
                                            const inputImportInfo = importInfoMap.get(validatorInfo.inputSchemaVariable);
                                            if (inputImportInfo) {
                                                routeConfig.inputSchemaImportPath = inputImportInfo.modulePath;
                                                routeConfig.inputSchemaImportedName =
                                                    inputImportInfo.importedName;
                                            }
                                        }
                                        if (validatorInfo.outputSchemaVariable) {
                                            routeConfig.outputSchemaVariable =
                                                validatorInfo.outputSchemaVariable;
                                            // Track where the schema is imported from (if imported)
                                            const outputImportInfo = importInfoMap.get(validatorInfo.outputSchemaVariable);
                                            if (outputImportInfo) {
                                                routeConfig.outputSchemaImportPath =
                                                    outputImportInfo.modulePath;
                                                routeConfig.outputSchemaImportedName =
                                                    outputImportInfo.importedName;
                                            }
                                        }
                                        if (validatorInfo.stream !== undefined) {
                                            routeConfig.stream = validatorInfo.stream;
                                        }
                                    }
                                    routes.push({
                                        id,
                                        method: httpMethod,
                                        type: 'api',
                                        filename: rel,
                                        path: thepath,
                                        version,
                                        config: Object.keys(routeConfig).length > 0 ? routeConfig : undefined,
                                    });
                                }
                                continue;
                            }
                            case 'all': {
                                // router.all(path, handler) - matches all HTTP methods
                                // First arg is path (same as get/post/etc.)
                                if (!action || action.type !== 'Literal') {
                                    continue;
                                }
                                const pathSuffix = String(action.value);
                                // Create a route entry for each supported method
                                for (const httpMethod of SUPPORTED_HTTP_METHODS) {
                                    const thepath = joinMountAndRoute(basePath, pathSuffix);
                                    const id = generateRouteId(projectId, deploymentId, 'api', httpMethod, rel, thepath, version);
                                    // Check if this route uses validator middleware
                                    const validatorInfo = hasValidatorCall(statement.expression.arguments);
                                    const routeConfig = {};
                                    if (validatorInfo.hasValidator) {
                                        routeConfig.hasValidator = true;
                                        if (validatorInfo.agentVariable) {
                                            routeConfig.agentVariable = validatorInfo.agentVariable;
                                            const agentImportPath = importMap.get(validatorInfo.agentVariable);
                                            if (agentImportPath) {
                                                routeConfig.agentImportPath = agentImportPath;
                                            }
                                        }
                                        // Validate that schema variables are exported (if defined locally)
                                        validateSchemaExports(validatorInfo.inputSchemaVariable, 'input', importedNames, exportedNames, rel, httpMethod, thepath);
                                        validateSchemaExports(validatorInfo.outputSchemaVariable, 'output', importedNames, exportedNames, rel, httpMethod, thepath);
                                        if (validatorInfo.inputSchemaVariable) {
                                            routeConfig.inputSchemaVariable =
                                                validatorInfo.inputSchemaVariable;
                                            // Track where the schema is imported from (if imported)
                                            const inputImportInfo = importInfoMap.get(validatorInfo.inputSchemaVariable);
                                            if (inputImportInfo) {
                                                routeConfig.inputSchemaImportPath = inputImportInfo.modulePath;
                                                routeConfig.inputSchemaImportedName =
                                                    inputImportInfo.importedName;
                                            }
                                        }
                                        if (validatorInfo.outputSchemaVariable) {
                                            routeConfig.outputSchemaVariable =
                                                validatorInfo.outputSchemaVariable;
                                            // Track where the schema is imported from (if imported)
                                            const outputImportInfo = importInfoMap.get(validatorInfo.outputSchemaVariable);
                                            if (outputImportInfo) {
                                                routeConfig.outputSchemaImportPath =
                                                    outputImportInfo.modulePath;
                                                routeConfig.outputSchemaImportedName =
                                                    outputImportInfo.importedName;
                                            }
                                        }
                                        if (validatorInfo.stream !== undefined) {
                                            routeConfig.stream = validatorInfo.stream;
                                        }
                                    }
                                    routes.push({
                                        id,
                                        method: httpMethod,
                                        type: 'api',
                                        filename: rel,
                                        path: thepath,
                                        version,
                                        config: Object.keys(routeConfig).length > 0 ? routeConfig : undefined,
                                    });
                                }
                                continue;
                            }
                            case 'get':
                            case 'put':
                            case 'post':
                            case 'patch':
                            case 'delete': {
                                if (action && action.type === 'Literal') {
                                    suffix = String(action.value);
                                    // Check if any argument is a middleware function call (websocket, sse, stream, cron)
                                    // New pattern: router.get('/ws', websocket((c, ws) => { ... }))
                                    for (const arg of statement.expression.arguments) {
                                        if (arg.type === 'CallExpression') {
                                            const callExpr = arg;
                                            // Only handle simple Identifier callees (e.g., websocket(), sse())
                                            // Skip MemberExpression callees (e.g., obj.method())
                                            if (callExpr.callee.type !== 'Identifier') {
                                                continue;
                                            }
                                            const calleeName = callExpr.callee.name;
                                            if (calleeName === 'websocket' ||
                                                calleeName === 'sse' ||
                                                calleeName === 'stream') {
                                                type = calleeName;
                                                // Capture SSE call expression for output schema extraction
                                                if (calleeName === 'sse') {
                                                    sseCallExpr = callExpr;
                                                }
                                                break;
                                            }
                                            if (calleeName === 'cron') {
                                                type = 'cron';
                                                // First argument to cron() is the schedule expression
                                                if (callExpr.arguments && callExpr.arguments.length > 0) {
                                                    const cronArg = callExpr.arguments[0];
                                                    if (cronArg.type === 'Literal') {
                                                        const expression = String(cronArg.value);
                                                        try {
                                                            parseCronExpression(expression, {
                                                                hasSeconds: false,
                                                            });
                                                        }
                                                        catch (ex) {
                                                            throw new InvalidRouterConfigError({
                                                                filename,
                                                                cause: ex,
                                                                line: body.loc?.start?.line,
                                                                message: `invalid cron expression "${expression}" in ${filename} at line ${body.loc?.start?.line}`,
                                                            });
                                                        }
                                                        config = { expression };
                                                    }
                                                }
                                                break;
                                            }
                                        }
                                    }
                                }
                                else {
                                    throw new InvalidRouterConfigError({
                                        filename,
                                        line: body.loc?.start?.line,
                                        message: `unsupported HTTP method ${method} in ${filename} at line ${body.loc?.start?.line}`,
                                    });
                                }
                                break;
                            }
                            case 'stream':
                            case 'sse':
                            case 'websocket': {
                                // DEPRECATED: router.stream(), router.sse(), router.websocket()
                                // These methods now throw errors at runtime
                                type = method;
                                method = 'post';
                                const theaction = action;
                                if (theaction.type === 'Literal') {
                                    suffix = String(theaction.value);
                                    break;
                                }
                                break;
                            }
                            case 'cron': {
                                // DEPRECATED: router.cron()
                                // This method now throws errors at runtime
                                type = method;
                                method = 'post';
                                const theaction = action;
                                if (theaction.type === 'Literal') {
                                    const expression = String(theaction.value);
                                    try {
                                        parseCronExpression(expression, { hasSeconds: false });
                                    }
                                    catch (ex) {
                                        throw new InvalidRouterConfigError({
                                            filename,
                                            cause: ex,
                                            line: body.loc?.start?.line,
                                            message: `invalid cron expression "${expression}" in ${filename} at line ${body.loc?.start?.line}`,
                                        });
                                    }
                                    suffix = hash(expression);
                                    config = { expression };
                                    break;
                                }
                                break;
                            }
                            default: {
                                throw new InvalidRouterConfigError({
                                    filename,
                                    line: body.loc?.start?.line,
                                    message: `unsupported router method ${method} in ${filename} at line ${body.loc?.start?.line}`,
                                });
                            }
                        }
                        const thepath = joinMountAndRoute(basePath, suffix);
                        const id = generateRouteId(projectId, deploymentId, type, method, rel, thepath, version);
                        // Check if this route uses validator middleware
                        const validatorInfo = hasValidatorCall(statement.expression.arguments);
                        // Store validator info in config if present
                        const routeConfig = config ? { ...config } : {};
                        if (validatorInfo.hasValidator) {
                            routeConfig.hasValidator = true;
                            if (validatorInfo.agentVariable) {
                                routeConfig.agentVariable = validatorInfo.agentVariable;
                                // Look up where this agent variable is imported from
                                const agentImportPath = importMap.get(validatorInfo.agentVariable);
                                if (agentImportPath) {
                                    routeConfig.agentImportPath = agentImportPath;
                                }
                            }
                            // Validate that schema variables are exported (if defined locally)
                            validateSchemaExports(validatorInfo.inputSchemaVariable, 'input', importedNames, exportedNames, rel, method, thepath);
                            validateSchemaExports(validatorInfo.outputSchemaVariable, 'output', importedNames, exportedNames, rel, method, thepath);
                            if (validatorInfo.inputSchemaVariable) {
                                routeConfig.inputSchemaVariable = validatorInfo.inputSchemaVariable;
                                // Track where the schema is imported from (if imported)
                                const inputImportInfo = importInfoMap.get(validatorInfo.inputSchemaVariable);
                                if (inputImportInfo) {
                                    routeConfig.inputSchemaImportPath = inputImportInfo.modulePath;
                                    routeConfig.inputSchemaImportedName = inputImportInfo.importedName;
                                }
                            }
                            if (validatorInfo.outputSchemaVariable) {
                                routeConfig.outputSchemaVariable = validatorInfo.outputSchemaVariable;
                                // Track where the schema is imported from (if imported)
                                const outputImportInfo = importInfoMap.get(validatorInfo.outputSchemaVariable);
                                if (outputImportInfo) {
                                    routeConfig.outputSchemaImportPath = outputImportInfo.modulePath;
                                    routeConfig.outputSchemaImportedName = outputImportInfo.importedName;
                                }
                            }
                            if (validatorInfo.stream !== undefined) {
                                routeConfig.stream = validatorInfo.stream;
                            }
                        }
                        // Extract output schema from SSE options: sse({ output: schema }, handler)
                        // For SSE routes, the sse({ output }) pattern takes precedence over any
                        // validator-provided schema. Imported schemas need not be exported, but
                        // locally-defined schemas must be exported and are validated below.
                        if (sseCallExpr) {
                            const sseSchemaInfo = extractSSEOutputSchema(sseCallExpr);
                            if (sseSchemaInfo.outputSchemaVariable) {
                                // Track where the schema is imported from (if imported)
                                const outputImportInfo = importInfoMap.get(sseSchemaInfo.outputSchemaVariable);
                                // Validate that locally-defined schemas are exported
                                // (skip validation if schema is imported from another module)
                                if (!outputImportInfo) {
                                    validateSchemaExports(sseSchemaInfo.outputSchemaVariable, 'output', importedNames, exportedNames, rel, method, thepath);
                                }
                                // Override any validator-provided schema with SSE-specific schema
                                routeConfig.outputSchemaVariable = sseSchemaInfo.outputSchemaVariable;
                                if (outputImportInfo) {
                                    routeConfig.outputSchemaImportPath = outputImportInfo.modulePath;
                                    routeConfig.outputSchemaImportedName = outputImportInfo.importedName;
                                }
                                else {
                                    // Clear any validator-provided import info since we're using local schema
                                    delete routeConfig.outputSchemaImportPath;
                                    delete routeConfig.outputSchemaImportedName;
                                }
                            }
                        }
                        // Fall back to exported schemas when validator doesn't provide them
                        // This works for all route types (API, WebSocket, SSE, stream)
                        // For API routes, this enables `export const outputSchema` pattern
                        // which is useful when using zValidator (input-only) but needing typed outputs
                        if (!routeConfig.inputSchemaVariable && exportedInputSchemaName) {
                            routeConfig.inputSchemaVariable = exportedInputSchemaName;
                            // Check if exported schema name is also imported
                            const inputImportInfo = importInfoMap.get(exportedInputSchemaName);
                            if (inputImportInfo) {
                                routeConfig.inputSchemaImportPath = inputImportInfo.modulePath;
                                routeConfig.inputSchemaImportedName = inputImportInfo.importedName;
                            }
                        }
                        if (!routeConfig.outputSchemaVariable && exportedOutputSchemaName) {
                            routeConfig.outputSchemaVariable = exportedOutputSchemaName;
                            // Check if exported schema name is also imported
                            const outputImportInfo = importInfoMap.get(exportedOutputSchemaName);
                            if (outputImportInfo) {
                                routeConfig.outputSchemaImportPath = outputImportInfo.modulePath;
                                routeConfig.outputSchemaImportedName = outputImportInfo.importedName;
                            }
                        }
                        routes.push({
                            id,
                            method: method,
                            type: type,
                            filename: rel,
                            path: thepath,
                            version,
                            config: Object.keys(routeConfig).length > 0 ? routeConfig : undefined,
                        });
                    }
                }
            }
        }
    }
    catch (error) {
        if (error instanceof InvalidRouterConfigError || error instanceof SchemaNotExportedError) {
            throw error;
        }
        throw new InvalidRouterConfigError({
            filename,
            cause: error,
        });
    }
    return routes;
}
/**
 * Check if a TypeScript file actively uses a specific function
 * (ignores comments and unused imports)
 *
 * @param content - The TypeScript source code
 * @param functionName - The function name to check for (e.g., 'createWorkbench')
 * @returns true if the function is both imported and called
 */
export function checkFunctionUsage(content, functionName) {
    try {
        const sourceFile = ts.createSourceFile('temp.ts', content, ts.ScriptTarget.Latest, true);
        let hasImport = false;
        let hasUsage = false;
        function visitNode(node) {
            // Check for import declarations with the function
            if (ts.isImportDeclaration(node) && node.importClause?.namedBindings) {
                if (ts.isNamedImports(node.importClause.namedBindings)) {
                    for (const element of node.importClause.namedBindings.elements) {
                        if (element.name.text === functionName) {
                            hasImport = true;
                        }
                    }
                }
            }
            // Check for function calls
            if (ts.isCallExpression(node) && ts.isIdentifier(node.expression)) {
                if (node.expression.text === functionName) {
                    hasUsage = true;
                }
            }
            // Recursively visit child nodes
            ts.forEachChild(node, visitNode);
        }
        visitNode(sourceFile);
        // Only return true if both import and usage are present
        return hasImport && hasUsage;
    }
    catch (error) {
        // Fallback to string check if AST parsing fails
        logger.warn(`AST parsing failed for ${functionName}, falling back to string check:`, error);
        return content.includes(functionName);
    }
}
/**
 * Check if app.ts contains conflicting routes for a given endpoint
 */
export function checkRouteConflicts(content, workbenchEndpoint) {
    try {
        const sourceFile = ts.createSourceFile('app.ts', content, ts.ScriptTarget.Latest, true);
        let hasConflict = false;
        function visitNode(node) {
            // Check for router.get calls
            if (ts.isCallExpression(node) &&
                ts.isPropertyAccessExpression(node.expression) &&
                ts.isIdentifier(node.expression.name) &&
                node.expression.name.text === 'get') {
                // Check if first argument is the workbench endpoint
                const firstArg = node.arguments[0];
                if (node.arguments.length > 0 && firstArg && ts.isStringLiteral(firstArg)) {
                    if (firstArg.text === workbenchEndpoint) {
                        hasConflict = true;
                    }
                }
            }
            ts.forEachChild(node, visitNode);
        }
        visitNode(sourceFile);
        return hasConflict;
    }
    catch (_error) {
        return false;
    }
}
/**
 * Extract AppState type from setup() return value in createApp call
 *
 * @param content - The TypeScript source code from app.ts
 * @returns Type definition string or null if no setup found
 */
export function extractAppStateType(content) {
    try {
        const sourceFile = ts.createSourceFile('app.ts', content, ts.ScriptTarget.Latest, true, ts.ScriptKind.TS);
        let appStateType = null;
        let foundCreateApp = false;
        let foundSetup = false;
        let exportedSetupFunc;
        function visitNode(node) {
            // Look for createApp call expression (can be on await expression)
            let callExpr;
            if (ts.isCallExpression(node) && ts.isIdentifier(node.expression)) {
                if (node.expression.text === 'createApp') {
                    foundCreateApp = true;
                    callExpr = node;
                }
            }
            else if (ts.isAwaitExpression(node) && ts.isCallExpression(node.expression)) {
                const call = node.expression;
                if (ts.isIdentifier(call.expression) && call.expression.text === 'createApp') {
                    foundCreateApp = true;
                    callExpr = call;
                }
            }
            if (callExpr) {
                // Check if it has a config object argument
                const configArg = callExpr.arguments[0];
                if (callExpr.arguments.length > 0 && configArg) {
                    if (ts.isObjectLiteralExpression(configArg)) {
                        // Find setup property
                        for (const prop of configArg.properties) {
                            if (ts.isPropertyAssignment(prop) &&
                                ts.isIdentifier(prop.name) &&
                                prop.name.text === 'setup') {
                                foundSetup = true;
                                // Found setup function - extract return type
                                const setupFunc = prop.initializer;
                                if (ts.isFunctionExpression(setupFunc) || ts.isArrowFunction(setupFunc)) {
                                    // Find return statement
                                    const returnObj = findReturnObject(setupFunc);
                                    if (returnObj) {
                                        appStateType = objectLiteralToTypeDefinition(returnObj, sourceFile);
                                    }
                                    else {
                                        logger.debug('No return object found in setup function');
                                    }
                                }
                                else {
                                    logger.debug(`Setup is not a function expression or arrow function, it's: ${ts.SyntaxKind[setupFunc.kind]}`);
                                }
                            }
                        }
                    }
                }
            }
            // Also record exported setup function
            if (ts.isFunctionDeclaration(node) &&
                node.name &&
                node.name.text === 'setup' &&
                node.modifiers?.some((m) => m.kind === ts.SyntaxKind.ExportKeyword)) {
                exportedSetupFunc = node;
            }
            ts.forEachChild(node, visitNode);
        }
        function findReturnObject(func) {
            let returnObject = null;
            // Handle arrow function with expression body: () => ({ ... })
            if (ts.isArrowFunction(func) && !ts.isBlock(func.body)) {
                const bodyExpr = func.body;
                // Handle parenthesized expression: () => ({ ... })
                const expr = ts.isParenthesizedExpression(bodyExpr) ? bodyExpr.expression : bodyExpr;
                if (ts.isObjectLiteralExpression(expr)) {
                    return expr;
                }
                if (ts.isIdentifier(expr)) {
                    // Support: const state = {...}; const setup = () => state;
                    // Walk up to find the enclosing source file or statement list
                    let scope = func;
                    while (scope && !ts.isSourceFile(scope) && !ts.isBlock(scope)) {
                        scope = scope.parent;
                    }
                    if (scope) {
                        findVariableDeclaration(scope, expr.text);
                    }
                    return returnObject;
                }
                // For other expressions, can't extract type
                return null;
            }
            function visitFuncNode(node) {
                if (ts.isReturnStatement(node) && node.expression) {
                    // Handle direct object literal
                    if (ts.isObjectLiteralExpression(node.expression)) {
                        returnObject = node.expression;
                    }
                    // Handle variable reference (const state = {...}; return state;)
                    else if (ts.isIdentifier(node.expression)) {
                        // Try to find the variable declaration
                        const varName = node.expression.text;
                        // Walk back through the function to find the declaration
                        if (func.body && ts.isBlock(func.body)) {
                            findVariableDeclaration(func.body, varName);
                        }
                    }
                }
                ts.forEachChild(node, visitFuncNode);
            }
            function findVariableDeclaration(body, varName) {
                function visitForVar(node) {
                    if (ts.isVariableStatement(node)) {
                        for (const decl of node.declarationList.declarations) {
                            if (ts.isIdentifier(decl.name) && decl.name.text === varName) {
                                if (decl.initializer && ts.isObjectLiteralExpression(decl.initializer)) {
                                    returnObject = decl.initializer;
                                }
                            }
                        }
                    }
                    ts.forEachChild(node, visitForVar);
                }
                visitForVar(body);
            }
            if (func.body && ts.isBlock(func.body)) {
                visitFuncNode(func.body);
            }
            return returnObject;
        }
        function objectLiteralToTypeDefinition(obj, sourceFile) {
            const properties = [];
            for (const prop of obj.properties) {
                if (ts.isPropertyAssignment(prop)) {
                    const name = prop.name.getText(sourceFile);
                    const value = prop.initializer;
                    const typeStr = inferTypeFromValue(value, sourceFile);
                    properties.push(`\t${name}: ${typeStr};`);
                }
                else if (ts.isShorthandPropertyAssignment(prop)) {
                    const name = prop.name.getText(sourceFile);
                    properties.push(`\t${name}: unknown;`);
                }
            }
            return `{\n${properties.join('\n')}\n}`;
        }
        function inferTypeFromValue(value, sourceFile) {
            if (ts.isStringLiteral(value)) {
                return 'string';
            }
            if (ts.isNumericLiteral(value)) {
                return 'number';
            }
            if (value.kind === ts.SyntaxKind.TrueKeyword ||
                value.kind === ts.SyntaxKind.FalseKeyword) {
                return 'boolean';
            }
            if (ts.isNewExpression(value) && ts.isIdentifier(value.expression)) {
                if (value.expression.text === 'Date') {
                    return 'Date';
                }
            }
            if (ts.isObjectLiteralExpression(value)) {
                return objectLiteralToTypeDefinition(value, sourceFile);
            }
            if (ts.isArrayLiteralExpression(value)) {
                return 'unknown[]';
            }
            return 'unknown';
        }
        visitNode(sourceFile);
        // If no inline setup found but we have an exported setup function, use that
        if (foundCreateApp && !foundSetup && exportedSetupFunc) {
            foundSetup = true;
            const returnObj = findReturnObject(exportedSetupFunc);
            if (returnObj) {
                appStateType = objectLiteralToTypeDefinition(returnObj, sourceFile);
            }
            else {
                logger.debug('Exported setup function found but no return object');
            }
        }
        if (!foundCreateApp) {
            logger.debug('Did not find createApp call in app.ts');
        }
        else if (!foundSetup) {
            logger.debug('Found createApp but no setup property');
        }
        else if (!appStateType) {
            logger.debug('Found createApp and setup but could not extract type');
        }
        return appStateType;
    }
    catch (error) {
        logger.warn('AppState type extraction failed:', error);
        return null;
    }
}
/**
 * Update tsconfig.json to add path mapping for @agentuity/runtime
 *
 * @param rootDir - Root directory of the project
 * @param shouldAdd - If true, add the mapping; if false, remove it
 */
async function updateTsconfigPathMapping(rootDir, shouldAdd) {
    const tsconfigPath = join(rootDir, 'tsconfig.json');
    if (!(await Bun.file(tsconfigPath).exists())) {
        logger.debug('No tsconfig.json found, skipping path mapping update');
        return;
    }
    try {
        const tsconfigContent = await Bun.file(tsconfigPath).text();
        // Use JSON5 to parse tsconfig.json (handles comments in input)
        const tsconfig = JSON5.parse(tsconfigContent);
        const _before = JSON.stringify(tsconfig);
        // Initialize compilerOptions and paths if they don't exist
        if (!tsconfig.compilerOptions) {
            tsconfig.compilerOptions = {};
        }
        if (!tsconfig.compilerOptions.paths) {
            tsconfig.compilerOptions.paths = {};
        }
        if (shouldAdd) {
            // Add or update the path mapping
            tsconfig.compilerOptions.paths['@agentuity/runtime'] = ['./src/generated/router.ts'];
            logger.debug('Added @agentuity/runtime path mapping to tsconfig.json');
        }
        else {
            // Remove the path mapping if it exists
            if (tsconfig.compilerOptions.paths['@agentuity/runtime']) {
                delete tsconfig.compilerOptions.paths['@agentuity/runtime'];
                logger.debug('Removed @agentuity/runtime path mapping from tsconfig.json');
            }
            // Clean up empty paths object
            if (Object.keys(tsconfig.compilerOptions.paths).length === 0) {
                delete tsconfig.compilerOptions.paths;
            }
        }
        const _after = JSON.stringify(tsconfig);
        if (_before === _after) {
            return;
        }
        // Write back using standard JSON (TypeScript requires strict JSON format)
        await Bun.write(tsconfigPath, JSON.stringify(tsconfig, null, '\t') + '\n');
    }
    catch (error) {
        logger.warn('Failed to update tsconfig.json:', error);
    }
}
const RuntimePackageNotFound = StructuredError('RuntimePackageNotFound');
/**
 * Generate lifecycle type files (src/generated/state.ts and src/generated/router.ts)
 *
 * @param rootDir - Root directory of the project
 * @param outDir - Output directory (typically src/generated/)
 * @param appFilePath - Path to app.ts file
 * @returns true if files were generated, false if no setup found
 */
export async function generateLifecycleTypes(rootDir, outDir, appFilePath) {
    const appContent = await Bun.file(appFilePath).text();
    if (typeof appContent !== 'string') {
        return false;
    }
    const appStateType = extractAppStateType(appContent);
    if (!appStateType) {
        logger.debug('No setup() function found in app.ts, skipping lifecycle type generation');
        // Remove path mapping if no setup found
        await updateTsconfigPathMapping(rootDir, false);
        return false;
    }
    // Ensure output directory exists (now src/generated instead of .agentuity)
    if (!existsSync(outDir)) {
        mkdirSync(outDir, { recursive: true });
    }
    // Find @agentuity/runtime by walking up directory tree
    // This works in any project structure - monorepos, nested projects, etc.
    let runtimePkgPath = null;
    let currentDir = rootDir;
    const searchedPaths = [];
    while (currentDir && currentDir !== '/' && currentDir !== '.') {
        const candidatePath = join(currentDir, 'node_modules', '@agentuity', 'runtime');
        searchedPaths.push(candidatePath);
        if (existsSync(candidatePath)) {
            runtimePkgPath = candidatePath;
            logger.debug(`Found runtime package at: ${candidatePath}`);
            break;
        }
        // Try packages/ for monorepo source layout
        const packagesPath = join(currentDir, 'packages', 'runtime');
        searchedPaths.push(packagesPath);
        if (existsSync(packagesPath)) {
            runtimePkgPath = packagesPath;
            logger.debug(`Found runtime package (source) at: ${packagesPath}`);
            break;
        }
        // Move up one directory
        const parent = dirname(currentDir);
        if (parent === currentDir)
            break; // Reached root
        currentDir = parent;
    }
    if (!runtimePkgPath) {
        throw new RuntimePackageNotFound({
            message: `@agentuity/runtime package not found.\n` +
                `Searched paths:\n${searchedPaths.map((p) => `  - ${p}`).join('\n')}\n` +
                `Make sure dependencies are installed by running 'bun install' or 'npm install'`,
        });
    }
    let runtimeImportPath = null;
    // Calculate relative path from src/generated/ to the package location
    // Don't resolve symlinks - we want to use the symlink path so it works in both
    // local dev (symlinked to packages/) and CI (actual node_modules)
    if (existsSync(runtimePkgPath)) {
        // Calculate relative path from src/generated/ to node_modules package
        const relPath = toForwardSlash(relative(outDir, runtimePkgPath));
        runtimeImportPath = relPath;
        logger.debug(`Using relative path to runtime package: ${relPath}`);
    }
    else {
        throw new RuntimePackageNotFound({
            message: `Failed to access @agentuity/runtime package at ${runtimePkgPath}\n` +
                `Make sure dependencies are installed`,
        });
    }
    if (!runtimeImportPath) {
        throw new RuntimePackageNotFound({
            message: `Failed to determine import path for @agentuity/runtime`,
        });
    }
    // Now generate state.ts with AppState type
    // NOTE: We can ONLY augment the package name, not relative paths
    // TypeScript resolves @agentuity/runtime through path mapping -> wrapper -> actual package
    const typesContent = `// @generated
// AUTO-GENERATED from app.ts setup() return type
// This file is auto-generated by the build tool - do not edit manually

/**
 * Application state type inferred from your createApp setup function.
 * This type is automatically generated and available throughout your app via ctx.app.
 *
 * @example
 * \`\`\`typescript
 * // In your agents:
 * const agent = createAgent({
 *   handler: async (ctx, input) => {
 *     // ctx.app is strongly typed as GeneratedAppState
 *     const value = ctx.app; // All properties from your setup return value
 *     return 'result';
 *   }
 * });
 * \`\`\`
 */
export type GeneratedAppState = ${appStateType};

// Augment the @agentuity/runtime module with AppState
// This will be picked up when imported through the wrapper
declare module '@agentuity/runtime' {
	interface AppState extends GeneratedAppState {}
}

// FOUND AN ERROR IN THIS FILE?
// Please file an issue at https://github.com/agentuity/sdk/issues
// or if you know the fix please submit a PR!
`;
    const typesPath = join(outDir, 'state.ts');
    await Bun.write(typesPath, typesContent);
    logger.debug(`Generated lifecycle types: ${typesPath}`);
    const wrapperContent = `// @generated
// AUTO-GENERATED runtime wrapper
// This file is auto-generated by the build tool - do not edit manually

// Import augmentations file (NOT type-only) to trigger module augmentation
import type { GeneratedAppState } from './state';
import './state';

// Import from actual package location
import { createRouter as baseCreateRouter, type Env } from '${runtimeImportPath}/src/index';
import type { Hono } from 'hono';

// Type aliases to avoid repeating the generic parameter
type AppEnv = Env<GeneratedAppState>;
type AppRouter = Hono<AppEnv>;

/**
 * Creates a Hono router with extended methods for Agentuity-specific routing patterns.
 *
 * In addition to standard HTTP methods (get, post, put, delete, patch), the router includes:
 * - **stream()** - Stream responses with ReadableStream
 * - **websocket()** - WebSocket connections
 * - **sse()** - Server-Sent Events
 * - **email()** - Email handler routing
 * - **sms()** - SMS handler routing
 * - **cron()** - Scheduled task routing
 *
 * @returns Extended Hono router with custom methods and app state typing
 *
 * @example
 * \`\`\`typescript
 * const router = createRouter();
 *
 * // Standard HTTP routes
 * router.get('/hello', (c) => c.text('Hello!'));
 * router.post('/data', async (c) => {
 *   const body = await c.req.json();
 *   return c.json({ received: body });
 * });
 *
 * // Access app state (strongly typed!)
 * router.get('/db', (c) => {
 *   const db = c.var.app; // Your app state from createApp setup
 *   return c.json({ connected: true });
 * });
 * \`\`\`
 */
export function createRouter(): AppRouter {
	return baseCreateRouter() as unknown as AppRouter;
}

// Re-export everything else
export * from '${runtimeImportPath}/src/index';

// FOUND AN ERROR IN THIS FILE?
// Please file an issue at https://github.com/agentuity/sdk/issues
// or if you know the fix please submit a PR!
`;
    const wrapperPath = join(outDir, 'router.ts');
    await Bun.write(wrapperPath, wrapperContent);
    logger.debug(`Generated lifecycle wrapper: ${wrapperPath}`);
    // Update tsconfig.json with path mapping
    await updateTsconfigPathMapping(rootDir, true);
    return true;
}
/**
 * Analyze workbench usage and extract configuration
 *
 * @param content - The TypeScript source code
 * @returns workbench analysis including usage and config
 */
export function analyzeWorkbench(content) {
    try {
        if (!content.includes('@agentuity/workbench')) {
            return {
                hasWorkbench: false,
                config: null,
            };
        }
        const sourceFile = ts.createSourceFile('app.ts', content, ts.ScriptTarget.Latest, true);
        let hasImport = false;
        const workbenchVariables = new Map();
        let usedInServices = false;
        function visitNode(node) {
            // Check for import declarations with createWorkbench
            if (ts.isImportDeclaration(node) && node.importClause?.namedBindings) {
                if (ts.isNamedImports(node.importClause.namedBindings)) {
                    for (const element of node.importClause.namedBindings.elements) {
                        if (element.name.text === 'createWorkbench') {
                            hasImport = true;
                        }
                    }
                }
            }
            // Track variables assigned from createWorkbench() calls
            if (ts.isVariableDeclaration(node) &&
                node.initializer &&
                ts.isCallExpression(node.initializer) &&
                ts.isIdentifier(node.initializer.expression) &&
                node.initializer.expression.text === 'createWorkbench') {
                // Extract variable name
                if (ts.isIdentifier(node.name)) {
                    // Extract configuration from the first argument (if any)
                    let varConfig;
                    const firstInitArg = node.initializer.arguments[0];
                    if (node.initializer.arguments.length > 0 && firstInitArg) {
                        varConfig = parseConfigObject(firstInitArg) || { route: '/workbench' };
                    }
                    else {
                        // Default config if no arguments provided
                        varConfig = { route: '/workbench' };
                    }
                    workbenchVariables.set(node.name.text, varConfig);
                }
            }
            // Check if workbench variable is used in createApp's services config
            if (ts.isCallExpression(node) &&
                ts.isIdentifier(node.expression) &&
                node.expression.text === 'createApp' &&
                node.arguments.length > 0) {
                const createAppConfigArg = node.arguments[0];
                if (createAppConfigArg && ts.isObjectLiteralExpression(createAppConfigArg)) {
                    // Find the services property
                    for (const prop of createAppConfigArg.properties) {
                        if (ts.isPropertyAssignment(prop) &&
                            ts.isIdentifier(prop.name) &&
                            prop.name.text === 'services') {
                            // Check if any workbench variable is referenced in services
                            const foundVariableName = checkForWorkbenchUsage(prop.initializer, workbenchVariables);
                            if (foundVariableName) {
                                usedInServices = true;
                            }
                            break;
                        }
                    }
                }
            }
            // Recursively visit child nodes
            ts.forEachChild(node, visitNode);
        }
        // Helper function to check if workbench variable is used in services config
        // Returns the variable name if found, otherwise null
        function checkForWorkbenchUsage(node, variables) {
            let foundVar = null;
            function visit(n) {
                if (foundVar)
                    return; // Already found
                // Check for identifier references
                if (ts.isIdentifier(n) && variables.has(n.text)) {
                    foundVar = n.text;
                    return;
                }
                // Check for property shorthand: { workbench } in services
                if (ts.isShorthandPropertyAssignment(n) && variables.has(n.name.text)) {
                    foundVar = n.name.text;
                    return;
                }
                // Check for property value: { workbench: workbench } in services
                if (ts.isPropertyAssignment(n) &&
                    n.initializer &&
                    ts.isIdentifier(n.initializer) &&
                    variables.has(n.initializer.text)) {
                    foundVar = n.initializer.text;
                    return;
                }
                ts.forEachChild(n, visit);
            }
            visit(node);
            return foundVar;
        }
        visitNode(sourceFile);
        // Get the config from the first used workbench variable
        let config = null;
        if (hasImport && usedInServices) {
            // Re-check which variable was used
            const ast = sourceFile;
            for (const [varName, varConfig] of workbenchVariables.entries()) {
                // Simple check: walk through and find if this variable is used in services
                let used = false;
                function checkUsage(node) {
                    if (ts.isCallExpression(node) &&
                        ts.isIdentifier(node.expression) &&
                        node.expression.text === 'createApp' &&
                        node.arguments.length > 0) {
                        const checkConfigArg = node.arguments[0];
                        if (checkConfigArg && ts.isObjectLiteralExpression(checkConfigArg)) {
                            for (const prop of checkConfigArg.properties) {
                                if (ts.isPropertyAssignment(prop) &&
                                    ts.isIdentifier(prop.name) &&
                                    prop.name.text === 'services') {
                                    const foundVar = checkForWorkbenchUsage(prop.initializer, workbenchVariables);
                                    if (foundVar === varName) {
                                        used = true;
                                        config = varConfig;
                                    }
                                    break;
                                }
                            }
                        }
                    }
                    ts.forEachChild(node, checkUsage);
                }
                checkUsage(ast);
                if (used)
                    break;
            }
        }
        return {
            hasWorkbench: hasImport && usedInServices,
            config: config,
        };
    }
    catch (error) {
        // Fallback to simple check if AST parsing fails
        logger.warn('Workbench AST parsing failed, falling back to string check:', error);
        const hasWorkbench = content.includes('createWorkbench');
        return {
            hasWorkbench,
            config: hasWorkbench ? { route: '/workbench' } : null,
        };
    }
}
/**
 * Parse a TypeScript object literal to extract configuration
 */
function parseConfigObject(node) {
    if (!ts.isObjectLiteralExpression(node)) {
        return { route: '/workbench' }; // Default config
    }
    const config = { route: '/workbench' };
    for (const property of node.properties) {
        if (ts.isPropertyAssignment(property) && ts.isIdentifier(property.name)) {
            const propertyName = property.name.text;
            if (propertyName === 'route' && ts.isStringLiteral(property.initializer)) {
                config.route = property.initializer.text;
            }
            else if (propertyName === 'headers' &&
                ts.isObjectLiteralExpression(property.initializer)) {
                // Parse headers object if needed (not implemented for now)
                config.headers = {};
            }
        }
    }
    return config;
}
/**
 * Find the end position of createApp call statement in the source code
 * Uses AST parsing to reliably find the complete statement including await/const assignment
 *
 * @param content - The source code content
 * @returns The character position after the createApp statement, or -1 if not found
 */
export function findCreateAppEndPosition(content) {
    try {
        const ast = acornLoose.parse(content, {
            ecmaVersion: 'latest',
            sourceType: 'module',
        });
        // Walk through all top-level statements
        for (const node of ast.body) {
            let targetNode;
            // Check for: const app = await createApp(...)
            if (node.type === 'VariableDeclaration') {
                const varDecl = node;
                for (const declarator of varDecl.declarations) {
                    if (declarator.init) {
                        // Handle await createApp(...)
                        if (declarator.init.type === 'AwaitExpression') {
                            const awaitExpr = declarator.init;
                            if (awaitExpr.argument?.type === 'CallExpression' &&
                                isCreateAppCall(awaitExpr.argument)) {
                                targetNode = node;
                                break;
                            }
                        }
                        // Handle createApp(...) without await
                        else if (declarator.init.type === 'CallExpression') {
                            if (isCreateAppCall(declarator.init)) {
                                targetNode = node;
                                break;
                            }
                        }
                    }
                }
            }
            // Check for: await createApp(...)
            else if (node.type === 'ExpressionStatement') {
                const exprStmt = node;
                if (exprStmt.expression.type === 'AwaitExpression') {
                    const awaitExpr = exprStmt.expression;
                    if (awaitExpr.argument?.type === 'CallExpression' &&
                        isCreateAppCall(awaitExpr.argument)) {
                        targetNode = node;
                    }
                }
                else if (exprStmt.expression.type === 'CallExpression') {
                    if (isCreateAppCall(exprStmt.expression)) {
                        targetNode = node;
                    }
                }
            }
            if (targetNode && targetNode.end !== undefined) {
                // Find the semicolon after the statement (if it exists)
                const afterStmt = content.slice(targetNode.end);
                const semiMatch = afterStmt.match(/^\s*;/);
                if (semiMatch) {
                    return targetNode.end + semiMatch[0].length;
                }
                // No semicolon, return end of statement
                return targetNode.end;
            }
        }
        return -1;
    }
    catch (error) {
        logger.warn('Failed to parse AST for createApp detection:', error);
        return -1;
    }
}
/**
 * Check if a CallExpression is a call to createApp
 */
function isCreateAppCall(node) {
    const callee = node.callee;
    if (callee.type === 'Identifier') {
        const id = callee;
        return id.name === 'createApp';
    }
    return false;
}
//# sourceMappingURL=ast.js.map