/**
 * Map of process names to internal agent short names.
 * The key is the process name (or substring) that appears in the parent process path or command line.
 * The value is the internal short name used to identify the agent.
 *
 * Process names verified via `agentuity cloud sandbox run --runtime <agent>:latest`:
 * - opencode: binary 'opencode' (from bun install -g opencode-ai)
 * - codex: binary 'codex' (from npm install -g @openai/codex)
 * - cursor: binary 'cursor-agent' (from curl installer)
 * - claude-code: binary 'claude', shows as 'node /usr/local/bin/claude'
 * - copilot: binary 'copilot', shows as 'node /usr/local/bin/copilot' and spawns native binary
 * - gemini: binary 'gemini', shows as 'node /usr/local/bin/gemini'
 * - amp: binary 'amp', shows as 'node --no-warnings /usr/local/bin/amp'
 *
 * IMPORTANT: Order matters! More specific patterns should come before less specific ones.
 * For example, 'opencode' must be checked before 'code' to avoid false matches.
 */
export declare const KNOWN_AGENTS: [string, string][];
export type KnownAgent = (typeof KNOWN_AGENTS)[number][1];
/**
 * Display names for known agents (human-friendly names)
 */
export declare const AGENT_DISPLAY_NAMES: Record<string, string>;
/**
 * Get the display name for an agent ID
 */
export declare function getAgentDisplayName(agentId: string): string;
/**
 * Get the executing agent if the CLI is being run from a known coding agent.
 * Returns the agent ID if detected, undefined otherwise.
 *
 * This function runs synchronously using FFI for fast process path resolution.
 * Results are cached after the first call.
 *
 * @example
 * ```typescript
 * const agent = getExecutingAgent();
 * if (agent) {
 *   logger.debug(`Running from agent: ${agent}`);
 * }
 * ```
 */
export declare function getExecutingAgent(): string | undefined;
/**
 * Get environment variables to pass to subprocesses for agent detection.
 * This allows child processes to skip re-detection by using the cached result.
 *
 * @example
 * ```typescript
 * const proc = Bun.spawn(['bun', 'run', 'dev'], {
 *   env: { ...process.env, ...getAgentEnv() },
 * });
 * ```
 */
export declare function getAgentEnv(): Record<string, string>;
//# sourceMappingURL=agent-detection.d.ts.map