/**
 * Auth React integration for @agentuity/auth.
 *
 * All React-specific code for auth.
 * Import from '@agentuity/auth/react' for React components and hooks.
 *
 * @module agentuity/react
 */
import React from 'react';
import { createAuthClient as createBetterAuthClient } from 'better-auth/react';
import type { BetterAuthClientPlugin } from 'better-auth/client';
import type { AuthSession, AuthUser } from './types';
/**
 * Options for creating the auth client.
 *
 * @typeParam TPlugins - Array of BetterAuth client plugins for type inference
 */
export interface AuthClientOptions<TPlugins extends BetterAuthClientPlugin[] = BetterAuthClientPlugin[]> {
    /**
     * Base URL for auth API requests.
     * Defaults to `window.location.origin` in browser environments.
     */
    baseURL?: string;
    /**
     * Base path for auth endpoints.
     * Defaults to '/api/auth' (Agentuity convention).
     */
    basePath?: string;
    /**
     * Skip default plugins (organizationClient, apiKeyClient).
     * Use this if you want full control over plugins.
     */
    skipDefaultPlugins?: boolean;
    /**
     * Additional plugins to include.
     * These are added after the default plugins (unless skipDefaultPlugins is true).
     *
     * Plugin types are inferred for full type safety.
     */
    plugins?: TPlugins;
}
/**
 * Get the default client plugins for auth.
 *
 * These mirror the server-side plugins:
 * - organizationClient: Multi-tenancy support
 * - apiKeyClient: Programmatic API key management
 *
 * Note: jwt() and bearer() are server-only plugins.
 */
export declare function getDefaultClientPlugins(): ({
    id: "organization";
    $InferServerPlugin: import("better-auth/plugins").OrganizationPlugin<{
        ac: import("better-auth/plugins").AccessControl<{
            readonly organization: readonly ["update", "delete"];
            readonly member: readonly ["create", "update", "delete"];
            readonly invitation: readonly ["create", "cancel"];
            readonly team: readonly ["create", "update", "delete"];
            readonly ac: readonly ["create", "read", "update", "delete"];
        }>;
        roles: {
            admin: import("better-auth/plugins").Role;
            member: import("better-auth/plugins").Role;
            owner: import("better-auth/plugins").Role;
        };
        teams: {
            enabled: false;
        };
        schema: {
            organization?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
            member?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
            invitation?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
            team?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
            organizationRole?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
        } | undefined;
        dynamicAccessControl: {
            enabled: false;
        };
    }>;
    getActions: ($fetch: import("better-auth/react").BetterFetch, _$store: import("better-auth").ClientStore, co: import("better-auth").BetterAuthClientOptions | undefined) => {
        $Infer: {
            ActiveOrganization: {
                members: {
                    id: string;
                    organizationId: string;
                    role: "member" | "admin" | "owner";
                    createdAt: Date;
                    userId: string;
                    user: {
                        id: string;
                        email: string;
                        name: string;
                        image?: string | undefined;
                    };
                }[];
                invitations: {
                    id: string;
                    organizationId: string;
                    email: string;
                    role: "member" | "admin" | "owner";
                    status: import("better-auth/plugins").InvitationStatus;
                    inviterId: string;
                    expiresAt: Date;
                    createdAt: Date;
                }[];
            } & {
                id: string;
                name: string;
                slug: string;
                createdAt: Date;
                logo?: string | null | undefined | undefined;
                metadata?: any;
            };
            Organization: {
                id: string;
                name: string;
                slug: string;
                createdAt: Date;
                logo?: string | null | undefined;
                metadata?: any;
            };
            Invitation: {
                id: string;
                organizationId: string;
                email: string;
                role: "member" | "admin" | "owner";
                status: import("better-auth/plugins").InvitationStatus;
                inviterId: string;
                expiresAt: Date;
                createdAt: Date;
            };
            Member: {
                id: string;
                organizationId: string;
                role: "member" | "admin" | "owner";
                createdAt: Date;
                userId: string;
                user: {
                    id: string;
                    email: string;
                    name: string;
                    image?: string | undefined;
                };
            };
            Team: {
                id: string;
                name: string;
                organizationId: string;
                createdAt: Date;
                updatedAt?: Date | undefined;
            };
        };
        organization: {
            checkRolePermission: <R extends "member" | "admin" | "owner">(data: ({
                permission: {
                    readonly organization?: ("update" | "delete")[] | undefined;
                    readonly member?: ("create" | "update" | "delete")[] | undefined;
                    readonly invitation?: ("create" | "cancel")[] | undefined;
                    readonly team?: ("create" | "update" | "delete")[] | undefined;
                    readonly ac?: ("create" | "update" | "delete" | "read")[] | undefined;
                };
                permissions?: never | undefined;
            } | {
                permissions: {
                    readonly organization?: ("update" | "delete")[] | undefined;
                    readonly member?: ("create" | "update" | "delete")[] | undefined;
                    readonly invitation?: ("create" | "cancel")[] | undefined;
                    readonly team?: ("create" | "update" | "delete")[] | undefined;
                    readonly ac?: ("create" | "update" | "delete" | "read")[] | undefined;
                };
                permission?: never | undefined;
            }) & {
                role: R;
            }) => boolean;
        };
    };
    getAtoms: ($fetch: import("better-auth/react").BetterFetch) => {
        $listOrg: import("better-auth/react").PreinitializedWritableAtom<boolean> & object;
        $activeOrgSignal: import("better-auth/react").PreinitializedWritableAtom<boolean> & object;
        $activeMemberSignal: import("better-auth/react").PreinitializedWritableAtom<boolean> & object;
        $activeMemberRoleSignal: import("better-auth/react").PreinitializedWritableAtom<boolean> & object;
        activeOrganization: import("better-auth/client").AuthQueryAtom<import("better-auth").Prettify<{
            id: string;
            name: string;
            slug: string;
            createdAt: Date;
            logo?: string | null | undefined | undefined;
            metadata?: any;
        } & {
            members: {
                id: string;
                organizationId: string;
                role: "member" | "admin" | "owner";
                createdAt: Date;
                userId: string;
                user: {
                    id: string;
                    email: string;
                    name: string;
                    image?: string | undefined;
                };
            }[];
            invitations: {
                id: string;
                organizationId: string;
                email: string;
                role: "member" | "admin" | "owner";
                status: import("better-auth/plugins").InvitationStatus;
                inviterId: string;
                expiresAt: Date;
                createdAt: Date;
            }[];
        }>>;
        listOrganizations: import("better-auth/client").AuthQueryAtom<{
            id: string;
            name: string;
            slug: string;
            createdAt: Date;
            logo?: string | null | undefined | undefined;
            metadata?: any;
        }[]>;
        activeMember: import("better-auth/client").AuthQueryAtom<{
            id: string;
            organizationId: string;
            userId: string;
            role: string;
            createdAt: Date;
        }>;
        activeMemberRole: import("better-auth/client").AuthQueryAtom<{
            role: string;
        }>;
    };
    pathMethods: {
        "/organization/get-full-organization": "GET";
        "/organization/list-user-teams": "GET";
    };
    atomListeners: ({
        matcher(path: string): path is "/organization/create" | "/organization/update" | "/organization/delete";
        signal: "$listOrg";
    } | {
        matcher(path: string): boolean;
        signal: "$activeOrgSignal";
    } | {
        matcher(path: string): boolean;
        signal: "$sessionSignal";
    } | {
        matcher(path: string): boolean;
        signal: "$activeMemberSignal";
    } | {
        matcher(path: string): boolean;
        signal: "$activeMemberRoleSignal";
    })[];
} | {
    id: "api-key";
    $InferServerPlugin: ReturnType<(options?: import("better-auth/plugins").ApiKeyOptions | undefined) => {
        id: "api-key";
        $ERROR_CODES: {
            readonly INVALID_METADATA_TYPE: "metadata must be an object or undefined";
            readonly REFILL_AMOUNT_AND_INTERVAL_REQUIRED: "refillAmount is required when refillInterval is provided";
            readonly REFILL_INTERVAL_AND_AMOUNT_REQUIRED: "refillInterval is required when refillAmount is provided";
            readonly USER_BANNED: "User is banned";
            readonly UNAUTHORIZED_SESSION: "Unauthorized or invalid session";
            readonly KEY_NOT_FOUND: "API Key not found";
            readonly KEY_DISABLED: "API Key is disabled";
            readonly KEY_EXPIRED: "API Key has expired";
            readonly USAGE_EXCEEDED: "API Key has reached its usage limit";
            readonly KEY_NOT_RECOVERABLE: "API Key is not recoverable";
            readonly EXPIRES_IN_IS_TOO_SMALL: "The expiresIn is smaller than the predefined minimum value.";
            readonly EXPIRES_IN_IS_TOO_LARGE: "The expiresIn is larger than the predefined maximum value.";
            readonly INVALID_REMAINING: "The remaining count is either too large or too small.";
            readonly INVALID_PREFIX_LENGTH: "The prefix length is either too large or too small.";
            readonly INVALID_NAME_LENGTH: "The name length is either too large or too small.";
            readonly METADATA_DISABLED: "Metadata is disabled.";
            readonly RATE_LIMIT_EXCEEDED: "Rate limit exceeded.";
            readonly NO_VALUES_TO_UPDATE: "No values to update.";
            readonly KEY_DISABLED_EXPIRATION: "Custom key expiration values are disabled.";
            readonly INVALID_API_KEY: "Invalid API key.";
            readonly INVALID_USER_ID_FROM_API_KEY: "The user id from the API key is invalid.";
            readonly INVALID_API_KEY_GETTER_RETURN_TYPE: "API Key getter returned an invalid key type. Expected string.";
            readonly SERVER_ONLY_PROPERTY: "The property you're trying to set can only be set from the server auth instance only.";
            readonly FAILED_TO_UPDATE_API_KEY: "Failed to update API key";
            readonly NAME_REQUIRED: "API Key name is required.";
        };
        hooks: {
            before: {
                matcher: (ctx: import("better-auth").HookEndpointContext) => boolean;
                handler: (inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                    user: {
                        id: string;
                        createdAt: Date;
                        updatedAt: Date;
                        email: string;
                        emailVerified: boolean;
                        name: string;
                        image?: string | null | undefined;
                    };
                    session: {
                        id: string;
                        token: string;
                        userId: string;
                        userAgent: string | null;
                        ipAddress: string | null;
                        createdAt: Date;
                        updatedAt: Date;
                        expiresAt: Date;
                    };
                } | {
                    context: import("better-auth").MiddlewareContext<import("better-auth").MiddlewareOptions, {
                        returned?: unknown | undefined;
                        responseHeaders?: Headers | undefined;
                    } & import("better-auth").PluginContext & import("better-auth").InfoContext & {
                        options: import("better-auth").BetterAuthOptions;
                        appName: string;
                        baseURL: string;
                        trustedOrigins: string[];
                        isTrustedOrigin: (url: string, settings?: {
                            allowRelativePaths: boolean;
                        }) => boolean;
                        oauthConfig: {
                            skipStateCookieCheck?: boolean | undefined;
                            storeStateStrategy: "database" | "cookie";
                        };
                        newSession: {
                            session: import("better-auth").Session & Record<string, any>;
                            user: import("better-auth").User & Record<string, any>;
                        } | null;
                        session: {
                            session: import("better-auth").Session & Record<string, any>;
                            user: import("better-auth").User & Record<string, any>;
                        } | null;
                        setNewSession: (session: {
                            session: import("better-auth").Session & Record<string, any>;
                            user: import("better-auth").User & Record<string, any>;
                        } | null) => void;
                        socialProviders: import("better-auth").OAuthProvider[];
                        authCookies: import("better-auth").BetterAuthCookies;
                        logger: ReturnType<(options?: import("better-auth").Logger | undefined) => import("better-auth").InternalLogger>;
                        rateLimit: {
                            enabled: boolean;
                            window: number;
                            max: number;
                            storage: "memory" | "database" | "secondary-storage";
                        } & Omit<import("better-auth").BetterAuthRateLimitOptions, "enabled" | "window" | "max" | "storage">;
                        adapter: import("better-auth").DBAdapter<import("better-auth").BetterAuthOptions>;
                        internalAdapter: import("better-auth").InternalAdapter<import("better-auth").BetterAuthOptions>;
                        createAuthCookie: (cookieName: string, overrideAttributes?: Partial<import("better-auth").CookieOptions> | undefined) => import("better-auth").BetterAuthCookie;
                        secret: string;
                        sessionConfig: {
                            updateAge: number;
                            expiresIn: number;
                            freshAge: number;
                            cookieRefreshCache: false | {
                                enabled: true;
                                updateAge: number;
                            };
                        };
                        generateId: (options: {
                            model: import("better-auth").ModelNames;
                            size?: number | undefined;
                        }) => string | false;
                        secondaryStorage: import("better-auth").SecondaryStorage | undefined;
                        password: {
                            hash: (password: string) => Promise<string>;
                            verify: (data: {
                                password: string;
                                hash: string;
                            }) => Promise<boolean>;
                            config: {
                                minPasswordLength: number;
                                maxPasswordLength: number;
                            };
                            checkPassword: (userId: string, ctx: import("better-auth").GenericEndpointContext<import("better-auth").BetterAuthOptions>) => Promise<boolean>;
                        };
                        tables: import("better-auth").BetterAuthDBSchema;
                        runMigrations: () => Promise<void>;
                        publishTelemetry: (event: {
                            type: string;
                            anonymousId?: string | undefined;
                            payload: Record<string, any>;
                        }) => Promise<void>;
                        skipOriginCheck: boolean | string[];
                        skipCSRFCheck: boolean;
                        runInBackground: (promise: Promise<unknown>) => void;
                        runInBackgroundOrAwait: (promise: Promise<unknown> | void) => import("better-auth").Awaitable<unknown>;
                    }>;
                }>;
            }[];
        };
        endpoints: {
            createApiKey: import("better-auth").StrictEndpoint<"/api-key/create", {
                method: "POST";
                body: import("zod").ZodObject<{
                    name: import("zod").ZodOptional<import("zod").ZodString>;
                    expiresIn: import("zod").ZodDefault<import("zod").ZodNullable<import("zod").ZodOptional<import("zod").ZodNumber>>>;
                    userId: import("zod").ZodOptional<import("zod").ZodCoercedString<unknown>>;
                    prefix: import("zod").ZodOptional<import("zod").ZodString>;
                    remaining: import("zod").ZodDefault<import("zod").ZodNullable<import("zod").ZodOptional<import("zod").ZodNumber>>>;
                    metadata: import("zod").ZodOptional<import("zod").ZodAny>;
                    refillAmount: import("zod").ZodOptional<import("zod").ZodNumber>;
                    refillInterval: import("zod").ZodOptional<import("zod").ZodNumber>;
                    rateLimitTimeWindow: import("zod").ZodOptional<import("zod").ZodNumber>;
                    rateLimitMax: import("zod").ZodOptional<import("zod").ZodNumber>;
                    rateLimitEnabled: import("zod").ZodOptional<import("zod").ZodBoolean>;
                    permissions: import("zod").ZodOptional<import("zod").ZodRecord<import("zod").ZodString, import("zod").ZodArray<import("zod").ZodString>>>;
                }, import("better-auth").$strip>;
                metadata: {
                    openapi: {
                        description: string;
                        responses: {
                            "200": {
                                description: string;
                                content: {
                                    "application/json": {
                                        schema: {
                                            type: "object";
                                            properties: {
                                                id: {
                                                    type: string;
                                                    description: string;
                                                };
                                                createdAt: {
                                                    type: string;
                                                    format: string;
                                                    description: string;
                                                };
                                                updatedAt: {
                                                    type: string;
                                                    format: string;
                                                    description: string;
                                                };
                                                name: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                prefix: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                start: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                key: {
                                                    type: string;
                                                    description: string;
                                                };
                                                enabled: {
                                                    type: string;
                                                    description: string;
                                                };
                                                expiresAt: {
                                                    type: string;
                                                    format: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                userId: {
                                                    type: string;
                                                    description: string;
                                                };
                                                lastRefillAt: {
                                                    type: string;
                                                    format: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                lastRequest: {
                                                    type: string;
                                                    format: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                metadata: {
                                                    type: string;
                                                    nullable: boolean;
                                                    additionalProperties: boolean;
                                                    description: string;
                                                };
                                                rateLimitMax: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                rateLimitTimeWindow: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                remaining: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                refillAmount: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                refillInterval: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                rateLimitEnabled: {
                                                    type: string;
                                                    description: string;
                                                };
                                                requestCount: {
                                                    type: string;
                                                    description: string;
                                                };
                                                permissions: {
                                                    type: string;
                                                    nullable: boolean;
                                                    additionalProperties: {
                                                        type: string;
                                                        items: {
                                                            type: string;
                                                        };
                                                    };
                                                    description: string;
                                                };
                                            };
                                            required: string[];
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            }, {
                key: string;
                metadata: any;
                permissions: any;
                id: string;
                name: string | null;
                start: string | null;
                prefix: string | null;
                userId: string;
                refillInterval: number | null;
                refillAmount: number | null;
                lastRefillAt: Date | null;
                enabled: boolean;
                rateLimitEnabled: boolean;
                rateLimitTimeWindow: number | null;
                rateLimitMax: number | null;
                requestCount: number;
                remaining: number | null;
                lastRequest: Date | null;
                expiresAt: Date | null;
                createdAt: Date;
                updatedAt: Date;
            }>;
            verifyApiKey: import("better-auth").StrictEndpoint<string, {
                method: "POST";
                body: import("zod").ZodObject<{
                    key: import("zod").ZodString;
                    permissions: import("zod").ZodOptional<import("zod").ZodRecord<import("zod").ZodString, import("zod").ZodArray<import("zod").ZodString>>>;
                }, import("better-auth").$strip>;
            }, {
                valid: boolean;
                error: {
                    message: "Invalid API key.";
                    code: "KEY_NOT_FOUND";
                };
                key: null;
            } | {
                valid: boolean;
                error: {
                    message: string | undefined;
                    code: string;
                    cause?: unknown;
                };
                key: null;
            } | {
                valid: boolean;
                error: {
                    message: "Invalid API key.";
                    code: "INVALID_API_KEY";
                };
                key: null;
            } | {
                valid: boolean;
                error: null;
                key: Omit<import("better-auth/plugins").ApiKey, "key"> | null;
            }>;
            getApiKey: import("better-auth").StrictEndpoint<"/api-key/get", {
                method: "GET";
                query: import("zod").ZodObject<{
                    id: import("zod").ZodString;
                }, import("better-auth").$strip>;
                use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                    session: {
                        session: Record<string, any> & {
                            id: string;
                            createdAt: Date;
                            updatedAt: Date;
                            userId: string;
                            expiresAt: Date;
                            token: string;
                            ipAddress?: string | null | undefined;
                            userAgent?: string | null | undefined;
                        };
                        user: Record<string, any> & {
                            id: string;
                            createdAt: Date;
                            updatedAt: Date;
                            email: string;
                            emailVerified: boolean;
                            name: string;
                            image?: string | null | undefined;
                        };
                    };
                }>)[];
                metadata: {
                    openapi: {
                        description: string;
                        responses: {
                            "200": {
                                description: string;
                                content: {
                                    "application/json": {
                                        schema: {
                                            type: "object";
                                            properties: {
                                                id: {
                                                    type: string;
                                                    description: string;
                                                };
                                                name: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                start: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                prefix: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                userId: {
                                                    type: string;
                                                    description: string;
                                                };
                                                refillInterval: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                refillAmount: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                lastRefillAt: {
                                                    type: string;
                                                    format: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                enabled: {
                                                    type: string;
                                                    description: string;
                                                    default: boolean;
                                                };
                                                rateLimitEnabled: {
                                                    type: string;
                                                    description: string;
                                                };
                                                rateLimitTimeWindow: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                rateLimitMax: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                requestCount: {
                                                    type: string;
                                                    description: string;
                                                };
                                                remaining: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                lastRequest: {
                                                    type: string;
                                                    format: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                expiresAt: {
                                                    type: string;
                                                    format: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                createdAt: {
                                                    type: string;
                                                    format: string;
                                                    description: string;
                                                };
                                                updatedAt: {
                                                    type: string;
                                                    format: string;
                                                    description: string;
                                                };
                                                metadata: {
                                                    type: string;
                                                    nullable: boolean;
                                                    additionalProperties: boolean;
                                                    description: string;
                                                };
                                                permissions: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                            };
                                            required: string[];
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            }, {
                metadata: Record<string, any> | null;
                permissions: {
                    [key: string]: string[];
                } | null;
                id: string;
                name: string | null;
                start: string | null;
                prefix: string | null;
                userId: string;
                refillInterval: number | null;
                refillAmount: number | null;
                lastRefillAt: Date | null;
                enabled: boolean;
                rateLimitEnabled: boolean;
                rateLimitTimeWindow: number | null;
                rateLimitMax: number | null;
                requestCount: number;
                remaining: number | null;
                lastRequest: Date | null;
                expiresAt: Date | null;
                createdAt: Date;
                updatedAt: Date;
            }>;
            updateApiKey: import("better-auth").StrictEndpoint<"/api-key/update", {
                method: "POST";
                body: import("zod").ZodObject<{
                    keyId: import("zod").ZodString;
                    userId: import("zod").ZodOptional<import("zod").ZodCoercedString<unknown>>;
                    name: import("zod").ZodOptional<import("zod").ZodString>;
                    enabled: import("zod").ZodOptional<import("zod").ZodBoolean>;
                    remaining: import("zod").ZodOptional<import("zod").ZodNumber>;
                    refillAmount: import("zod").ZodOptional<import("zod").ZodNumber>;
                    refillInterval: import("zod").ZodOptional<import("zod").ZodNumber>;
                    metadata: import("zod").ZodOptional<import("zod").ZodAny>;
                    expiresIn: import("zod").ZodNullable<import("zod").ZodOptional<import("zod").ZodNumber>>;
                    rateLimitEnabled: import("zod").ZodOptional<import("zod").ZodBoolean>;
                    rateLimitTimeWindow: import("zod").ZodOptional<import("zod").ZodNumber>;
                    rateLimitMax: import("zod").ZodOptional<import("zod").ZodNumber>;
                    permissions: import("zod").ZodNullable<import("zod").ZodOptional<import("zod").ZodRecord<import("zod").ZodString, import("zod").ZodArray<import("zod").ZodString>>>>;
                }, import("better-auth").$strip>;
                metadata: {
                    openapi: {
                        description: string;
                        responses: {
                            "200": {
                                description: string;
                                content: {
                                    "application/json": {
                                        schema: {
                                            type: "object";
                                            properties: {
                                                id: {
                                                    type: string;
                                                    description: string;
                                                };
                                                name: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                start: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                prefix: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                userId: {
                                                    type: string;
                                                    description: string;
                                                };
                                                refillInterval: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                refillAmount: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                lastRefillAt: {
                                                    type: string;
                                                    format: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                enabled: {
                                                    type: string;
                                                    description: string;
                                                    default: boolean;
                                                };
                                                rateLimitEnabled: {
                                                    type: string;
                                                    description: string;
                                                };
                                                rateLimitTimeWindow: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                rateLimitMax: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                requestCount: {
                                                    type: string;
                                                    description: string;
                                                };
                                                remaining: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                lastRequest: {
                                                    type: string;
                                                    format: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                expiresAt: {
                                                    type: string;
                                                    format: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                                createdAt: {
                                                    type: string;
                                                    format: string;
                                                    description: string;
                                                };
                                                updatedAt: {
                                                    type: string;
                                                    format: string;
                                                    description: string;
                                                };
                                                metadata: {
                                                    type: string;
                                                    nullable: boolean;
                                                    additionalProperties: boolean;
                                                    description: string;
                                                };
                                                permissions: {
                                                    type: string;
                                                    nullable: boolean;
                                                    description: string;
                                                };
                                            };
                                            required: string[];
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            }, {
                metadata: Record<string, any> | null;
                permissions: {
                    [key: string]: string[];
                } | null;
                id: string;
                name: string | null;
                start: string | null;
                prefix: string | null;
                userId: string;
                refillInterval: number | null;
                refillAmount: number | null;
                lastRefillAt: Date | null;
                enabled: boolean;
                rateLimitEnabled: boolean;
                rateLimitTimeWindow: number | null;
                rateLimitMax: number | null;
                requestCount: number;
                remaining: number | null;
                lastRequest: Date | null;
                expiresAt: Date | null;
                createdAt: Date;
                updatedAt: Date;
            }>;
            deleteApiKey: import("better-auth").StrictEndpoint<"/api-key/delete", {
                method: "POST";
                body: import("zod").ZodObject<{
                    keyId: import("zod").ZodString;
                }, import("better-auth").$strip>;
                use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                    session: {
                        session: Record<string, any> & {
                            id: string;
                            createdAt: Date;
                            updatedAt: Date;
                            userId: string;
                            expiresAt: Date;
                            token: string;
                            ipAddress?: string | null | undefined;
                            userAgent?: string | null | undefined;
                        };
                        user: Record<string, any> & {
                            id: string;
                            createdAt: Date;
                            updatedAt: Date;
                            email: string;
                            emailVerified: boolean;
                            name: string;
                            image?: string | null | undefined;
                        };
                    };
                }>)[];
                metadata: {
                    openapi: {
                        description: string;
                        requestBody: {
                            content: {
                                "application/json": {
                                    schema: {
                                        type: "object";
                                        properties: {
                                            keyId: {
                                                type: string;
                                                description: string;
                                            };
                                        };
                                        required: string[];
                                    };
                                };
                            };
                        };
                        responses: {
                            "200": {
                                description: string;
                                content: {
                                    "application/json": {
                                        schema: {
                                            type: "object";
                                            properties: {
                                                success: {
                                                    type: string;
                                                    description: string;
                                                };
                                            };
                                            required: string[];
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            }, {
                success: boolean;
            }>;
            listApiKeys: import("better-auth").StrictEndpoint<"/api-key/list", {
                method: "GET";
                use: ((inputContext: import("better-auth").MiddlewareInputContext<import("better-auth").MiddlewareOptions>) => Promise<{
                    session: {
                        session: Record<string, any> & {
                            id: string;
                            createdAt: Date;
                            updatedAt: Date;
                            userId: string;
                            expiresAt: Date;
                            token: string;
                            ipAddress?: string | null | undefined;
                            userAgent?: string | null | undefined;
                        };
                        user: Record<string, any> & {
                            id: string;
                            createdAt: Date;
                            updatedAt: Date;
                            email: string;
                            emailVerified: boolean;
                            name: string;
                            image?: string | null | undefined;
                        };
                    };
                }>)[];
                metadata: {
                    openapi: {
                        description: string;
                        responses: {
                            "200": {
                                description: string;
                                content: {
                                    "application/json": {
                                        schema: {
                                            type: "array";
                                            items: {
                                                type: string;
                                                properties: {
                                                    id: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    name: {
                                                        type: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    start: {
                                                        type: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    prefix: {
                                                        type: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    userId: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    refillInterval: {
                                                        type: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    refillAmount: {
                                                        type: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    lastRefillAt: {
                                                        type: string;
                                                        format: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    enabled: {
                                                        type: string;
                                                        description: string;
                                                        default: boolean;
                                                    };
                                                    rateLimitEnabled: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    rateLimitTimeWindow: {
                                                        type: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    rateLimitMax: {
                                                        type: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    requestCount: {
                                                        type: string;
                                                        description: string;
                                                    };
                                                    remaining: {
                                                        type: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    lastRequest: {
                                                        type: string;
                                                        format: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    expiresAt: {
                                                        type: string;
                                                        format: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                    createdAt: {
                                                        type: string;
                                                        format: string;
                                                        description: string;
                                                    };
                                                    updatedAt: {
                                                        type: string;
                                                        format: string;
                                                        description: string;
                                                    };
                                                    metadata: {
                                                        type: string;
                                                        nullable: boolean;
                                                        additionalProperties: boolean;
                                                        description: string;
                                                    };
                                                    permissions: {
                                                        type: string;
                                                        nullable: boolean;
                                                        description: string;
                                                    };
                                                };
                                                required: string[];
                                            };
                                        };
                                    };
                                };
                            };
                        };
                    };
                };
            }, {
                metadata: Record<string, any> | null;
                permissions: {
                    [key: string]: string[];
                } | null;
                id: string;
                name: string | null;
                start: string | null;
                prefix: string | null;
                userId: string;
                refillInterval: number | null;
                refillAmount: number | null;
                lastRefillAt: Date | null;
                enabled: boolean;
                rateLimitEnabled: boolean;
                rateLimitTimeWindow: number | null;
                rateLimitMax: number | null;
                requestCount: number;
                remaining: number | null;
                lastRequest: Date | null;
                expiresAt: Date | null;
                createdAt: Date;
                updatedAt: Date;
            }[]>;
            deleteAllExpiredApiKeys: import("better-auth").StrictEndpoint<string, {
                method: "POST";
            }, {
                success: boolean;
                error: unknown;
            }>;
        };
        schema: {
            apikey: {
                fields: {
                    name: {
                        type: "string";
                        required: false;
                        input: false;
                    };
                    start: {
                        type: "string";
                        required: false;
                        input: false;
                    };
                    prefix: {
                        type: "string";
                        required: false;
                        input: false;
                    };
                    key: {
                        type: "string";
                        required: true;
                        input: false;
                        index: true;
                    };
                    userId: {
                        type: "string";
                        references: {
                            model: string;
                            field: string;
                            onDelete: "cascade";
                        };
                        required: true;
                        input: false;
                        index: true;
                    };
                    refillInterval: {
                        type: "number";
                        required: false;
                        input: false;
                    };
                    refillAmount: {
                        type: "number";
                        required: false;
                        input: false;
                    };
                    lastRefillAt: {
                        type: "date";
                        required: false;
                        input: false;
                    };
                    enabled: {
                        type: "boolean";
                        required: false;
                        input: false;
                        defaultValue: true;
                    };
                    rateLimitEnabled: {
                        type: "boolean";
                        required: false;
                        input: false;
                        defaultValue: true;
                    };
                    rateLimitTimeWindow: {
                        type: "number";
                        required: false;
                        input: false;
                        defaultValue: number;
                    };
                    rateLimitMax: {
                        type: "number";
                        required: false;
                        input: false;
                        defaultValue: number;
                    };
                    requestCount: {
                        type: "number";
                        required: false;
                        input: false;
                        defaultValue: number;
                    };
                    remaining: {
                        type: "number";
                        required: false;
                        input: false;
                    };
                    lastRequest: {
                        type: "date";
                        required: false;
                        input: false;
                    };
                    expiresAt: {
                        type: "date";
                        required: false;
                        input: false;
                    };
                    createdAt: {
                        type: "date";
                        required: true;
                        input: false;
                    };
                    updatedAt: {
                        type: "date";
                        required: true;
                        input: false;
                    };
                    permissions: {
                        type: "string";
                        required: false;
                        input: false;
                    };
                    metadata: {
                        type: "string";
                        required: false;
                        input: true;
                        transform: {
                            input(value: import("better-auth").DBPrimitive): string;
                            output(value: import("better-auth").DBPrimitive): any;
                        };
                    };
                };
            };
        };
        options: import("better-auth/plugins").ApiKeyOptions | undefined;
    }>;
    pathMethods: {
        "/api-key/create": "POST";
        "/api-key/delete": "POST";
        "/api-key/delete-all-expired-api-keys": "POST";
    };
})[];
/**
 * Create a pre-configured Auth client.
 *
 * This factory provides sensible defaults for Agentuity projects:
 * - Uses `/api/auth` as the default base path
 * - Automatically uses `window.location.origin` as base URL in browsers
 * - Includes organization and API key plugins by default
 *
 * @example Basic usage (zero config)
 * ```typescript
 * import { createAuthClient } from '@agentuity/auth/react';
 *
 * export const authClient = createAuthClient();
 * export const { signIn, signUp, signOut, useSession, getSession } = authClient;
 * ```
 *
 * @example With custom base path
 * ```typescript
 * export const authClient = createAuthClient({
 *   basePath: '/auth', // If mounted at /auth instead of /api/auth
 * });
 * ```
 *
 * @example With additional plugins
 * ```typescript
 * import { twoFactorClient } from 'better-auth/client/plugins';
 *
 * export const authClient = createAuthClient({
 *   plugins: [twoFactorClient()],
 * });
 * ```
 *
 * @example With custom plugins only (no defaults)
 * ```typescript
 * import { organizationClient } from 'better-auth/client/plugins';
 *
 * export const authClient = createAuthClient({
 *   skipDefaultPlugins: true,
 *   plugins: [organizationClient()],
 * });
 * ```
 */
export declare function createAuthClient<TPlugins extends BetterAuthClientPlugin[] = []>(options?: AuthClientOptions<TPlugins>): ReturnType<typeof createBetterAuthClient<{
    plugins: TPlugins;
}>>;
/**
 * Type helper for the auth client return type.
 */
export type AuthClient = ReturnType<typeof createAuthClient>;
/**
 * Context value for Auth.
 */
export interface AuthContextValue {
    /** The auth client instance */
    authClient: AuthClient;
    /** Current authenticated user, or null if not signed in */
    user: AuthUser | null;
    /** Current session object (if available) */
    session: AuthSession | null;
    /** Whether the auth state is still loading */
    isPending: boolean;
    /** Any error that occurred while fetching auth state */
    error: Error | null;
    /** Whether the user is authenticated */
    isAuthenticated: boolean;
}
export interface AuthProviderProps {
    /** React children to render */
    children: React.ReactNode;
    /**
     * The auth client instance created with createAuthClient().
     * Required for session management.
     */
    authClient: AuthClient;
    /**
     * Token refresh interval in milliseconds.
     *
     * **Default:** `3600000` (1 hour)
     *
     * Controls how frequently the auth state is refreshed by polling the session endpoint.
     * A longer interval reduces server load and API calls, but means auth state changes
     * (like session expiration or revocation) may not be detected for up to the interval duration.
     *
     * **Security Implications:**
     * - Longer intervals mean staler auth state: revoked sessions or permission changes
     *   may not be detected until the next refresh cycle (up to the interval duration)
     * - Shorter intervals provide fresher state but increase server load and API calls
     * - Consider your security requirements when choosing an interval
     *
     * **Recommended Intervals:**
     * - `30000` - `60000` (30s - 1m): High-security applications requiring near-real-time
     *   detection of session revocation or permission changes
     * - `300000` - `900000` (5m - 15m): Sensitive features (admin panels, financial operations)
     *   where timely detection of auth changes is important
     * - `3600000` (1h): Typical applications where occasional staleness is acceptable
     *
     * **Override Example:**
     * ```tsx
     * // High-security: refresh every 30 seconds
     * <AuthProvider authClient={authClient} refreshInterval={30000}>
     *
     * // Sensitive features: refresh every 5 minutes
     * <AuthProvider authClient={authClient} refreshInterval={300000}>
     *
     * // Default: refresh every hour
     * <AuthProvider authClient={authClient}>
     * ```
     */
    refreshInterval?: number;
    /**
     * Override the token endpoint path.
     * Defaults to '/token' (relative to the auth client's basePath).
     * Set to `false` to disable token fetching entirely.
     *
     * @example Custom token endpoint
     * ```tsx
     * <AuthProvider authClient={authClient} tokenEndpoint="/api/custom/jwt">
     * ```
     *
     * @example Disable token fetching
     * ```tsx
     * <AuthProvider authClient={authClient} tokenEndpoint={false}>
     * ```
     */
    tokenEndpoint?: string | false;
}
/**
 * Auth provider component.
 *
 * This component integrates Auth with Agentuity's React context,
 * automatically injecting auth tokens into API calls via useAgent and useWebsocket.
 *
 * Must be a child of AgentuityProvider.
 *
 * @example
 * ```tsx
 * import { AgentuityProvider } from '@agentuity/react';
 * import { createAuthClient, AuthProvider } from '@agentuity/auth/react';
 *
 * const authClient = createAuthClient();
 *
 * <AgentuityProvider>
 *   <AuthProvider authClient={authClient}>
 *     <App />
 *   </AuthProvider>
 * </AgentuityProvider>
 * ```
 */
export declare function AuthProvider({ children, authClient, refreshInterval, tokenEndpoint, }: AuthProviderProps): import("react/jsx-runtime").JSX.Element;
/**
 * Hook to access Auth state.
 *
 * This hook provides access to the current user and session.
 * Must be used within an AuthProvider.
 *
 * @example
 * ```tsx
 * import { useAuth } from '@agentuity/auth/react';
 *
 * function Profile() {
 *   const { user, session, isPending, isAuthenticated } = useAuth();
 *
 *   if (isPending) return <div>Loading...</div>;
 *   if (!isAuthenticated) return <div>Not signed in</div>;
 *
 *   return <div>Welcome, {user.name}!</div>;
 * }
 * ```
 */
export declare function useAuth(): AuthContextValue;
//# sourceMappingURL=react.d.ts.map