import { z } from 'zod';
/**
 * Queue type schema for validation.
 *
 * - `worker`: Messages are consumed by workers with acknowledgment. Each message is processed by exactly one consumer.
 * - `pubsub`: Messages are broadcast to all subscribers. Multiple consumers can receive the same message.
 *
 * @example
 * ```typescript
 * const queueType = QueueTypeSchema.parse('worker'); // 'worker' | 'pubsub'
 * ```
 */
export declare const QueueTypeSchema: z.ZodEnum<{
    worker: "worker";
    pubsub: "pubsub";
}>;
/**
 * Queue type - either 'worker' for task queues or 'pubsub' for broadcast messaging.
 */
export type QueueType = z.infer<typeof QueueTypeSchema>;
export type QueueSortField = 'name' | 'created' | 'updated' | 'message_count' | 'dlq_count';
/**
 * Queue settings schema for configuring queue behavior.
 *
 * These settings control message lifecycle, retry behavior, and concurrency limits.
 * This schema includes defaults and is used for output types and documentation.
 *
 * @example
 * ```typescript
 * const settings = QueueSettingsSchema.parse({
 *   default_ttl_seconds: 3600,           // Messages expire after 1 hour
 *   default_visibility_timeout_seconds: 60, // Processing timeout
 *   default_max_retries: 3,              // Retry failed messages 3 times
 * });
 * ```
 */
export declare const QueueSettingsSchema: z.ZodObject<{
    default_ttl_seconds: z.ZodOptional<z.ZodNullable<z.ZodNumber>>;
    default_visibility_timeout_seconds: z.ZodDefault<z.ZodNumber>;
    default_max_retries: z.ZodDefault<z.ZodNumber>;
    default_retry_backoff_ms: z.ZodDefault<z.ZodNumber>;
    default_retry_max_backoff_ms: z.ZodDefault<z.ZodNumber>;
    default_retry_multiplier: z.ZodDefault<z.ZodNumber>;
    max_in_flight_per_client: z.ZodDefault<z.ZodNumber>;
    retention_seconds: z.ZodDefault<z.ZodNumber>;
}, z.core.$strip>;
/**
 * Queue settings configuration type.
 */
export type QueueSettings = z.infer<typeof QueueSettingsSchema>;
/**
 * Queue statistics schema showing current queue state.
 */
export declare const QueueStatsSchema: z.ZodObject<{
    message_count: z.ZodNumber;
    dlq_count: z.ZodNumber;
    next_offset: z.ZodNumber;
}, z.core.$strip>;
/**
 * Queue statistics type.
 */
export type QueueStats = z.infer<typeof QueueStatsSchema>;
/**
 * Queue schema representing a message queue.
 *
 * @example
 * ```typescript
 * const queue = await getQueue(client, 'my-queue');
 * console.log(`Queue ${queue.name} has ${queue.message_count} messages`);
 * ```
 */
export declare const QueueSchema: z.ZodObject<{
    id: z.ZodString;
    name: z.ZodString;
    description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    queue_type: z.ZodEnum<{
        worker: "worker";
        pubsub: "pubsub";
    }>;
    default_ttl_seconds: z.ZodOptional<z.ZodNullable<z.ZodNumber>>;
    default_visibility_timeout_seconds: z.ZodOptional<z.ZodNumber>;
    default_max_retries: z.ZodOptional<z.ZodNumber>;
    default_retry_backoff_ms: z.ZodOptional<z.ZodNumber>;
    default_retry_max_backoff_ms: z.ZodOptional<z.ZodNumber>;
    default_retry_multiplier: z.ZodOptional<z.ZodNumber>;
    max_in_flight_per_client: z.ZodOptional<z.ZodNumber>;
    next_offset: z.ZodOptional<z.ZodNumber>;
    message_count: z.ZodOptional<z.ZodNumber>;
    dlq_count: z.ZodOptional<z.ZodNumber>;
    created_at: z.ZodString;
    updated_at: z.ZodString;
    paused_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    retention_seconds: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
/**
 * Queue type representing a message queue instance.
 */
export type Queue = z.infer<typeof QueueSchema>;
/**
 * Message state schema for tracking message lifecycle.
 *
 * - `pending`: Message is waiting to be processed.
 * - `leased`: Message has been received and is currently being processed by a consumer.
 * - `processing`: Message has been received and is being processed (legacy, equivalent to leased).
 * - `delivered`: Message was successfully acknowledged.
 * - `failed`: Message processing failed but may be retried.
 * - `dead`: Message exceeded retry limit and was moved to DLQ.
 */
export declare const MessageStateSchema: z.ZodEnum<{
    pending: "pending";
    failed: "failed";
    leased: "leased";
    processing: "processing";
    delivered: "delivered";
    dead: "dead";
}>;
/**
 * Message state type.
 */
export type MessageState = z.infer<typeof MessageStateSchema>;
/**
 * Message schema representing a queue message.
 *
 * @example
 * ```typescript
 * const message = await publishMessage(client, 'my-queue', { payload: 'Hello' });
 * console.log(`Published message ${message.id} at offset ${message.offset}`);
 * ```
 */
export declare const MessageSchema: z.ZodObject<{
    id: z.ZodString;
    queue_id: z.ZodString;
    offset: z.ZodNumber;
    payload: z.ZodUnknown;
    size: z.ZodOptional<z.ZodNumber>;
    metadata: z.ZodOptional<z.ZodNullable<z.ZodRecord<z.ZodString, z.ZodUnknown>>>;
    state: z.ZodOptional<z.ZodEnum<{
        pending: "pending";
        failed: "failed";
        leased: "leased";
        processing: "processing";
        delivered: "delivered";
        dead: "dead";
    }>>;
    idempotency_key: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    partition_key: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    ttl_seconds: z.ZodOptional<z.ZodNullable<z.ZodNumber>>;
    delivery_attempts: z.ZodOptional<z.ZodNumber>;
    max_retries: z.ZodOptional<z.ZodNumber>;
    published_at: z.ZodOptional<z.ZodString>;
    expires_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    delivered_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    acknowledged_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    created_at: z.ZodOptional<z.ZodString>;
    updated_at: z.ZodOptional<z.ZodString>;
}, z.core.$strip>;
/**
 * Message type representing a queue message.
 */
export type Message = z.infer<typeof MessageSchema>;
/**
 * Destination type schema. Currently only HTTP webhooks are supported.
 */
export declare const DestinationTypeSchema: z.ZodEnum<{
    http: "http";
}>;
/**
 * Destination type.
 */
export type DestinationType = z.infer<typeof DestinationTypeSchema>;
/**
 * HTTP destination configuration schema for webhook delivery.
 *
 * @example
 * ```typescript
 * const config: HttpDestinationConfig = {
 *   url: 'https://api.example.com/webhook',
 *   method: 'POST',
 *   headers: { 'X-Custom-Header': 'value' },
 *   timeout_ms: 10000,
 *   retry_policy: { max_attempts: 3 },
 * };
 * ```
 */
export declare const HttpDestinationConfigSchema: z.ZodObject<{
    url: z.ZodString;
    headers: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
    method: z.ZodDefault<z.ZodString>;
    timeout_ms: z.ZodDefault<z.ZodNumber>;
    retry_policy: z.ZodOptional<z.ZodObject<{
        max_attempts: z.ZodDefault<z.ZodNumber>;
        initial_backoff_ms: z.ZodDefault<z.ZodNumber>;
        max_backoff_ms: z.ZodDefault<z.ZodNumber>;
        backoff_multiplier: z.ZodDefault<z.ZodNumber>;
    }, z.core.$strip>>;
    signing: z.ZodOptional<z.ZodObject<{
        enabled: z.ZodDefault<z.ZodBoolean>;
        secret_key: z.ZodOptional<z.ZodString>;
    }, z.core.$strip>>;
}, z.core.$strip>;
/**
 * HTTP destination configuration type.
 */
export type HttpDestinationConfig = z.infer<typeof HttpDestinationConfigSchema>;
/**
 * Destination statistics schema showing delivery metrics.
 */
export declare const DestinationStatsSchema: z.ZodObject<{
    total_deliveries: z.ZodNumber;
    successful_deliveries: z.ZodNumber;
    failed_deliveries: z.ZodNumber;
    last_delivery_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
}, z.core.$strip>;
/**
 * Destination statistics type.
 */
export type DestinationStats = z.infer<typeof DestinationStatsSchema>;
/**
 * Destination schema representing a webhook endpoint for message delivery.
 *
 * Destinations are attached to queues and automatically receive messages when published.
 *
 * @example
 * ```typescript
 * const destination = await createDestination(client, 'my-queue', {
 *   destination_type: 'http',
 *   config: { url: 'https://api.example.com/webhook' },
 *   enabled: true,
 * });
 * ```
 */
export declare const DestinationSchema: z.ZodObject<{
    id: z.ZodString;
    queue_id: z.ZodString;
    destination_type: z.ZodEnum<{
        http: "http";
    }>;
    config: z.ZodObject<{
        url: z.ZodString;
        headers: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
        method: z.ZodDefault<z.ZodString>;
        timeout_ms: z.ZodDefault<z.ZodNumber>;
        retry_policy: z.ZodOptional<z.ZodObject<{
            max_attempts: z.ZodDefault<z.ZodNumber>;
            initial_backoff_ms: z.ZodDefault<z.ZodNumber>;
            max_backoff_ms: z.ZodDefault<z.ZodNumber>;
            backoff_multiplier: z.ZodDefault<z.ZodNumber>;
        }, z.core.$strip>>;
        signing: z.ZodOptional<z.ZodObject<{
            enabled: z.ZodDefault<z.ZodBoolean>;
            secret_key: z.ZodOptional<z.ZodString>;
        }, z.core.$strip>>;
    }, z.core.$strip>;
    enabled: z.ZodBoolean;
    stats: z.ZodOptional<z.ZodObject<{
        total_deliveries: z.ZodNumber;
        successful_deliveries: z.ZodNumber;
        failed_deliveries: z.ZodNumber;
        last_delivery_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
        last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
        last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    }, z.core.$strip>>;
    created_at: z.ZodString;
    updated_at: z.ZodString;
}, z.core.$strip>;
/**
 * Destination type representing a webhook endpoint.
 */
export type Destination = z.infer<typeof DestinationSchema>;
/**
 * Dead letter message schema for messages that failed processing.
 *
 * Messages are moved to the dead letter queue (DLQ) after exceeding the maximum
 * retry limit. They can be inspected, replayed, or deleted.
 *
 * @example
 * ```typescript
 * const { messages } = await listDeadLetterMessages(client, 'my-queue');
 * for (const msg of messages) {
 *   console.log(`Failed message: ${msg.failure_reason}`);
 *   await replayDeadLetterMessage(client, 'my-queue', msg.id);
 * }
 * ```
 */
export declare const DeadLetterMessageSchema: z.ZodObject<{
    id: z.ZodString;
    queue_id: z.ZodString;
    original_message_id: z.ZodOptional<z.ZodString>;
    offset: z.ZodNumber;
    payload: z.ZodUnknown;
    metadata: z.ZodOptional<z.ZodNullable<z.ZodRecord<z.ZodString, z.ZodUnknown>>>;
    failure_reason: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    delivery_attempts: z.ZodNumber;
    moved_at: z.ZodOptional<z.ZodString>;
    original_published_at: z.ZodOptional<z.ZodString>;
    published_at: z.ZodOptional<z.ZodString>;
    created_at: z.ZodString;
}, z.core.$strip>;
/**
 * Dead letter message type.
 */
export type DeadLetterMessage = z.infer<typeof DeadLetterMessageSchema>;
/**
 * Common options for queue API calls.
 *
 * Used to pass organization context when calling from CLI or other
 * contexts where the org is not implicit in the authentication token.
 */
export interface QueueApiOptions {
    /**
     * Organization ID for the request.
     * Required when using user authentication (CLI) instead of SDK key.
     */
    orgId?: string;
    /**
     * Whether to publish synchronously.
     * When true, the API waits for the message to be fully persisted before returning.
     * When false (default), the API returns immediately with a pending message.
     */
    sync?: boolean;
}
/**
 * Request schema for creating a new queue.
 *
 * @example
 * ```typescript
 * const request: CreateQueueRequest = {
 *   name: 'my-worker-queue',
 *   queue_type: 'worker',
 *   description: 'Processes background jobs',
 *   settings: { default_max_retries: 3 },
 * };
 * ```
 */
export declare const CreateQueueRequestSchema: z.ZodObject<{
    name: z.ZodOptional<z.ZodString>;
    description: z.ZodOptional<z.ZodString>;
    queue_type: z.ZodEnum<{
        worker: "worker";
        pubsub: "pubsub";
    }>;
    settings: z.ZodOptional<z.ZodObject<{
        default_ttl_seconds: z.ZodOptional<z.ZodOptional<z.ZodNullable<z.ZodNumber>>>;
        default_visibility_timeout_seconds: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        default_max_retries: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        default_retry_backoff_ms: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        default_retry_max_backoff_ms: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        default_retry_multiplier: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        max_in_flight_per_client: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        retention_seconds: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
    }, z.core.$strip>>;
}, z.core.$strip>;
/** Request type for creating a queue. */
export type CreateQueueRequest = z.infer<typeof CreateQueueRequestSchema>;
/**
 * Request schema for updating an existing queue.
 */
export declare const UpdateQueueRequestSchema: z.ZodObject<{
    description: z.ZodOptional<z.ZodString>;
    settings: z.ZodOptional<z.ZodObject<{
        default_ttl_seconds: z.ZodOptional<z.ZodOptional<z.ZodNullable<z.ZodNumber>>>;
        default_visibility_timeout_seconds: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        default_max_retries: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        default_retry_backoff_ms: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        default_retry_max_backoff_ms: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        default_retry_multiplier: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        max_in_flight_per_client: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
        retention_seconds: z.ZodOptional<z.ZodOptional<z.ZodNumber>>;
    }, z.core.$strip>>;
}, z.core.$strip>;
/** Request type for updating a queue. */
export type UpdateQueueRequest = z.infer<typeof UpdateQueueRequestSchema>;
/**
 * Request schema for listing queues with pagination.
 */
export declare const ListQueuesRequestSchema: z.ZodObject<{
    limit: z.ZodOptional<z.ZodNumber>;
    offset: z.ZodOptional<z.ZodNumber>;
    name: z.ZodOptional<z.ZodString>;
    queue_type: z.ZodOptional<z.ZodEnum<{
        worker: "worker";
        pubsub: "pubsub";
    }>>;
    status: z.ZodOptional<z.ZodEnum<{
        paused: "paused";
        active: "active";
    }>>;
    sort: z.ZodOptional<z.ZodEnum<{
        name: "name";
        created: "created";
        updated: "updated";
        message_count: "message_count";
        dlq_count: "dlq_count";
    }>>;
    direction: z.ZodOptional<z.ZodEnum<{
        asc: "asc";
        desc: "desc";
    }>>;
}, z.core.$strip>;
/** Request type for listing queues. */
export type ListQueuesRequest = z.infer<typeof ListQueuesRequestSchema>;
/**
 * Request schema for publishing a message to a queue.
 *
 * @example
 * ```typescript
 * const request: PublishMessageRequest = {
 *   payload: { task: 'process-order', orderId: 123 },
 *   metadata: { priority: 'high' },
 *   idempotency_key: 'order-123-v1',
 *   ttl_seconds: 3600,
 * };
 * ```
 */
export declare const PublishMessageRequestSchema: z.ZodObject<{
    payload: z.ZodUnknown;
    metadata: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodUnknown>>;
    idempotency_key: z.ZodOptional<z.ZodString>;
    partition_key: z.ZodOptional<z.ZodString>;
    ttl_seconds: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
/** Request type for publishing a message. */
export type PublishMessageRequest = z.infer<typeof PublishMessageRequestSchema>;
/**
 * Request schema for batch publishing multiple messages.
 *
 * @example
 * ```typescript
 * const request: BatchPublishMessagesRequest = {
 *   messages: [
 *     { payload: { task: 'a' } },
 *     { payload: { task: 'b' } },
 *   ],
 * };
 * ```
 */
export declare const BatchPublishMessagesRequestSchema: z.ZodObject<{
    messages: z.ZodArray<z.ZodObject<{
        payload: z.ZodUnknown;
        metadata: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodUnknown>>;
        idempotency_key: z.ZodOptional<z.ZodString>;
        partition_key: z.ZodOptional<z.ZodString>;
        ttl_seconds: z.ZodOptional<z.ZodNumber>;
    }, z.core.$strip>>;
}, z.core.$strip>;
/** Request type for batch publishing messages. */
export type BatchPublishMessagesRequest = z.infer<typeof BatchPublishMessagesRequestSchema>;
/**
 * Request schema for listing messages with pagination and filtering.
 */
export declare const ListMessagesRequestSchema: z.ZodObject<{
    limit: z.ZodOptional<z.ZodNumber>;
    offset: z.ZodOptional<z.ZodNumber>;
    state: z.ZodOptional<z.ZodEnum<{
        pending: "pending";
        failed: "failed";
        leased: "leased";
        processing: "processing";
        delivered: "delivered";
        dead: "dead";
    }>>;
}, z.core.$strip>;
/** Request type for listing messages. */
export type ListMessagesRequest = z.infer<typeof ListMessagesRequestSchema>;
/**
 * Request schema for consuming messages from a specific offset.
 */
export declare const ConsumeMessagesRequestSchema: z.ZodObject<{
    offset: z.ZodNumber;
    limit: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
/** Request type for consuming messages. */
export type ConsumeMessagesRequest = z.infer<typeof ConsumeMessagesRequestSchema>;
/**
 * Request schema for creating a destination webhook.
 *
 * @example
 * ```typescript
 * const request: CreateDestinationRequest = {
 *   destination_type: 'http',
 *   config: {
 *     url: 'https://api.example.com/webhook',
 *     method: 'POST',
 *   },
 *   enabled: true,
 * };
 * ```
 */
export declare const CreateDestinationRequestSchema: z.ZodObject<{
    destination_type: z.ZodEnum<{
        http: "http";
    }>;
    config: z.ZodObject<{
        url: z.ZodString;
        headers: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
        method: z.ZodDefault<z.ZodString>;
        timeout_ms: z.ZodDefault<z.ZodNumber>;
        retry_policy: z.ZodOptional<z.ZodObject<{
            max_attempts: z.ZodDefault<z.ZodNumber>;
            initial_backoff_ms: z.ZodDefault<z.ZodNumber>;
            max_backoff_ms: z.ZodDefault<z.ZodNumber>;
            backoff_multiplier: z.ZodDefault<z.ZodNumber>;
        }, z.core.$strip>>;
        signing: z.ZodOptional<z.ZodObject<{
            enabled: z.ZodDefault<z.ZodBoolean>;
            secret_key: z.ZodOptional<z.ZodString>;
        }, z.core.$strip>>;
    }, z.core.$strip>;
    enabled: z.ZodDefault<z.ZodBoolean>;
}, z.core.$strip>;
/** Request type for creating a destination. */
export type CreateDestinationRequest = z.infer<typeof CreateDestinationRequestSchema>;
/**
 * Request schema for updating a destination.
 */
export declare const UpdateDestinationRequestSchema: z.ZodObject<{
    config: z.ZodOptional<z.ZodObject<{
        url: z.ZodOptional<z.ZodString>;
        headers: z.ZodOptional<z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>>;
        method: z.ZodOptional<z.ZodDefault<z.ZodString>>;
        timeout_ms: z.ZodOptional<z.ZodDefault<z.ZodNumber>>;
        retry_policy: z.ZodOptional<z.ZodOptional<z.ZodObject<{
            max_attempts: z.ZodDefault<z.ZodNumber>;
            initial_backoff_ms: z.ZodDefault<z.ZodNumber>;
            max_backoff_ms: z.ZodDefault<z.ZodNumber>;
            backoff_multiplier: z.ZodDefault<z.ZodNumber>;
        }, z.core.$strip>>>;
        signing: z.ZodOptional<z.ZodOptional<z.ZodObject<{
            enabled: z.ZodDefault<z.ZodBoolean>;
            secret_key: z.ZodOptional<z.ZodString>;
        }, z.core.$strip>>>;
    }, z.core.$strip>>;
    enabled: z.ZodOptional<z.ZodBoolean>;
}, z.core.$strip>;
/** Request type for updating a destination. */
export type UpdateDestinationRequest = z.infer<typeof UpdateDestinationRequestSchema>;
/**
 * Request schema for listing dead letter queue messages with pagination.
 */
export declare const ListDlqRequestSchema: z.ZodObject<{
    limit: z.ZodOptional<z.ZodNumber>;
    offset: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
/** Request type for listing DLQ messages. */
export type ListDlqRequest = z.infer<typeof ListDlqRequestSchema>;
/**
 * Time bucket granularity for analytics queries.
 *
 * - `minute`: 1-minute buckets, max range 24 hours. Best for real-time monitoring.
 * - `hour`: 1-hour buckets, max range 7 days. Best for short-term trend analysis.
 * - `day`: 1-day buckets, max range 90 days. Best for long-term analysis.
 *
 * @example
 * ```typescript
 * const analytics = await getQueueAnalytics(client, 'my-queue', {
 *   granularity: 'hour',
 *   start: '2026-01-14T00:00:00Z',
 * });
 * ```
 */
export declare const AnalyticsGranularitySchema: z.ZodEnum<{
    minute: "minute";
    hour: "hour";
    day: "day";
}>;
/**
 * Time bucket granularity type.
 */
export type AnalyticsGranularity = z.infer<typeof AnalyticsGranularitySchema>;
/**
 * Options for analytics queries.
 *
 * Use these options to filter and configure analytics requests by time range,
 * granularity, and optional filters like project or agent ID.
 *
 * @example
 * ```typescript
 * const options: AnalyticsOptions = {
 *   start: '2026-01-14T00:00:00Z',
 *   end: '2026-01-15T00:00:00Z',
 *   granularity: 'hour',
 *   projectId: 'proj_abc123',
 * };
 * const analytics = await getQueueAnalytics(client, 'my-queue', options);
 * ```
 */
export interface AnalyticsOptions extends QueueApiOptions {
    /**
     * Start of time range in ISO8601 format.
     * @default 24 hours ago
     * @example '2026-01-14T00:00:00Z'
     */
    start?: string;
    /**
     * End of time range in ISO8601 format.
     * @default Current time
     * @example '2026-01-15T00:00:00Z'
     */
    end?: string;
    /**
     * Time bucket granularity for aggregation.
     * @default 'hour'
     */
    granularity?: AnalyticsGranularity;
    /**
     * Filter analytics to messages from a specific project.
     * Only messages with matching `project_id` will be included.
     */
    projectId?: string;
    /**
     * Filter analytics to messages from a specific agent.
     * Only messages with matching `agent_id` will be included.
     */
    agentId?: string;
}
/**
 * Options for real-time SSE streaming analytics.
 *
 * SSE (Server-Sent Events) streams provide live updates of queue statistics
 * at a configurable interval. The stream stays open until closed by the client.
 *
 * @example
 * ```typescript
 * const stream = streamQueueAnalytics(client, 'my-queue', { interval: 5 });
 * for await (const event of stream) {
 *   console.log(`Backlog: ${event.backlog}`);
 * }
 * ```
 */
export interface StreamAnalyticsOptions extends QueueApiOptions {
    /**
     * Interval between stats updates in seconds.
     * @default 5
     * @minimum 1
     * @maximum 60
     */
    interval?: number;
}
/**
 * Time period for analytics responses.
 *
 * Represents the time range and granularity of the analytics data.
 */
export declare const TimePeriodSchema: z.ZodObject<{
    start: z.ZodString;
    end: z.ZodString;
    granularity: z.ZodOptional<z.ZodEnum<{
        minute: "minute";
        hour: "hour";
        day: "day";
    }>>;
}, z.core.$strip>;
/**
 * Time period type representing a date range for analytics.
 */
export type TimePeriod = z.infer<typeof TimePeriodSchema>;
/**
 * Latency statistics with percentile distributions.
 *
 * Provides average, percentile (p50, p95, p99), and maximum latency values
 * for message delivery operations.
 *
 * @example
 * ```typescript
 * const { latency } = await getQueueAnalytics(client, 'my-queue');
 * console.log(`Average: ${latency.avg_ms}ms, P95: ${latency.p95_ms}ms`);
 * ```
 */
export declare const LatencyStatsSchema: z.ZodObject<{
    avg_ms: z.ZodNumber;
    p50_ms: z.ZodOptional<z.ZodNumber>;
    p95_ms: z.ZodOptional<z.ZodNumber>;
    p99_ms: z.ZodOptional<z.ZodNumber>;
    max_ms: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
/**
 * Latency statistics type.
 */
export type LatencyStats = z.infer<typeof LatencyStatsSchema>;
/**
 * Current real-time queue state.
 *
 * Represents the instantaneous state of a queue, useful for monitoring
 * dashboards and alerting on queue health.
 *
 * @example
 * ```typescript
 * const { current } = await getQueueAnalytics(client, 'my-queue');
 * if (current.backlog > 1000) {
 *   console.warn('Queue backlog is high!');
 * }
 * ```
 */
export declare const QueueCurrentStatsSchema: z.ZodObject<{
    backlog: z.ZodNumber;
    dlq_count: z.ZodNumber;
    messages_in_flight: z.ZodNumber;
    active_consumers: z.ZodNumber;
    oldest_message_age_seconds: z.ZodOptional<z.ZodNullable<z.ZodNumber>>;
}, z.core.$strip>;
/**
 * Current queue state type.
 */
export type QueueCurrentStats = z.infer<typeof QueueCurrentStatsSchema>;
/**
 * Aggregated statistics for a time period.
 *
 * Contains counts and metrics aggregated over the requested time range.
 *
 * @example
 * ```typescript
 * const { period_stats } = await getQueueAnalytics(client, 'my-queue');
 * const successRate = period_stats.messages_acknowledged / period_stats.messages_published;
 * console.log(`Success rate: ${(successRate * 100).toFixed(1)}%`);
 * ```
 */
export declare const QueuePeriodStatsSchema: z.ZodObject<{
    messages_published: z.ZodNumber;
    messages_delivered: z.ZodNumber;
    messages_acknowledged: z.ZodNumber;
    messages_failed: z.ZodNumber;
    messages_replayed: z.ZodNumber;
    bytes_published: z.ZodNumber;
    delivery_attempts: z.ZodNumber;
    retry_count: z.ZodNumber;
}, z.core.$strip>;
/**
 * Period statistics type.
 */
export type QueuePeriodStats = z.infer<typeof QueuePeriodStatsSchema>;
/**
 * Analytics for a webhook destination.
 *
 * Provides delivery statistics for a configured webhook endpoint.
 *
 * @example
 * ```typescript
 * const { destinations } = await getQueueAnalytics(client, 'my-queue');
 * for (const dest of destinations ?? []) {
 *   const successRate = dest.success_count / (dest.success_count + dest.failure_count);
 *   console.log(`${dest.url}: ${(successRate * 100).toFixed(1)}% success`);
 * }
 * ```
 */
export declare const DestinationAnalyticsSchema: z.ZodObject<{
    id: z.ZodString;
    type: z.ZodString;
    url: z.ZodString;
    success_count: z.ZodNumber;
    failure_count: z.ZodNumber;
    avg_response_time_ms: z.ZodOptional<z.ZodNumber>;
    last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
}, z.core.$strip>;
/**
 * Destination analytics type.
 */
export type DestinationAnalytics = z.infer<typeof DestinationAnalyticsSchema>;
/**
 * Complete analytics for a single queue.
 *
 * Provides comprehensive analytics including current state, period statistics,
 * latency metrics, and destination performance.
 *
 * @example
 * ```typescript
 * const analytics = await getQueueAnalytics(client, 'order-processing');
 * console.log(`Queue: ${analytics.queue_name} (${analytics.queue_type})`);
 * console.log(`Backlog: ${analytics.current.backlog}`);
 * console.log(`Published (24h): ${analytics.period_stats.messages_published}`);
 * console.log(`P95 Latency: ${analytics.latency.p95_ms}ms`);
 * ```
 */
export declare const QueueAnalyticsSchema: z.ZodObject<{
    queue_id: z.ZodString;
    queue_name: z.ZodString;
    queue_type: z.ZodString;
    period: z.ZodObject<{
        start: z.ZodString;
        end: z.ZodString;
        granularity: z.ZodOptional<z.ZodEnum<{
            minute: "minute";
            hour: "hour";
            day: "day";
        }>>;
    }, z.core.$strip>;
    current: z.ZodObject<{
        backlog: z.ZodNumber;
        dlq_count: z.ZodNumber;
        messages_in_flight: z.ZodNumber;
        active_consumers: z.ZodNumber;
        oldest_message_age_seconds: z.ZodOptional<z.ZodNullable<z.ZodNumber>>;
    }, z.core.$strip>;
    period_stats: z.ZodObject<{
        messages_published: z.ZodNumber;
        messages_delivered: z.ZodNumber;
        messages_acknowledged: z.ZodNumber;
        messages_failed: z.ZodNumber;
        messages_replayed: z.ZodNumber;
        bytes_published: z.ZodNumber;
        delivery_attempts: z.ZodNumber;
        retry_count: z.ZodNumber;
    }, z.core.$strip>;
    latency: z.ZodObject<{
        avg_ms: z.ZodNumber;
        p50_ms: z.ZodOptional<z.ZodNumber>;
        p95_ms: z.ZodOptional<z.ZodNumber>;
        p99_ms: z.ZodOptional<z.ZodNumber>;
        max_ms: z.ZodOptional<z.ZodNumber>;
    }, z.core.$strip>;
    consumer_latency: z.ZodObject<{
        avg_ms: z.ZodNumber;
        p50_ms: z.ZodOptional<z.ZodNumber>;
        p95_ms: z.ZodOptional<z.ZodNumber>;
        p99_ms: z.ZodOptional<z.ZodNumber>;
        max_ms: z.ZodOptional<z.ZodNumber>;
    }, z.core.$strip>;
    destinations: z.ZodOptional<z.ZodArray<z.ZodObject<{
        id: z.ZodString;
        type: z.ZodString;
        url: z.ZodString;
        success_count: z.ZodNumber;
        failure_count: z.ZodNumber;
        avg_response_time_ms: z.ZodOptional<z.ZodNumber>;
        last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
        last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    }, z.core.$strip>>>;
}, z.core.$strip>;
/**
 * Queue analytics type.
 */
export type QueueAnalytics = z.infer<typeof QueueAnalyticsSchema>;
/**
 * Summary statistics for a queue in org-level analytics.
 *
 * Provides a condensed view of queue metrics for listing in dashboards.
 */
export declare const QueueSummarySchema: z.ZodObject<{
    id: z.ZodString;
    name: z.ZodString;
    queue_type: z.ZodString;
    messages_published: z.ZodNumber;
    messages_delivered: z.ZodNumber;
    messages_acknowledged: z.ZodNumber;
    backlog: z.ZodNumber;
    dlq_count: z.ZodNumber;
    avg_latency_ms: z.ZodNumber;
    error_rate_percent: z.ZodNumber;
}, z.core.$strip>;
/**
 * Queue summary type for org-level listings.
 */
export type QueueSummary = z.infer<typeof QueueSummarySchema>;
/**
 * Aggregated summary across all queues in an organization.
 *
 * @example
 * ```typescript
 * const { summary } = await getOrgAnalytics(client);
 * console.log(`Total queues: ${summary.total_queues}`);
 * console.log(`Total messages: ${summary.total_messages_published}`);
 * console.log(`Error rate: ${summary.error_rate_percent.toFixed(2)}%`);
 * ```
 */
export declare const OrgAnalyticsSummarySchema: z.ZodObject<{
    total_queues: z.ZodNumber;
    total_messages_published: z.ZodNumber;
    total_messages_delivered: z.ZodNumber;
    total_messages_acknowledged: z.ZodNumber;
    total_dlq_messages: z.ZodNumber;
    total_bytes_published: z.ZodNumber;
    avg_latency_ms: z.ZodNumber;
    p95_latency_ms: z.ZodNumber;
    error_rate_percent: z.ZodNumber;
}, z.core.$strip>;
/**
 * Org-level analytics summary type.
 */
export type OrgAnalyticsSummary = z.infer<typeof OrgAnalyticsSummarySchema>;
/**
 * Complete organization-level analytics.
 *
 * Provides an overview of all queues with aggregated metrics and per-queue summaries.
 *
 * @example
 * ```typescript
 * const analytics = await getOrgAnalytics(client);
 * console.log(`Org: ${analytics.org_id}`);
 * console.log(`Queues: ${analytics.summary.total_queues}`);
 * for (const queue of analytics.queues) {
 *   console.log(`  ${queue.name}: ${queue.backlog} pending`);
 * }
 * ```
 */
export declare const OrgAnalyticsSchema: z.ZodObject<{
    org_id: z.ZodString;
    period: z.ZodObject<{
        start: z.ZodString;
        end: z.ZodString;
        granularity: z.ZodOptional<z.ZodEnum<{
            minute: "minute";
            hour: "hour";
            day: "day";
        }>>;
    }, z.core.$strip>;
    summary: z.ZodObject<{
        total_queues: z.ZodNumber;
        total_messages_published: z.ZodNumber;
        total_messages_delivered: z.ZodNumber;
        total_messages_acknowledged: z.ZodNumber;
        total_dlq_messages: z.ZodNumber;
        total_bytes_published: z.ZodNumber;
        avg_latency_ms: z.ZodNumber;
        p95_latency_ms: z.ZodNumber;
        error_rate_percent: z.ZodNumber;
    }, z.core.$strip>;
    queues: z.ZodArray<z.ZodObject<{
        id: z.ZodString;
        name: z.ZodString;
        queue_type: z.ZodString;
        messages_published: z.ZodNumber;
        messages_delivered: z.ZodNumber;
        messages_acknowledged: z.ZodNumber;
        backlog: z.ZodNumber;
        dlq_count: z.ZodNumber;
        avg_latency_ms: z.ZodNumber;
        error_rate_percent: z.ZodNumber;
    }, z.core.$strip>>;
}, z.core.$strip>;
/**
 * Org-level analytics type.
 */
export type OrgAnalytics = z.infer<typeof OrgAnalyticsSchema>;
/**
 * Single data point in a time series.
 *
 * Represents metrics for one time bucket (minute, hour, or day).
 * Used for building charts and visualizing trends over time.
 *
 * @example
 * ```typescript
 * const { series } = await getQueueTimeSeries(client, 'my-queue', { granularity: 'hour' });
 * for (const point of series) {
 *   console.log(`${point.timestamp}: ${point.throughput} msg/h, ${point.avg_latency_ms}ms avg`);
 * }
 * ```
 */
export declare const TimeSeriesPointSchema: z.ZodObject<{
    timestamp: z.ZodString;
    throughput: z.ZodNumber;
    delivery_rate: z.ZodNumber;
    ack_rate: z.ZodNumber;
    error_rate: z.ZodNumber;
    avg_latency_ms: z.ZodNumber;
    p95_latency_ms: z.ZodOptional<z.ZodNumber>;
    backlog: z.ZodOptional<z.ZodNumber>;
    messages_in_flight: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>;
/**
 * Time series data point type.
 */
export type TimeSeriesPoint = z.infer<typeof TimeSeriesPointSchema>;
/**
 * Time series analytics data for charting and visualization.
 *
 * Contains an array of data points at the requested granularity for building
 * time-based charts and dashboards.
 *
 * @example
 * ```typescript
 * const timeseries = await getQueueTimeSeries(client, 'order-processing', {
 *   granularity: 'hour',
 *   start: '2026-01-14T00:00:00Z',
 *   end: '2026-01-15T00:00:00Z',
 * });
 *
 * // Plot throughput over time
 * const chartData = timeseries.series.map(p => ({
 *   x: new Date(p.timestamp),
 *   y: p.throughput,
 * }));
 * ```
 */
export declare const TimeSeriesDataSchema: z.ZodObject<{
    queue_id: z.ZodString;
    queue_name: z.ZodString;
    period: z.ZodObject<{
        start: z.ZodString;
        end: z.ZodString;
        granularity: z.ZodOptional<z.ZodEnum<{
            minute: "minute";
            hour: "hour";
            day: "day";
        }>>;
    }, z.core.$strip>;
    series: z.ZodArray<z.ZodObject<{
        timestamp: z.ZodString;
        throughput: z.ZodNumber;
        delivery_rate: z.ZodNumber;
        ack_rate: z.ZodNumber;
        error_rate: z.ZodNumber;
        avg_latency_ms: z.ZodNumber;
        p95_latency_ms: z.ZodOptional<z.ZodNumber>;
        backlog: z.ZodOptional<z.ZodNumber>;
        messages_in_flight: z.ZodOptional<z.ZodNumber>;
    }, z.core.$strip>>;
}, z.core.$strip>;
/**
 * Time series data type.
 */
export type TimeSeriesData = z.infer<typeof TimeSeriesDataSchema>;
/**
 * Real-time stats event from SSE stream.
 *
 * Represents a single snapshot of queue statistics delivered via Server-Sent Events.
 * Events are pushed at the interval specified when opening the stream.
 *
 * @example
 * ```typescript
 * const stream = streamQueueAnalytics(client, 'my-queue', { interval: 5 });
 * for await (const event of stream) {
 *   updateDashboard({
 *     backlog: event.backlog,
 *     throughput: event.throughput_1m,
 *     latency: event.avg_latency_ms,
 *     consumers: event.active_consumers,
 *   });
 * }
 * ```
 */
export declare const SSEStatsEventSchema: z.ZodObject<{
    timestamp: z.ZodString;
    backlog: z.ZodNumber;
    messages_in_flight: z.ZodNumber;
    throughput_1m: z.ZodNumber;
    delivery_rate_1m: z.ZodNumber;
    error_rate_1m: z.ZodNumber;
    avg_latency_ms: z.ZodNumber;
    active_consumers: z.ZodNumber;
}, z.core.$strip>;
/**
 * SSE stats event type for real-time streaming.
 */
export type SSEStatsEvent = z.infer<typeof SSEStatsEventSchema>;
/**
 * Source authentication type schema.
 */
export declare const SourceAuthTypeSchema: z.ZodEnum<{
    none: "none";
    basic: "basic";
    header: "header";
}>;
/**
 * Source authentication type.
 */
export type SourceAuthType = z.infer<typeof SourceAuthTypeSchema>;
/**
 * Queue source schema representing an HTTP ingestion endpoint.
 *
 * Sources provide public URLs for ingesting data into queues from external sources.
 * They support various authentication methods to secure access.
 *
 * @example
 * ```typescript
 * const source = await getSource(client, 'my-queue', 'qsrc_abc123');
 * console.log(`Source URL: ${source.url}`);
 * console.log(`Success rate: ${source.success_count}/${source.request_count}`);
 * ```
 */
export declare const SourceSchema: z.ZodObject<{
    id: z.ZodString;
    queue_id: z.ZodString;
    name: z.ZodString;
    description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    auth_type: z.ZodEnum<{
        none: "none";
        basic: "basic";
        header: "header";
    }>;
    enabled: z.ZodBoolean;
    url: z.ZodString;
    request_count: z.ZodNumber;
    success_count: z.ZodNumber;
    failure_count: z.ZodNumber;
    last_request_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    last_failure_error: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    created_at: z.ZodString;
    updated_at: z.ZodString;
}, z.core.$strip>;
/**
 * Queue source type.
 */
export type Source = z.infer<typeof SourceSchema>;
/**
 * Create source request schema.
 *
 * @example
 * ```typescript
 * const request: CreateSourceRequest = {
 *   name: 'webhook-ingestion',
 *   description: 'Receives webhooks from external service',
 *   auth_type: 'header',
 *   auth_value: 'Bearer my-secret-token',
 * };
 * ```
 */
export declare const CreateSourceRequestSchema: z.ZodObject<{
    name: z.ZodString;
    description: z.ZodOptional<z.ZodString>;
    auth_type: z.ZodDefault<z.ZodOptional<z.ZodEnum<{
        none: "none";
        basic: "basic";
        header: "header";
    }>>>;
    auth_value: z.ZodOptional<z.ZodString>;
}, z.core.$strip>;
/**
 * Create source request type.
 */
export type CreateSourceRequest = z.infer<typeof CreateSourceRequestSchema>;
/**
 * Update source request schema.
 *
 * All fields are optional - only provided fields will be updated.
 *
 * @example
 * ```typescript
 * // Disable a source
 * const request: UpdateSourceRequest = { enabled: false };
 *
 * // Update authentication
 * const request: UpdateSourceRequest = {
 *   auth_type: 'basic',
 *   auth_value: 'user:password',
 * };
 * ```
 */
export declare const UpdateSourceRequestSchema: z.ZodObject<{
    name: z.ZodOptional<z.ZodString>;
    description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
    auth_type: z.ZodOptional<z.ZodEnum<{
        none: "none";
        basic: "basic";
        header: "header";
    }>>;
    auth_value: z.ZodOptional<z.ZodString>;
    enabled: z.ZodOptional<z.ZodBoolean>;
}, z.core.$strip>;
/**
 * Update source request type.
 */
export type UpdateSourceRequest = z.infer<typeof UpdateSourceRequestSchema>;
//# sourceMappingURL=types.d.ts.map