import { z } from 'zod';
import { AgentRoleSchema } from '../types';
import { agents } from '../agents';
export const BackgroundTaskArgsSchema = z.object({
    agent: AgentRoleSchema.describe('Agent role to run the task'),
    task: z.string().describe('Task prompt to run in the background'),
    description: z.string().optional().describe('Short description of the task'),
});
export const BackgroundOutputArgsSchema = z.object({
    task_id: z.string().describe('Background task ID'),
});
export const BackgroundCancelArgsSchema = z.object({
    task_id: z.string().describe('Background task ID'),
});
export function createBackgroundTools(manager) {
    const backgroundTaskTool = {
        name: 'agentuity_background_task',
        description: 'Launch a task to run in the background.',
        args: BackgroundTaskArgsSchema,
        async execute(args, context) {
            const agentName = resolveAgentName(args.agent);
            const task = await manager.launch({
                description: args.description ?? args.task,
                prompt: args.task,
                agent: agentName,
                parentSessionId: context.sessionID,
                parentMessageId: context.messageID,
            });
            return {
                taskId: task.id,
                sessionId: task.sessionId,
                status: task.status,
                message: task.status === 'error'
                    ? (task.error ?? 'Failed to launch background task.')
                    : 'Background task launched.',
            };
        },
    };
    const backgroundOutputTool = {
        name: 'agentuity_background_output',
        description: 'Retrieve output for a background task.',
        args: BackgroundOutputArgsSchema,
        async execute(args) {
            const task = manager.getTask(args.task_id);
            if (!task) {
                return {
                    taskId: args.task_id,
                    status: 'error',
                    error: 'Task not found.',
                };
            }
            return {
                taskId: task.id,
                sessionId: task.sessionId,
                status: task.status,
                result: task.result,
                error: task.error,
            };
        },
    };
    const backgroundCancelTool = {
        name: 'agentuity_background_cancel',
        description: 'Cancel a running background task.',
        args: BackgroundCancelArgsSchema,
        async execute(args) {
            const success = manager.cancel(args.task_id);
            return {
                taskId: args.task_id,
                success,
                message: success ? 'Background task cancelled.' : 'Unable to cancel task.',
            };
        },
    };
    return { backgroundTaskTool, backgroundOutputTool, backgroundCancelTool };
}
function resolveAgentName(role) {
    const agent = agents[role];
    return agent?.displayName ?? role;
}
//# sourceMappingURL=background.js.map