import type { PluginInput } from '@opencode-ai/plugin';
import type { OpenCodeDBReader } from '../sqlite';
import type { BackgroundTask, BackgroundTaskConfig, BackgroundTaskStatus, LaunchInput, TaskInspection } from './types';
type EventPayload = {
    type: string;
    properties?: Record<string, unknown>;
};
export interface BackgroundManagerCallbacks {
    onSubagentSessionCreated?: (event: {
        sessionId: string;
        parentId: string;
        title: string;
    }) => void;
    onSubagentSessionDeleted?: (event: {
        sessionId: string;
    }) => void;
    onShutdown?: () => void | Promise<void>;
}
export declare class BackgroundManager {
    private ctx;
    private config;
    private concurrency;
    private callbacks?;
    private dbReader?;
    private serverUrl;
    private authHeaders;
    private tasks;
    private tasksByParent;
    private tasksBySession;
    private notifications;
    private toolCallIds;
    private shuttingDown;
    constructor(ctx: PluginInput, config?: BackgroundTaskConfig, callbacks?: BackgroundManagerCallbacks, dbReader?: OpenCodeDBReader);
    /**
     * Resolve the server URL from the plugin context.
     * Mirrors the defensive pattern used in the tmux manager to handle
     * sandbox environments where the client may not have a baseUrl configured.
     */
    private resolveServerUrl;
    /**
     * Resolve authentication headers from environment variables.
     *
     * Reads `OPENCODE_SERVER_USERNAME` and `OPENCODE_SERVER_PASSWORD` (set
     * automatically by the OpenCode server in sandbox environments) and
     * produces a Basic Auth header (`base64("username:password")`).
     *
     * In sandbox environments the SDK client's default auth may not carry over
     * when a per-call `baseUrl` override is provided, so we need to explicitly
     * attach these credentials for server-to-server requests.
     */
    private resolveAuthHeaders;
    /**
     * Build the per-call client overrides (baseUrl + auth headers).
     * Spread this into every SDK client call so both the server URL and
     * authentication are correctly forwarded in sandbox environments.
     */
    private getClientOverrides;
    launch(input: LaunchInput): Promise<BackgroundTask>;
    getTask(id: string): BackgroundTask | undefined;
    getTasksByParent(sessionId: string): BackgroundTask[];
    findBySession(sessionId: string): BackgroundTask | undefined;
    /**
     * Inspect a background task by fetching its session messages.
     * Useful for seeing what a child Lead or other agent is doing.
     */
    inspectTask(taskId: string): Promise<TaskInspection | undefined>;
    /**
     * Refresh task statuses from the server.
     * Useful for recovering state after issues or checking on stuck tasks.
     */
    refreshStatuses(): Promise<Map<string, BackgroundTaskStatus>>;
    /**
     * Recover background tasks from existing sessions.
     * Call this on plugin startup to restore state after restart.
     *
     * This method queries all sessions and reconstructs task state from
     * sessions that have JSON-encoded task metadata in their title.
     *
     * @returns The number of tasks recovered
     */
    recoverTasks(): Promise<number>;
    private mapSessionStatusToTaskStatus;
    cancel(taskId: string): boolean;
    handleEvent(event: EventPayload): void;
    markForNotification(task: BackgroundTask): void;
    getPendingNotifications(sessionId: string): BackgroundTask[];
    clearNotifications(sessionId: string): void;
    shutdown(): void;
    private indexTask;
    private startTask;
    private updateProgress;
    private createProgress;
    private completeTask;
    private failTask;
    private releaseConcurrency;
    private notifyParent;
    private abortSession;
    private fetchLatestResult;
    private mapDbStatusToTaskStatus;
    private getConcurrencyGroup;
    private getConcurrencyKey;
    private expireStaleTasks;
}
export {};
//# sourceMappingURL=manager.d.ts.map