import type { JSONSchema7 } from 'ai';
import type { GetAuthHeaders } from '../components/internal/workbench-provider';
export interface AgentSchema {
    input?: {
        code?: string;
        json?: JSONSchema7;
    };
    output?: {
        code?: string;
        json?: string;
    };
}
export interface AgentMetadata {
    agentId: string;
    description?: string;
    filename?: string;
    id: string;
    identifier?: string;
    name: string;
    version?: string;
}
export interface AgentSchemaData {
    metadata: AgentMetadata;
    schema: AgentSchema;
}
export interface AgentSchemasResponse {
    agents: Record<string, AgentSchemaData>;
}
export interface UseAgentSchemasOptions {
    apiKey?: string;
    baseUrl?: string;
    enabled?: boolean;
    headers?: Record<string, string>;
    getAuthHeaders?: GetAuthHeaders;
}
export interface UseAgentSchemasResult {
    data: AgentSchemasResponse | null;
    error: Error | null;
    isLoading: boolean;
    refetch: () => void;
}
/**
 * React hook for fetching agent schemas from the workbench metadata endpoint
 *
 * @example
 * ```tsx
 * const { data, isLoading, error, refetch } = useAgentSchemas({
 *   baseUrl: 'http://localhost:3000',
 *   apiKey: 'your-api-key', // optional
 *   enabled: true
 * });
 *
 * if (isLoading) return <div>Loading schemas...</div>;
 * if (error) return <div>Error: {error.message}</div>;
 * if (data) {
 *   Object.entries(data.agents).forEach(([name, agentData]) => {
 *     console.log(`Agent ${name}:`, agentData.schema);
 *   });
 * }
 * ```
 */
export declare function useAgentSchemas(options?: UseAgentSchemasOptions): UseAgentSchemasResult;
/**
 * Helper hook to get a specific agent's schema by name
 */
export declare function useAgentSchema(agentName: string, options?: UseAgentSchemasOptions): {
    data: AgentSchemaData | null;
    error: Error | null;
    isLoading: boolean;
    metadata: AgentMetadata | null;
    refetch: () => void;
    schema: AgentSchema | null;
};
//# sourceMappingURL=useAgentSchemas.d.ts.map