import { APIClient } from '../api';
import { sandboxCreate } from './create';
import { sandboxDestroy } from './destroy';
import { sandboxGet } from './get';
import { sandboxExecute } from './execute';
import { sandboxWriteFiles, sandboxReadFile } from './files';
import { sandboxRun } from './run';
import { executionGet } from './execution';
import { ConsoleLogger } from '../../logger';
import { getServiceUrls } from '../../config';
import { writeAndDrain } from './util';
// Server-side long-poll wait duration (max 5 minutes supported by server)
const EXECUTION_WAIT_DURATION = '5m';
/**
 * Wait for execution completion using server-side long-polling.
 * This is more efficient than client-side polling and provides immediate
 * error detection if the sandbox is terminated.
 */
async function waitForExecution(client, executionId, orgId, signal) {
    if (signal?.aborted) {
        throw new DOMException('The operation was aborted.', 'AbortError');
    }
    // Use server-side long-polling - the server will hold the connection
    // until the execution reaches a terminal state or the wait duration expires
    return executionGet(client, {
        executionId,
        orgId,
        wait: EXECUTION_WAIT_DURATION,
    });
}
/**
 * Pipes a remote stream URL to a local writable stream with proper backpressure handling
 */
async function pipeStreamToWritable(streamUrl, writable, signal) {
    const response = await fetch(streamUrl, { signal });
    if (!response.ok) {
        throw new Error(`Failed to fetch stream: ${response.status} ${response.statusText}`);
    }
    if (!response.body) {
        return;
    }
    const reader = response.body.getReader();
    try {
        while (true) {
            const { done, value } = await reader.read();
            if (done)
                break;
            if (value) {
                await writeAndDrain(writable, value);
            }
        }
    }
    finally {
        try {
            await reader.cancel();
        }
        catch {
            // Ignore cancel errors - stream may already be closed
        }
        reader.releaseLock();
    }
}
/**
 * Convenience client for sandbox operations.
 *
 * @example
 * ```typescript
 * // Interactive sandbox usage
 * const client = new SandboxClient();
 * const sandbox = await client.create();
 * const result = await sandbox.execute({ command: ['echo', 'hello'] });
 * await sandbox.destroy();
 *
 * // One-shot execution with streaming
 * const result = await client.run(
 *   { command: { exec: ['bun', 'run', 'script.ts'] } },
 *   { stdout: process.stdout, stderr: process.stderr }
 * );
 * ```
 */
export class SandboxClient {
    #client;
    #orgId;
    #apiKey;
    #region;
    #logger;
    constructor(options = {}) {
        const apiKey = options.apiKey || process.env.AGENTUITY_SDK_KEY || process.env.AGENTUITY_CLI_KEY;
        const region = process.env.AGENTUITY_REGION ?? 'usc';
        const serviceUrls = getServiceUrls(region);
        const url = options.url ||
            process.env.AGENTUITY_SANDBOX_URL ||
            process.env.AGENTUITY_CATALYST_URL ||
            process.env.AGENTUITY_TRANSPORT_URL ||
            serviceUrls.sandbox;
        const logger = options.logger ?? new ConsoleLogger('warn');
        // Disable retries for sandbox operations - 409 Conflict means sandbox is busy,
        // not a retryable rate limit. Retrying would waste ~360s (4 attempts × 90s timeout).
        this.#client = new APIClient(url, logger, apiKey ?? '', { maxRetries: 0 });
        this.#orgId = options.orgId;
        this.#apiKey = apiKey;
        this.#region = region;
        this.#logger = logger;
    }
    /**
     * Run a one-shot command in a new sandbox (creates, executes, destroys)
     *
     * This is a high-level convenience method that handles the full lifecycle:
     * creating a sandbox, streaming I/O, polling for completion, and cleanup.
     *
     * @param options - Execution options including command and configuration
     * @param io - Optional I/O streams and abort signal
     * @returns The run result including exit code and duration
     * @throws {Error} If stdin is provided without an API key
     *
     * @example
     * ```typescript
     * const client = new SandboxClient();
     * const result = await client.run(
     *   { command: { exec: ['bun', 'run', 'script.ts'] } },
     *   { stdout: process.stdout, stderr: process.stderr }
     * );
     * console.log('Exit code:', result.exitCode);
     * ```
     */
    async run(options, io = {}) {
        if (io.stdin && !this.#apiKey) {
            throw new Error('SandboxClient.run(): stdin streaming requires an API key');
        }
        return sandboxRun(this.#client, {
            options,
            orgId: this.#orgId,
            region: this.#region,
            apiKey: this.#apiKey,
            signal: io.signal,
            stdin: io.stdin,
            stdout: io.stdout,
            stderr: io.stderr,
            logger: io.logger ?? this.#logger,
        });
    }
    /**
     * Create a new sandbox instance
     *
     * @param options - Optional sandbox configuration
     * @returns A sandbox instance with execute and destroy methods
     */
    async create(options) {
        const response = await sandboxCreate(this.#client, {
            options,
            orgId: this.#orgId,
        });
        const sandboxId = response.sandboxId;
        const client = this.#client;
        const orgId = this.#orgId;
        return {
            id: sandboxId,
            status: response.status,
            stdoutStreamUrl: response.stdoutStreamUrl,
            stderrStreamUrl: response.stderrStreamUrl,
            async execute(executeOptions) {
                const { pipe, ...coreOptions } = executeOptions;
                const initialResult = await sandboxExecute(client, {
                    sandboxId,
                    options: coreOptions,
                    orgId,
                    signal: coreOptions.signal,
                });
                // If pipe options provided, stream the output to the writable streams
                if (pipe) {
                    const streamPromises = [];
                    if (pipe.stdout && initialResult.stdoutStreamUrl) {
                        streamPromises.push(pipeStreamToWritable(initialResult.stdoutStreamUrl, pipe.stdout, coreOptions.signal));
                    }
                    if (pipe.stderr && initialResult.stderrStreamUrl) {
                        streamPromises.push(pipeStreamToWritable(initialResult.stderrStreamUrl, pipe.stderr, coreOptions.signal));
                    }
                    // Wait for all streams to complete
                    if (streamPromises.length > 0) {
                        await Promise.all(streamPromises);
                    }
                }
                // Wait for execution to complete and get final result with exit code
                const finalResult = await waitForExecution(client, initialResult.executionId, orgId, coreOptions.signal);
                return {
                    executionId: finalResult.executionId,
                    status: finalResult.status,
                    exitCode: finalResult.exitCode,
                    durationMs: finalResult.durationMs,
                    stdoutStreamUrl: initialResult.stdoutStreamUrl,
                    stderrStreamUrl: initialResult.stderrStreamUrl,
                };
            },
            async writeFiles(files) {
                const result = await sandboxWriteFiles(client, { sandboxId, files, orgId });
                return result.filesWritten;
            },
            async readFile(path) {
                return sandboxReadFile(client, { sandboxId, path, orgId });
            },
            async get() {
                return sandboxGet(client, { sandboxId, orgId });
            },
            async destroy() {
                return sandboxDestroy(client, { sandboxId, orgId });
            },
        };
    }
    /**
     * Get sandbox information by ID
     *
     * @param sandboxId - The sandbox ID
     * @returns Sandbox information
     */
    async get(sandboxId) {
        return sandboxGet(this.#client, { sandboxId, orgId: this.#orgId });
    }
    /**
     * Destroy a sandbox by ID
     *
     * @param sandboxId - The sandbox ID to destroy
     */
    async destroy(sandboxId) {
        return sandboxDestroy(this.#client, { sandboxId, orgId: this.#orgId });
    }
    /**
     * Write files to a sandbox workspace
     *
     * @param sandboxId - The sandbox ID
     * @param files - Array of files to write with path and content
     * @param signal - Optional AbortSignal to cancel the operation
     * @returns The number of files written
     */
    async writeFiles(sandboxId, files, signal) {
        const result = await sandboxWriteFiles(this.#client, {
            sandboxId,
            files,
            orgId: this.#orgId,
            signal,
        });
        return result.filesWritten;
    }
    /**
     * Read a file from a sandbox workspace
     *
     * @param sandboxId - The sandbox ID
     * @param path - Path to the file relative to the sandbox workspace
     * @param signal - Optional AbortSignal to cancel the operation
     * @returns A ReadableStream of the file contents
     */
    async readFile(sandboxId, path, signal) {
        return sandboxReadFile(this.#client, {
            sandboxId,
            path,
            orgId: this.#orgId,
            signal,
        });
    }
}
//# sourceMappingURL=client.js.map