import React from 'react';
/**
 * Result of the useAnalytics hook
 */
export interface UseAnalyticsResult {
    /**
     * Track a custom event
     */
    track: (eventName: string, properties?: Record<string, unknown>) => void;
    /**
     * Get a click handler that tracks an event
     */
    trackClick: (eventName: string, properties?: Record<string, unknown>) => (event: React.MouseEvent) => void;
    /**
     * Identify the current user
     */
    identify: (userId: string, traits?: Record<string, unknown>) => void;
    /**
     * Flush pending events
     */
    flush: () => void;
    /**
     * Whether the analytics client is available
     */
    ready: boolean;
}
/**
 * Hook for tracking analytics events in React components
 *
 * @example
 * ```tsx
 * function MyComponent() {
 *   const { track, trackClick } = useAnalytics();
 *
 *   useEffect(() => {
 *     track('component_viewed', { name: 'MyComponent' });
 *   }, [track]);
 *
 *   return (
 *     <button onClick={trackClick('button_clicked', { button: 'submit' })}>
 *       Submit
 *     </button>
 *   );
 * }
 * ```
 */
export declare function useAnalytics(): UseAnalyticsResult;
/**
 * Options for useTrackOnMount
 */
export interface TrackOnMountOptions {
    /**
     * Event name to track
     */
    eventName: string;
    /**
     * Event properties
     */
    properties?: Record<string, unknown>;
    /**
     * Only track once (default: true)
     */
    once?: boolean;
}
/**
 * Hook to track an event when a component mounts
 *
 * @example
 * ```tsx
 * function ProductPage({ productId }: { productId: string }) {
 *   useTrackOnMount({
 *     eventName: 'product_viewed',
 *     properties: { productId }
 *   });
 *
 *   return <div>Product {productId}</div>;
 * }
 * ```
 */
export declare function useTrackOnMount(options: TrackOnMountOptions): void;
/**
 * Higher-order component for tracking page views
 *
 * @example
 * ```tsx
 * const TrackedHomePage = withPageTracking(HomePage, 'home');
 * ```
 */
export declare function withPageTracking<P extends object>(Component: React.ComponentType<P>, pageName?: string): React.ComponentType<P>;
//# sourceMappingURL=analytics.d.ts.map