/**
 * npm registry availability checking utilities.
 * Used to verify a version is available via bun's resolver before attempting upgrade.
 */
/**
 * Run a command via Bun.spawn with a timeout that kills the process.
 * Returns { exitCode, stdout, stderr } similar to Bun's $ shell result.
 */
export declare function spawnWithTimeout(cmd: string[], options: {
    cwd?: string;
    timeout: number;
}): Promise<{
    exitCode: number;
    stdout: Buffer;
    stderr: Buffer;
}>;
/**
 * Check if a specific version of @agentuity/cli is resolvable by bun.
 * Uses `bun info` to verify bun's own resolver/CDN can see the version,
 * which avoids the race where npm registry returns 200 but bun's CDN
 * has not yet propagated the version.
 *
 * @param version - Version to check (with or without 'v' prefix)
 * @returns true if version is available, false otherwise
 */
export declare function isVersionAvailableOnNpm(version: string): Promise<boolean>;
/**
 * Quick check if a version is available via bun's resolver.
 * Used for implicit version checks (auto-upgrade flow).
 *
 * @param version - Version to check (with or without 'v' prefix)
 * @returns true if version is available, false if unavailable or error
 */
export declare function isVersionAvailableOnNpmQuick(version: string): Promise<boolean>;
export interface WaitForNpmOptions {
    /** Maximum number of attempts (default: 6) */
    maxAttempts?: number;
    /** Initial delay between attempts in ms (default: 2000) */
    initialDelayMs?: number;
    /** Maximum delay between attempts in ms (default: 10000) */
    maxDelayMs?: number;
    /** Callback called before each retry */
    onRetry?: (attempt: number, delayMs: number) => void;
}
/**
 * Wait for a version to become available on npm with exponential backoff.
 *
 * @param version - Version to wait for (with or without 'v' prefix)
 * @param options - Configuration options
 * @returns true if version became available, false if timed out
 */
export declare function waitForNpmAvailability(version: string, options?: WaitForNpmOptions): Promise<boolean>;
/**
 * Check whether a bun install failure is a transient resolution error
 * caused by npm CDN propagation delays.
 */
export declare function isResolutionError(stderr: string): boolean;
export interface InstallWithRetryOptions {
    /** Maximum number of attempts including the first (default: 7 → 1 initial + 6 retries) */
    maxAttempts?: number;
    /** Initial delay in ms before the first retry (default: 5000) */
    initialDelayMs?: number;
    /** Maximum delay cap in ms (default: 30000) */
    maxDelayMs?: number;
    /** Multiplier applied to the delay after each retry (default: 2) */
    multiplier?: number;
    /** Callback invoked before each retry with the attempt number and upcoming delay */
    onRetry?: (attempt: number, delayMs: number) => void;
}
/**
 * Run an install function and retry on transient resolution errors with
 * exponential backoff. This covers the window (~2 min) where npm CDN nodes
 * have not yet propagated a newly-published version.
 *
 * Total wait with defaults: 5 + 10 + 20 + 30 + 30 + 30 = 125 s ≈ 2 min
 *
 * @param installFn - Async function that performs the install and returns exitCode + stderr
 * @param options - Retry configuration
 * @returns The successful result (exitCode 0)
 * @throws Error if all retries are exhausted or a non-resolution error occurs
 */
export declare function installWithRetry(installFn: () => Promise<{
    exitCode: number;
    stderr: Buffer;
}>, options?: InstallWithRetryOptions): Promise<{
    exitCode: number;
    stderr: Buffer;
}>;
//# sourceMappingURL=npm-availability.d.ts.map