import { agents } from '../agents';
import { ConcurrencyManager } from './concurrency';
const DEFAULT_BACKGROUND_CONFIG = {
    enabled: true,
    defaultConcurrency: 1,
    staleTimeoutMs: 30 * 60 * 1000,
};
export class BackgroundManager {
    ctx;
    config;
    concurrency;
    callbacks;
    dbReader;
    tasks = new Map();
    tasksByParent = new Map();
    tasksBySession = new Map();
    notifications = new Map();
    toolCallIds = new Map();
    shuttingDown = false;
    constructor(ctx, config, callbacks, dbReader) {
        this.ctx = ctx;
        this.config = { ...DEFAULT_BACKGROUND_CONFIG, ...config };
        this.concurrency = new ConcurrencyManager({
            defaultLimit: this.config.defaultConcurrency,
            limits: buildConcurrencyLimits(this.config),
        });
        this.callbacks = callbacks;
        this.dbReader = dbReader;
    }
    async launch(input) {
        const task = {
            id: createTaskId(),
            parentSessionId: input.parentSessionId,
            parentMessageId: input.parentMessageId,
            description: input.description,
            prompt: input.prompt,
            agent: input.agent,
            status: 'pending',
            queuedAt: new Date(),
            concurrencyGroup: this.getConcurrencyGroup(input.agent),
        };
        this.tasks.set(task.id, task);
        this.indexTask(task);
        if (!this.config.enabled) {
            task.status = 'error';
            task.error = 'Background tasks are disabled.';
            task.completedAt = new Date();
            this.markForNotification(task);
            return task;
        }
        void this.startTask(task);
        return task;
    }
    getTask(id) {
        return this.tasks.get(id);
    }
    getTasksByParent(sessionId) {
        const ids = this.tasksByParent.get(sessionId);
        if (!ids)
            return [];
        return Array.from(ids)
            .map((id) => this.tasks.get(id))
            .filter((task) => Boolean(task));
    }
    findBySession(sessionId) {
        const taskId = this.tasksBySession.get(sessionId);
        if (!taskId)
            return undefined;
        return this.tasks.get(taskId);
    }
    /**
     * Inspect a background task by fetching its session messages.
     * Useful for seeing what a child Lead or other agent is doing.
     */
    async inspectTask(taskId) {
        const task = this.tasks.get(taskId);
        if (!task?.sessionId)
            return undefined;
        try {
            if (this.dbReader?.isAvailable()) {
                const session = this.dbReader.getSession(task.sessionId);
                const messageCount = this.dbReader.getMessageCount(task.sessionId);
                const messageLimit = messageCount > 0 ? messageCount : 100;
                const messageRows = this.dbReader.getMessages(task.sessionId, {
                    limit: messageLimit,
                    offset: 0,
                });
                const textParts = this.dbReader.getTextParts(task.sessionId, {
                    limit: Math.max(messageLimit * 5, 200),
                });
                const partsByMessage = groupTextPartsByMessage(textParts);
                const messages = messageRows
                    .sort((a, b) => a.timeCreated - b.timeCreated)
                    .map((message) => ({
                    info: {
                        role: message.role,
                        agent: message.agent,
                        model: message.model,
                        cost: message.cost,
                        tokens: message.tokens,
                        error: message.error,
                        timeCreated: message.timeCreated,
                        timeUpdated: message.timeUpdated,
                    },
                    parts: buildTextParts(partsByMessage.get(message.id)),
                }));
                const activeTools = this.dbReader.getActiveToolCalls(task.sessionId).map((tool) => ({
                    tool: tool.tool,
                    status: tool.status,
                    callId: tool.callId,
                }));
                const todos = this.dbReader.getTodos(task.sessionId).map((todo) => ({
                    content: todo.content,
                    status: todo.status,
                    priority: todo.priority,
                }));
                const cost = this.dbReader.getSessionCost(task.sessionId);
                const childSessionCount = this.dbReader.getChildSessions(task.sessionId).length;
                return {
                    taskId: task.id,
                    sessionId: task.sessionId,
                    status: task.status,
                    session,
                    messages,
                    lastActivity: task.progress?.lastUpdate?.toISOString(),
                    messageCount,
                    activeTools,
                    todos,
                    costSummary: {
                        totalCost: cost.totalCost,
                        totalTokens: cost.totalTokens,
                    },
                    childSessionCount,
                };
            }
            // Get session details
            const sessionResponse = await this.ctx.client.session.get({
                path: { id: task.sessionId },
                throwOnError: false,
            });
            // Get messages from the session
            const messagesResponse = await this.ctx.client.session.messages({
                path: { id: task.sessionId },
                throwOnError: false,
            });
            const session = unwrapResponse(sessionResponse);
            const rawMessages = unwrapResponse(messagesResponse);
            // Defensive array coercion (response may be non-array when throwOnError is false)
            const messages = Array.isArray(rawMessages) ? rawMessages : [];
            // Return structured inspection result
            return {
                taskId: task.id,
                sessionId: task.sessionId,
                status: task.status,
                session,
                messages,
                lastActivity: task.progress?.lastUpdate?.toISOString(),
            };
        }
        catch {
            // Session might not exist anymore
            return undefined;
        }
    }
    /**
     * Refresh task statuses from the server.
     * Useful for recovering state after issues or checking on stuck tasks.
     */
    async refreshStatuses() {
        const results = new Map();
        // Get all our tracked session IDs
        const sessionIds = Array.from(this.tasksBySession.keys());
        if (sessionIds.length === 0)
            return results;
        try {
            // Fetch children for each unique parent (more efficient than individual gets)
            const parentIds = new Set();
            for (const task of this.tasks.values()) {
                if (task.parentSessionId) {
                    parentIds.add(task.parentSessionId);
                }
            }
            const completionPromises = [];
            for (const parentId of parentIds) {
                const childrenResponse = await this.ctx.client.session.children({
                    path: { id: parentId },
                    throwOnError: false,
                });
                const children = unwrapResponse(childrenResponse) ?? [];
                for (const child of children) {
                    const childSession = child;
                    if (!childSession.id)
                        continue;
                    const matchedTaskId = this.tasksBySession.get(childSession.id);
                    if (matchedTaskId) {
                        const task = this.tasks.get(matchedTaskId);
                        if (task) {
                            const newStatus = this.mapSessionStatusToTaskStatus(childSession);
                            if (newStatus !== task.status) {
                                // Use proper handlers to trigger side effects (concurrency, notifications, etc.)
                                if (newStatus === 'completed' && task.status === 'running') {
                                    completionPromises.push(this.completeTask(task));
                                    results.set(matchedTaskId, newStatus);
                                }
                                else if (newStatus === 'error') {
                                    this.failTask(task, 'Session ended with error');
                                    results.set(matchedTaskId, newStatus);
                                }
                                else {
                                    // For other transitions (e.g., pending -> running), direct update is fine
                                    task.status = newStatus;
                                    results.set(matchedTaskId, newStatus);
                                }
                            }
                        }
                    }
                }
            }
            // Wait for all completion handlers to finish
            await Promise.all(completionPromises);
        }
        catch (error) {
            // Log but don't fail - this is a best-effort refresh
            console.error('Failed to refresh task statuses:', error);
        }
        return results;
    }
    /**
     * Recover background tasks from existing sessions.
     * Call this on plugin startup to restore state after restart.
     *
     * This method queries all sessions and reconstructs task state from
     * sessions that have JSON-encoded task metadata in their title.
     *
     * @returns The number of tasks recovered
     */
    async recoverTasks() {
        let recovered = 0;
        try {
            if (this.dbReader?.isAvailable()) {
                const parentSessionId = process.env.AGENTUITY_OPENCODE_SESSION;
                if (parentSessionId) {
                    const sessions = this.dbReader.getChildSessions(parentSessionId);
                    for (const sess of sessions) {
                        if (!sess.title?.startsWith('{'))
                            continue;
                        try {
                            const metadata = JSON.parse(sess.title);
                            if (!metadata.taskId || !metadata.taskId.startsWith('bg_'))
                                continue;
                            if (this.tasks.has(metadata.taskId))
                                continue;
                            const agentName = metadata.agent ?? 'unknown';
                            const task = {
                                id: metadata.taskId,
                                sessionId: sess.id,
                                parentSessionId: sess.parentId ?? '',
                                agent: agentName,
                                description: metadata.description ?? '',
                                prompt: '',
                                status: this.mapDbStatusToTaskStatus(sess.id),
                                queuedAt: metadata.createdAt ? new Date(metadata.createdAt) : new Date(),
                                startedAt: metadata.createdAt ? new Date(metadata.createdAt) : new Date(),
                                concurrencyGroup: this.getConcurrencyGroup(agentName),
                                progress: {
                                    toolCalls: 0,
                                    lastUpdate: new Date(),
                                },
                            };
                            this.tasks.set(task.id, task);
                            this.tasksBySession.set(sess.id, task.id);
                            if (task.parentSessionId) {
                                const parentTasks = this.tasksByParent.get(task.parentSessionId) ?? new Set();
                                parentTasks.add(task.id);
                                this.tasksByParent.set(task.parentSessionId, parentTasks);
                            }
                            recovered++;
                        }
                        catch {
                            continue;
                        }
                    }
                    return recovered;
                }
            }
            // Get all sessions
            const sessionsResponse = await this.ctx.client.session.list({
                throwOnError: false,
            });
            const sessions = unwrapResponse(sessionsResponse) ?? [];
            for (const session of sessions) {
                const sess = session;
                // Check if this is one of our background task sessions
                // Our sessions have JSON-encoded task metadata in the title
                if (!sess.title?.startsWith('{'))
                    continue;
                try {
                    const metadata = JSON.parse(sess.title);
                    // Skip if not a valid task metadata (must have taskId starting with 'bg_')
                    if (!metadata.taskId || !metadata.taskId.startsWith('bg_'))
                        continue;
                    // Skip if we already have this task
                    if (this.tasks.has(metadata.taskId))
                        continue;
                    // Skip sessions without an ID
                    if (!sess.id)
                        continue;
                    // Reconstruct the task
                    const agentName = metadata.agent ?? 'unknown';
                    const task = {
                        id: metadata.taskId,
                        sessionId: sess.id,
                        parentSessionId: sess.parentID ?? '',
                        agent: agentName,
                        description: metadata.description ?? '',
                        prompt: '', // Original prompt not stored in metadata
                        status: this.mapSessionStatusToTaskStatus(sess),
                        queuedAt: metadata.createdAt ? new Date(metadata.createdAt) : new Date(),
                        startedAt: metadata.createdAt ? new Date(metadata.createdAt) : new Date(),
                        concurrencyGroup: this.getConcurrencyGroup(agentName),
                        progress: {
                            toolCalls: 0,
                            lastUpdate: new Date(),
                        },
                    };
                    // Add to our tracking maps
                    this.tasks.set(task.id, task);
                    this.tasksBySession.set(sess.id, task.id);
                    if (task.parentSessionId) {
                        const parentTasks = this.tasksByParent.get(task.parentSessionId) ?? new Set();
                        parentTasks.add(task.id);
                        this.tasksByParent.set(task.parentSessionId, parentTasks);
                    }
                    recovered++;
                }
                catch {
                    // Not valid JSON or not our task, skip
                    continue;
                }
            }
        }
        catch (error) {
            console.error('Failed to recover tasks:', error);
        }
        return recovered;
    }
    mapSessionStatusToTaskStatus(session) {
        // Map OpenCode session status to our task status
        // Session status types: 'idle' | 'pending' | 'running' | 'error'
        const status = session?.status?.type;
        switch (status) {
            case 'idle':
                return 'completed';
            case 'pending':
                return 'pending';
            case 'running':
                return 'running';
            case 'error':
                return 'error';
            default:
                // Unknown session status - default to pending for best-effort recovery
                return 'pending';
        }
    }
    cancel(taskId) {
        const task = this.tasks.get(taskId);
        if (!task || task.status === 'completed' || task.status === 'error') {
            return false;
        }
        task.status = 'cancelled';
        task.completedAt = new Date();
        this.releaseConcurrency(task);
        this.markForNotification(task);
        if (task.sessionId) {
            void this.abortSession(task.sessionId);
            this.callbacks?.onSubagentSessionDeleted?.({ sessionId: task.sessionId });
        }
        return true;
    }
    handleEvent(event) {
        if (!event || typeof event.type !== 'string')
            return;
        this.expireStaleTasks();
        if (event.type === 'message.part.updated') {
            const part = event.properties?.part;
            if (!part)
                return;
            const sessionId = part.sessionID;
            if (!sessionId)
                return;
            const task = this.findBySession(sessionId);
            if (!task)
                return;
            this.updateProgress(task, part);
            return;
        }
        if (event.type === 'session.idle') {
            const sessionId = extractSessionId(event.properties);
            const task = sessionId ? this.findBySession(sessionId) : undefined;
            if (!task)
                return;
            void this.completeTask(task);
            return;
        }
        if (event.type === 'session.error') {
            const sessionId = extractSessionId(event.properties);
            const task = sessionId ? this.findBySession(sessionId) : undefined;
            if (!task)
                return;
            const error = extractError(event.properties);
            this.failTask(task, error ?? 'Session error.');
            return;
        }
    }
    markForNotification(task) {
        const sessionId = task.parentSessionId;
        if (!sessionId)
            return;
        const queue = this.notifications.get(sessionId) ?? new Set();
        queue.add(task.id);
        this.notifications.set(sessionId, queue);
    }
    getPendingNotifications(sessionId) {
        const queue = this.notifications.get(sessionId);
        if (!queue)
            return [];
        return Array.from(queue)
            .map((id) => this.tasks.get(id))
            .filter((task) => Boolean(task));
    }
    clearNotifications(sessionId) {
        this.notifications.delete(sessionId);
    }
    shutdown() {
        this.shuttingDown = true;
        this.concurrency.clear();
        this.notifications.clear();
        try {
            void this.callbacks?.onShutdown?.();
        }
        catch {
            // Ignore shutdown callback errors
        }
    }
    indexTask(task) {
        const parentList = this.tasksByParent.get(task.parentSessionId) ?? new Set();
        parentList.add(task.id);
        this.tasksByParent.set(task.parentSessionId, parentList);
    }
    async startTask(task) {
        if (this.shuttingDown)
            return;
        const concurrencyKey = this.getConcurrencyKey(task.agent);
        task.concurrencyKey = concurrencyKey;
        try {
            await this.concurrency.acquire(concurrencyKey);
        }
        catch (error) {
            if (task.status !== 'cancelled') {
                task.status = 'error';
                task.error = error instanceof Error ? error.message : 'Failed to acquire slot.';
                task.completedAt = new Date();
                this.markForNotification(task);
            }
            return;
        }
        if (task.status === 'cancelled') {
            this.releaseConcurrency(task);
            return;
        }
        try {
            // Store task metadata in session title for persistence/recovery
            const taskMetadata = JSON.stringify({
                taskId: task.id,
                agent: task.agent,
                description: task.description,
                createdAt: task.queuedAt?.toISOString() ?? new Date().toISOString(),
            });
            const sessionResult = await this.ctx.client.session.create({
                body: {
                    parentID: task.parentSessionId,
                    title: taskMetadata,
                },
                throwOnError: true,
            });
            const session = unwrapResponse(sessionResult);
            if (!session?.id) {
                throw new Error('Failed to create session.');
            }
            task.sessionId = session.id;
            task.status = 'running';
            task.startedAt = new Date();
            this.tasksBySession.set(session.id, task.id);
            this.callbacks?.onSubagentSessionCreated?.({
                sessionId: session.id,
                parentId: task.parentSessionId,
                title: task.description,
            });
            await this.ctx.client.session.prompt({
                path: { id: session.id },
                body: {
                    agent: task.agent,
                    parts: [{ type: 'text', text: task.prompt }],
                },
                throwOnError: true,
            });
        }
        catch (error) {
            this.failTask(task, error instanceof Error ? error.message : 'Failed to launch background task.');
        }
    }
    updateProgress(task, part) {
        const progress = task.progress ?? this.createProgress();
        progress.lastUpdate = new Date();
        if (part.type === 'tool') {
            const callId = part.callID;
            const toolName = part.tool;
            if (toolName) {
                progress.lastTool = toolName;
            }
            if (callId) {
                const seen = this.toolCallIds.get(task.id) ?? new Set();
                if (!seen.has(callId)) {
                    seen.add(callId);
                    progress.toolCalls += 1;
                    this.toolCallIds.set(task.id, seen);
                }
            }
        }
        if (part.type === 'text' && part.text) {
            progress.lastMessage = part.text;
            progress.lastMessageAt = new Date();
        }
        task.progress = progress;
    }
    createProgress() {
        return {
            toolCalls: 0,
            lastUpdate: new Date(),
        };
    }
    async completeTask(task) {
        if (task.status !== 'running')
            return;
        task.status = 'completed';
        task.completedAt = new Date();
        this.releaseConcurrency(task);
        if (task.sessionId) {
            const result = await this.fetchLatestResult(task.sessionId);
            if (result) {
                task.result = result;
            }
            this.callbacks?.onSubagentSessionDeleted?.({ sessionId: task.sessionId });
        }
        this.markForNotification(task);
        void this.notifyParent(task);
    }
    failTask(task, error) {
        if (task.status === 'completed' || task.status === 'error')
            return;
        task.status = 'error';
        task.error = error;
        task.completedAt = new Date();
        this.releaseConcurrency(task);
        if (task.sessionId) {
            this.callbacks?.onSubagentSessionDeleted?.({ sessionId: task.sessionId });
        }
        this.markForNotification(task);
        void this.notifyParent(task);
    }
    releaseConcurrency(task) {
        if (!task.concurrencyKey)
            return;
        this.concurrency.release(task.concurrencyKey);
        delete task.concurrencyKey;
    }
    async notifyParent(task) {
        if (!task.parentSessionId)
            return;
        // Prevent duplicate notifications for the same task+status combination
        // This guards against OpenCode firing multiple events for the same status transition
        const notifiedStatuses = task.notifiedStatuses ?? new Set();
        // Self-healing for tasks created before deduplication was added:
        // If a task is already in a terminal state but has no notification history,
        // assume it was already notified and skip to prevent duplicate notifications.
        if (notifiedStatuses.size === 0 &&
            (task.status === 'completed' || task.status === 'error' || task.status === 'cancelled')) {
            notifiedStatuses.add(task.status);
            task.notifiedStatuses = notifiedStatuses;
            return;
        }
        if (notifiedStatuses.has(task.status)) {
            return; // Already notified for this status, skip duplicate
        }
        // Mark as notified BEFORE sending to prevent race conditions
        notifiedStatuses.add(task.status);
        task.notifiedStatuses = notifiedStatuses;
        const statusLine = task.status === 'completed' ? 'completed' : task.status;
        const message = `[BACKGROUND TASK ${statusLine.toUpperCase()}]

Task: ${task.description}
Agent: ${task.agent}
Status: ${task.status}
Task ID: ${task.id}

Use the agentuity_background_output tool with task_id "${task.id}" to view the result.`;
        try {
            await this.ctx.client.session.prompt({
                path: { id: task.parentSessionId },
                body: {
                    parts: [{ type: 'text', text: message }],
                },
                throwOnError: true,
                responseStyle: 'data',
            });
        }
        catch {
            // Ignore notification errors
        }
    }
    async abortSession(sessionId) {
        try {
            await this.ctx.client.session.abort({
                path: { id: sessionId },
                throwOnError: false,
            });
        }
        catch {
            // Ignore abort errors
        }
    }
    async fetchLatestResult(sessionId) {
        try {
            if (this.dbReader?.isAvailable()) {
                const messages = this.dbReader.getMessages(sessionId, { limit: 100, offset: 0 });
                const textParts = this.dbReader.getTextParts(sessionId, { limit: 300 });
                const partsByMessage = groupTextPartsByMessage(textParts);
                for (const message of messages) {
                    if (message.role !== 'assistant')
                        continue;
                    const text = joinTextParts(partsByMessage.get(message.id));
                    if (text)
                        return text;
                }
                return undefined;
            }
            const messagesResult = await this.ctx.client.session.messages({
                path: { id: sessionId },
                throwOnError: true,
            });
            const messages = unwrapResponse(messagesResult) ?? [];
            const entries = Array.isArray(messages) ? messages : [];
            for (let i = entries.length - 1; i >= 0; i -= 1) {
                const entry = entries[i];
                if (entry?.info?.role !== 'assistant')
                    continue;
                const text = extractTextFromParts(entry.parts ?? []);
                if (text)
                    return text;
            }
        }
        catch {
            return undefined;
        }
        return undefined;
    }
    mapDbStatusToTaskStatus(sessionId) {
        if (!this.dbReader)
            return 'pending';
        const status = this.dbReader.getSessionStatus(sessionId).status;
        switch (status) {
            case 'idle':
            case 'archived':
                return 'completed';
            case 'active':
            case 'compacting':
                return 'running';
            case 'error':
                return 'error';
            default:
                return 'pending';
        }
    }
    getConcurrencyGroup(agentName) {
        const model = getAgentModel(agentName);
        if (!model)
            return undefined;
        const provider = model.split('/')[0];
        if (model && this.config.modelConcurrency?.[model] !== undefined) {
            return `model:${model}`;
        }
        if (provider && this.config.providerConcurrency?.[provider] !== undefined) {
            return `provider:${provider}`;
        }
        return undefined;
    }
    getConcurrencyKey(agentName) {
        const group = this.getConcurrencyGroup(agentName);
        return group ?? 'default';
    }
    expireStaleTasks() {
        const now = Date.now();
        for (const task of this.tasks.values()) {
            if (task.status !== 'pending' && task.status !== 'running')
                continue;
            const start = task.startedAt?.getTime() ?? task.queuedAt?.getTime();
            if (!start)
                continue;
            if (now - start > this.config.staleTimeoutMs) {
                this.failTask(task, 'Background task timed out.');
            }
        }
    }
}
function buildConcurrencyLimits(config) {
    const limits = {};
    if (config.providerConcurrency) {
        for (const [provider, limit] of Object.entries(config.providerConcurrency)) {
            limits[`provider:${provider}`] = limit;
        }
    }
    if (config.modelConcurrency) {
        for (const [model, limit] of Object.entries(config.modelConcurrency)) {
            limits[`model:${model}`] = limit;
        }
    }
    return limits;
}
function getAgentModel(agentName) {
    const agent = findAgentDefinition(agentName);
    return agent?.defaultModel;
}
function findAgentDefinition(agentName) {
    return Object.values(agents).find((agent) => agent.displayName === agentName || agent.id === agentName || agent.role === agentName);
}
function createTaskId() {
    return `bg_${Math.random().toString(36).slice(2, 8)}`;
}
function extractSessionId(properties) {
    return (properties?.sessionId ?? properties?.sessionID);
}
function extractError(properties) {
    const error = properties?.error;
    return error?.data?.message ?? (typeof error?.name === 'string' ? error.name : undefined);
}
function extractTextFromParts(parts) {
    const textParts = [];
    for (const part of parts) {
        if (typeof part !== 'object' || part === null)
            continue;
        const typed = part;
        if (typed.type === 'text' && typeof typed.text === 'string') {
            textParts.push(typed.text);
        }
    }
    if (textParts.length === 0)
        return undefined;
    return textParts.join('\n');
}
function groupTextPartsByMessage(parts) {
    const grouped = new Map();
    for (const part of parts) {
        const list = grouped.get(part.messageId) ?? [];
        list.push(part);
        grouped.set(part.messageId, list);
    }
    for (const list of grouped.values()) {
        list.sort((a, b) => a.timeCreated - b.timeCreated);
    }
    return grouped;
}
function buildTextParts(parts) {
    if (!parts || parts.length === 0)
        return [];
    return parts.map((part) => ({ type: 'text', text: part.text }));
}
function joinTextParts(parts) {
    if (!parts || parts.length === 0)
        return undefined;
    return parts.map((part) => part.text).join('\n');
}
function unwrapResponse(result) {
    if (typeof result === 'object' && result !== null && 'data' in result) {
        return result.data;
    }
    return result;
}
//# sourceMappingURL=manager.js.map