import { FetchAdapter } from './adapter';
import type { SortDirection } from './pagination';
/**
 * Minimum TTL value in seconds (1 minute)
 */
export declare const KV_MIN_TTL_SECONDS = 60;
/**
 * Maximum TTL value in seconds (90 days)
 */
export declare const KV_MAX_TTL_SECONDS = 7776000;
/**
 * Default TTL value in seconds (7 days) - used when namespace is auto-created or no TTL specified
 */
export declare const KV_DEFAULT_TTL_SECONDS = 604800;
/**
 * the result of a data operation when the data is found
 */
export interface DataResultFound<T> {
    /**
     * the data from the result of the operation
     */
    data: T;
    /**
     * the content type of the data
     */
    contentType: string;
    /**
     * the data was found
     */
    exists: true;
    /**
     * the expiration time of the data as an ISO 8601 timestamp.
     * undefined if the key does not expire.
     */
    expiresAt?: string;
}
/**
 * the result of a data operation when the data is not found
 */
export interface DataResultNotFound {
    data: never;
    /**
     * the data was not found
     */
    exists: false;
}
/**
 * the result of a data operation
 */
export type DataResult<T> = DataResultFound<T> | DataResultNotFound;
export interface KeyValueStorageSetParams {
    /**
     * Time-to-live in seconds for the key. Controls when the key expires and is automatically deleted.
     * - `undefined` (not provided): Key inherits the namespace's default TTL (7 days if not configured)
     * - `null` or `0`: Key never expires
     * - positive number (≥60): Key expires after the specified number of seconds (max 90 days)
     *
     * @remarks
     * TTL values below 60 seconds are clamped to 60 seconds by the server.
     * TTL values above 7,776,000 seconds (90 days) are clamped to 90 days.
     */
    ttl?: number | null;
    /**
     * the content type of the value
     */
    contentType?: string;
}
/**
 * Parameters for creating a namespace
 */
export interface CreateNamespaceParams {
    /**
     * Default TTL for keys in this namespace (in seconds).
     * - If undefined/omitted: uses server default (7 days / 604,800 seconds)
     * - If 0: keys will not expire by default
     * - If 60-7,776,000: custom TTL in seconds (1 minute to 90 days)
     *
     * Keys can override this default by specifying TTL in the set() call.
     * Active keys are automatically extended (sliding expiration) when read
     * if their remaining TTL is less than 50% of the original TTL.
     */
    defaultTTLSeconds?: number;
}
/**
 * Statistics for a key-value store namespace
 */
export interface KeyValueStats {
    sum: number;
    count: number;
    createdAt?: number;
    lastUsedAt?: number;
}
/**
 * Item with metadata from search results
 */
export interface KeyValueItemWithMetadata<T = unknown> {
    value: T;
    contentType: string;
    size: number;
    created_at: string;
    updated_at: string;
}
export type KVSortField = 'name' | 'size' | 'records' | 'created' | 'lastUsed';
/**
 * Parameters for getting all namespace statistics with optional pagination
 */
export interface GetAllStatsParams {
    /**
     * Filter namespaces by name substring
     */
    name?: string;
    /**
     * Maximum number of namespaces to return (default: 100, max: 1000)
     */
    limit?: number;
    /**
     * Number of namespaces to skip for pagination (default: 0)
     */
    offset?: number;
    /**
     * Field to sort by
     */
    sort?: KVSortField;
    /**
     * Sort direction (default: 'desc')
     */
    direction?: SortDirection;
    /**
     * Filter by project ID
     */
    projectId?: string;
    /**
     * Filter by agent ID
     */
    agentId?: string;
    /**
     * Filter by project name
     */
    projectName?: string;
    /**
     * Filter by agent name
     */
    agentName?: string;
}
/**
 * Paginated response for namespace statistics
 */
export interface KeyValueStatsPaginated {
    /**
     * Map of namespace names to their statistics
     */
    namespaces: Record<string, KeyValueStats>;
    /**
     * Total number of namespaces across all pages
     */
    total: number;
    /**
     * Number of namespaces requested per page
     */
    limit: number;
    /**
     * Number of namespaces skipped
     */
    offset: number;
    /**
     * Whether there are more namespaces available
     */
    hasMore: boolean;
}
export interface KeyValueStorage {
    /**
     * get a value from the key value storage
     *
     * @param name - the name of the key value storage
     * @param key - the key to get the value of
     * @returns the DataResult object
     */
    get<T>(name: string, key: string): Promise<DataResult<T>>;
    /**
     * set a value in the key value storage
     *
     * @param name - the name of the key value storage
     * @param key - the key to set the value of
     * @param value - the value to set in any of the supported data types
     * @param params - the KeyValueStorageSetParams
     */
    set<T = unknown>(name: string, key: string, value: T, params?: KeyValueStorageSetParams): Promise<void>;
    /**
     * delete a value from the key value storage
     *
     * @param name - the name of the key value storage
     * @param key - the key to delete
     */
    delete(name: string, key: string): Promise<void>;
    /**
     * get statistics for a specific namespace
     *
     * @param name - the name of the key value storage
     * @returns statistics for the namespace
     */
    getStats(name: string): Promise<KeyValueStats>;
    /**
     * get statistics for all namespaces
     *
     * @param params - optional pagination parameters
     * @returns map of namespace names to statistics, or paginated response if params provided
     */
    getAllStats(params?: GetAllStatsParams): Promise<Record<string, KeyValueStats> | KeyValueStatsPaginated>;
    /**
     * get all namespace names
     *
     * @returns array of namespace names (up to 1000)
     *
     * @remarks
     * This method returns a maximum of 1000 namespace names.
     * If you have more than 1000 namespaces, only the first 1000
     * (ordered by creation date, most recent first) will be returned.
     */
    getNamespaces(): Promise<string[]>;
    /**
     * search for keys matching a keyword
     *
     * @param name - the name of the key value storage
     * @param keyword - the keyword to search for
     * @returns map of keys to items with metadata
     */
    search<T = unknown>(name: string, keyword: string): Promise<Record<string, KeyValueItemWithMetadata<T>>>;
    /**
     * get all keys in a namespace
     *
     * @param name - the name of the key value storage
     * @returns array of keys
     */
    getKeys(name: string): Promise<string[]>;
    /**
     * delete all keys in a namespace
     *
     * @param name - the name of the key value storage
     */
    deleteNamespace(name: string): Promise<void>;
    /**
     * create a new namespace
     *
     * @param name - the name of the key value storage to create
     * @param params - optional parameters including default TTL
     */
    createNamespace(name: string, params?: CreateNamespaceParams): Promise<void>;
}
export declare class KeyValueStorageService implements KeyValueStorage {
    #private;
    constructor(baseUrl: string, adapter: FetchAdapter);
    get<T>(name: string, key: string): Promise<DataResult<T>>;
    /**
     * set a value in the key value storage
     *
     * @param name - the name of the key value storage
     * @param key - the key to set the value of
     * @param value - the value to set in any of the supported data types
     * @param params - the KeyValueStorageSetParams
     *
     * @remarks
     * TTL behavior:
     * - If TTL is not specified (undefined), the key inherits the namespace's default TTL
     * - If TTL is null or 0, the key will not expire
     * - If TTL is a positive number, the key expires after that many seconds
     * - TTL values below 60 seconds are clamped to 60 seconds by the server
     * - TTL values above 7,776,000 seconds (90 days) are clamped to 90 days
     * - If the namespace doesn't exist, it is auto-created with a 7-day default TTL
     */
    set<T = unknown>(name: string, key: string, value: T, params?: KeyValueStorageSetParams): Promise<void>;
    delete(name: string, key: string): Promise<void>;
    getStats(name: string): Promise<KeyValueStats>;
    getAllStats(params?: GetAllStatsParams): Promise<Record<string, KeyValueStats> | KeyValueStatsPaginated>;
    getNamespaces(): Promise<string[]>;
    search<T = unknown>(name: string, keyword: string): Promise<Record<string, KeyValueItemWithMetadata<T>>>;
    getKeys(name: string): Promise<string[]>;
    deleteNamespace(name: string): Promise<void>;
    createNamespace(name: string, params?: CreateNamespaceParams): Promise<void>;
}
//# sourceMappingURL=keyvalue.d.ts.map