/**
 * Steps UI Component v2 - Clean state-driven implementation
 *
 * Key principle: Render entire step list from state on every update cycle.
 * Track total lines rendered to calculate cursor movement.
 */
import type { LogLevel } from './types';
/**
 * Error thrown when step execution is interrupted by a signal (e.g., Ctrl+C).
 * Callers can catch this to perform cleanup before exiting.
 */
export declare class StepInterruptError extends Error {
    readonly exitCode: number;
    constructor(exitCode?: number);
}
/**
 * Step outcome returned by step.run()
 */
export type StepOutcome = {
    status: 'success';
    output?: string[];
} | {
    status: 'skipped';
    reason?: string;
    output?: string[];
} | {
    status: 'error';
    message: string;
    cause?: Error;
    output?: string[];
};
/**
 * Helper functions for creating step outcomes
 */
export declare const stepSuccess: (output?: string[]) => StepOutcome;
export declare const stepSkipped: (reason?: string, output?: string[]) => StepOutcome;
export declare const stepError: (message: string, cause?: Error, output?: string[]) => StepOutcome;
/**
 * Context passed to step run function
 */
export interface StepContext {
    progress: (n: number) => void;
    signal: AbortSignal;
}
/**
 * Step definition
 */
export interface Step {
    label: string;
    run: (ctx: StepContext) => Promise<StepOutcome>;
}
export declare function isStepUIPaused(): boolean;
/**
 * Pause step rendering for interactive input
 * Returns resume function
 */
export declare function pauseStepUI(clear?: boolean): () => void;
/**
 * Run a series of steps with animated progress
 */
export declare function runSteps(steps: Step[], logLevel?: LogLevel): Promise<void>;
//# sourceMappingURL=steps.d.ts.map