import type { EventEmitter } from 'node:events';
interface WritableWithDrain extends EventEmitter {
    write(chunk: Uint8Array): boolean;
}
/**
 * Machine-readable error codes for sandbox operations.
 * These codes allow programmatic error handling without fragile string matching.
 */
export type SandboxErrorCode = 'SANDBOX_NOT_FOUND' | 'SANDBOX_TERMINATED' | 'SANDBOX_BUSY' | 'EXECUTION_NOT_FOUND' | 'EXECUTION_TIMEOUT' | 'EXECUTION_CANCELLED' | 'SNAPSHOT_NOT_FOUND';
/**
 * Error thrown when a sandbox API request fails.
 *
 * Includes optional context about which sandbox or execution caused the error.
 */
export declare const SandboxResponseError: {
    new (args?: ({
        /** The sandbox ID associated with the error, if applicable */
        sandboxId?: string;
        /** The execution ID associated with the error, if applicable */
        executionId?: string;
        /** The session ID (trace ID) from the x-session-id response header for OTel correlation */
        sessionId?: string | null;
        /** Machine-readable error code for programmatic error handling */
        code?: SandboxErrorCode;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "SandboxResponseError";
    } & Readonly<{
        /** The sandbox ID associated with the error, if applicable */
        sandboxId?: string;
        /** The execution ID associated with the error, if applicable */
        executionId?: string;
        /** The session ID (trace ID) from the x-session-id response header for OTel correlation */
        sessionId?: string | null;
        /** Machine-readable error code for programmatic error handling */
        code?: SandboxErrorCode;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a sandbox is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await sandboxGet(client, { sandboxId: 'non-existent' });
 * } catch (error) {
 *   if (error._tag === 'SandboxNotFoundError') {
 *     console.error(`Sandbox not found: ${error.sandboxId}`);
 *   }
 * }
 * ```
 */
export declare const SandboxNotFoundError: {
    new (args?: ({
        sandboxId: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "SandboxNotFoundError";
    } & Readonly<{
        sandboxId: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a sandbox has already terminated.
 *
 * @example
 * ```typescript
 * try {
 *   await sandboxExecute(client, { sandboxId: 'terminated-sandbox', command: ['ls'] });
 * } catch (error) {
 *   if (error._tag === 'SandboxTerminatedError') {
 *     console.error(`Sandbox terminated: ${error.sandboxId}`);
 *   }
 * }
 * ```
 */
export declare const SandboxTerminatedError: {
    new (args?: ({
        sandboxId: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "SandboxTerminatedError";
    } & Readonly<{
        sandboxId: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a sandbox is currently busy executing another command.
 *
 * This typically occurs when a second execute request is sent before the
 * previous execution has completed. Sandbox executions are serialized -
 * wait for the current execution to complete before sending a new one.
 *
 * @example
 * ```typescript
 * try {
 *   await sandbox.execute({ command: ['ls'] });
 * } catch (error) {
 *   if (error._tag === 'SandboxBusyError') {
 *     console.error('Sandbox is busy, waiting for current execution to finish');
 *     // Wait and retry, or use executionGet with long-polling to wait for completion
 *   }
 * }
 * ```
 */
export declare const SandboxBusyError: {
    new (args?: ({
        sandboxId?: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "SandboxBusyError";
    } & Readonly<{
        sandboxId?: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when an execution is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await executionGet(client, { executionId: 'non-existent' });
 * } catch (error) {
 *   if (error._tag === 'ExecutionNotFoundError') {
 *     console.error(`Execution not found: ${error.executionId}`);
 *   }
 * }
 * ```
 */
export declare const ExecutionNotFoundError: {
    new (args?: ({
        executionId: string;
        sandboxId?: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "ExecutionNotFoundError";
    } & Readonly<{
        executionId: string;
        sandboxId?: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when an execution times out.
 *
 * @example
 * ```typescript
 * try {
 *   await sandboxExecute(client, { sandboxId, command: ['long-running'], timeout: '30s' });
 * } catch (error) {
 *   if (error._tag === 'ExecutionTimeoutError') {
 *     console.error('Execution timed out');
 *   }
 * }
 * ```
 */
export declare const ExecutionTimeoutError: {
    new (args?: ({
        executionId?: string;
        sandboxId?: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "ExecutionTimeoutError";
    } & Readonly<{
        executionId?: string;
        sandboxId?: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when an execution is cancelled.
 *
 * @example
 * ```typescript
 * try {
 *   await sandboxRun(client, params, { signal: controller.signal });
 * } catch (error) {
 *   if (error._tag === 'ExecutionCancelledError') {
 *     console.error('Execution was cancelled');
 *   }
 * }
 * ```
 */
export declare const ExecutionCancelledError: {
    new (args?: ({
        sandboxId?: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "ExecutionCancelledError";
    } & Readonly<{
        sandboxId?: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a snapshot is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await snapshotGet(client, { snapshotId: 'non-existent' });
 * } catch (error) {
 *   if (error._tag === 'SnapshotNotFoundError') {
 *     console.error(`Snapshot not found: ${error.snapshotId}`);
 *   }
 * }
 * ```
 */
export declare const SnapshotNotFoundError: {
    new (args?: ({
        snapshotId?: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "SnapshotNotFoundError";
    } & Readonly<{
        snapshotId?: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Context for throwing sandbox errors.
 */
export interface SandboxErrorContext {
    sandboxId?: string;
    executionId?: string;
    sessionId?: string | null;
    snapshotId?: string;
}
/**
 * Throws the appropriate sandbox error based on the response code.
 *
 * This helper centralizes error mapping logic, throwing specific error types
 * when the backend returns a known error code, and falling back to
 * SandboxResponseError for unknown codes.
 *
 * @param resp - The API response containing message and optional code
 * @param context - Context about the operation (sandbox ID, execution ID, etc.)
 * @throws {SandboxNotFoundError} When code is 'SANDBOX_NOT_FOUND'
 * @throws {SandboxTerminatedError} When code is 'SANDBOX_TERMINATED'
 * @throws {SandboxBusyError} When code is 'SANDBOX_BUSY'
 * @throws {ExecutionNotFoundError} When code is 'EXECUTION_NOT_FOUND'
 * @throws {ExecutionTimeoutError} When code is 'EXECUTION_TIMEOUT'
 * @throws {ExecutionCancelledError} When code is 'EXECUTION_CANCELLED'
 * @throws {SnapshotNotFoundError} When code is 'SNAPSHOT_NOT_FOUND'
 * @throws {SandboxResponseError} For unknown codes or when no code is provided
 */
export declare function throwSandboxError(resp: {
    message?: string;
    code?: string;
}, context: SandboxErrorContext): never;
/** Current sandbox API version */
export declare const API_VERSION = "2025-03-17";
/**
 * Write a chunk to a writable stream and wait for it to drain if necessary.
 * Properly cleans up event listeners to avoid memory leaks.
 */
export declare function writeAndDrain(writable: WritableWithDrain, chunk: Uint8Array): Promise<void>;
export {};
//# sourceMappingURL=util.d.ts.map