import { type Context } from '@opentelemetry/api';
/**
 * Entries to set on the W3C tracestate header. Each key-value pair is added
 * to the parent context's existing traceState (if any). Values that are
 * `undefined` or empty strings are skipped.
 */
export type TraceStateEntries = Record<string, string | undefined>;
/**
 * Build a context whose span context carries an enriched W3C traceState.
 *
 * The returned context is intended to be passed as the **parent context**
 * to `tracer.startActiveSpan(name, opts, enrichedCtx, fn)` or
 * `tracer.startSpan(name, opts, enrichedCtx)`.  Because the OTel SDK
 * copies `traceState` from a *valid* parent span context into every new
 * child span, the recording span that gets exported to OTLP will carry the
 * enriched traceState — making it visible in backends like ClickHouse.
 *
 * ### How it works
 *
 * 1. If the supplied `parentContext` already contains a span with a valid
 *    span context (e.g. from an incoming `traceparent` header), we enrich
 *    that span context's traceState and wrap it in a `NonRecordingSpan`.
 *
 * 2. If there is **no** valid parent span (e.g. no incoming `traceparent`),
 *    we synthesise a minimal remote span context with a freshly generated
 *    traceId.  The OTel SDK will treat this as a valid remote parent,
 *    inherit both the traceId **and** the traceState, and mark the new
 *    span as a continuation of that trace.
 *
 * @param parentContext  The context to enrich (typically `context.active()`).
 * @param entries        Key-value pairs to merge into the traceState.
 * @returns A new `Context` ready to be used as a parent for span creation.
 */
export declare function enrichContextWithTraceState(parentContext: Context, entries: TraceStateEntries): Context;
/**
 * Generate a random 32-hex-char trace ID (16 bytes).
 * Uses the Web Crypto API which is available in Bun and Node 20+.
 */
export declare function generateTraceId(): string;
/**
 * Generate a random 16-hex-char span ID (8 bytes).
 */
export declare function generateSpanId(): string;
//# sourceMappingURL=tracestate.d.ts.map