import type { InferOutput } from '@agentuity/core';
import { type SSERouteRegistry } from '@agentuity/frontend';
/**
 * Extract SSE route keys (e.g., '/events', '/notifications')
 */
export type SSERouteKey = keyof SSERouteRegistry;
/**
 * Extract output type for an SSE route (SSE is typically one-way server->client)
 */
export type SSERouteOutput<TRoute extends SSERouteKey> = TRoute extends keyof SSERouteRegistry ? SSERouteRegistry[TRoute] extends {
    outputSchema: infer TSchema;
} ? TSchema extends undefined | never ? void : InferOutput<TSchema> : void : void;
/**
 * Options for EventStream hooks
 */
export interface EventStreamOptions {
    /**
     * Optional query parameters to append to the EventStream URL
     */
    query?: URLSearchParams;
    /**
     * Optional subpath to append to the EventStream path
     */
    subpath?: string;
    /**
     * Optional AbortSignal to cancel the EventStream connection
     */
    signal?: AbortSignal;
}
/**
 * Type-safe EventStream (SSE) hook for connecting to SSE routes.
 *
 * Provides automatic type inference for route outputs based on
 * the SSERouteRegistry generated from your routes.
 *
 * @template TRoute - SSE route key from SSERouteRegistry (e.g., '/events', '/notifications')
 * @template TOutput - Optional type override for SSE event data. When provided, this type
 *   is used instead of the inferred type from the route registry. This is useful for SSE
 *   routes where outputSchema is `never` in the generated types.
 *
 * @example Simple SSE connection
 * ```typescript
 * const { isConnected, data } = useEventStream('/events');
 *
 * // data is fully typed based on route output schema!
 * ```
 *
 * @example SSE with query parameters
 * ```typescript
 * const { isConnected, data } = useEventStream('/notifications', {
 *   query: new URLSearchParams({ userId: '123' })
 * });
 * ```
 *
 * @example SSE with custom output type (when registry has outputSchema: never)
 * ```typescript
 * interface StreamMessage {
 *   type: 'token' | 'complete';
 *   content?: string;
 * }
 *
 * const { isConnected, data } = useEventStream<'/api/search', StreamMessage>('/api/search');
 *
 * // data is typed as StreamMessage | undefined
 * if (data?.type === 'token') {
 *   console.log(data.content);
 * }
 * ```
 */
export declare function useEventStream<TRoute extends SSERouteKey, TOutput = SSERouteOutput<TRoute>>(route: TRoute, options?: EventStreamOptions): {
    isConnected: boolean;
    close: () => void;
    data?: TOutput;
    error: Error | null;
    isError: boolean;
    reset: () => void;
    readyState: number;
};
//# sourceMappingURL=eventstream.d.ts.map