/**
 * npm registry availability checking utilities.
 * Used to verify a version is available via bun's resolver before attempting upgrade.
 */

import { $ } from 'bun';
import { tmpdir } from 'node:os';

const PACKAGE_SPEC = '@agentuity/cli';

/**
 * Check if a specific version of @agentuity/cli is resolvable by bun.
 * Uses `bun info` to verify bun's own resolver/CDN can see the version,
 * which avoids the race where npm registry returns 200 but bun's CDN
 * has not yet propagated the version.
 *
 * @param version - Version to check (with or without 'v' prefix)
 * @returns true if version is available, false otherwise
 */
export async function isVersionAvailableOnNpm(version: string): Promise<boolean> {
	const normalizedVersion = version.replace(/^v/, '');
	try {
		const result = await $`bun info ${PACKAGE_SPEC}@${normalizedVersion} --json`
			.cwd(tmpdir())
			.quiet()
			.nothrow();
		if (result.exitCode !== 0) {
			return false;
		}
		const info = JSON.parse(result.stdout.toString());
		if (info.error) {
			return false;
		}
		return info.version === normalizedVersion;
	} catch {
		return false;
	}
}

/**
 * Quick check if a version is available via bun's resolver.
 * Used for implicit version checks (auto-upgrade flow).
 *
 * @param version - Version to check (with or without 'v' prefix)
 * @returns true if version is available, false if unavailable or error
 */
export async function isVersionAvailableOnNpmQuick(version: string): Promise<boolean> {
	return isVersionAvailableOnNpm(version);
}

export interface WaitForNpmOptions {
	/** Maximum number of attempts (default: 6) */
	maxAttempts?: number;
	/** Initial delay between attempts in ms (default: 2000) */
	initialDelayMs?: number;
	/** Maximum delay between attempts in ms (default: 10000) */
	maxDelayMs?: number;
	/** Callback called before each retry */
	onRetry?: (attempt: number, delayMs: number) => void;
}

/**
 * Wait for a version to become available on npm with exponential backoff.
 *
 * @param version - Version to wait for (with or without 'v' prefix)
 * @param options - Configuration options
 * @returns true if version became available, false if timed out
 */
export async function waitForNpmAvailability(
	version: string,
	options: WaitForNpmOptions = {}
): Promise<boolean> {
	const { maxAttempts = 6, initialDelayMs = 2000, maxDelayMs = 10000, onRetry } = options;

	// First check - no delay
	if (await isVersionAvailableOnNpm(version)) {
		return true;
	}

	// Retry with exponential backoff
	let delay = initialDelayMs;
	for (let attempt = 1; attempt < maxAttempts; attempt++) {
		onRetry?.(attempt, delay);
		await new Promise((resolve) => setTimeout(resolve, delay));

		if (await isVersionAvailableOnNpm(version)) {
			return true;
		}

		delay = Math.min(Math.round(delay * 1.5), maxDelayMs);
	}

	return false;
}

/**
 * Patterns in bun's stderr that indicate a resolution/CDN propagation failure
 * (as opposed to a permanent install error like permissions or disk space).
 */
const RESOLUTION_ERROR_PATTERNS = [/failed to resolve/i, /no version matching/i];

/**
 * Check whether a bun install failure is a transient resolution error
 * caused by npm CDN propagation delays.
 */
export function isResolutionError(stderr: string): boolean {
	return RESOLUTION_ERROR_PATTERNS.some((pattern) => pattern.test(stderr));
}

export interface InstallWithRetryOptions {
	/** Maximum number of attempts including the first (default: 7 → 1 initial + 6 retries) */
	maxAttempts?: number;
	/** Initial delay in ms before the first retry (default: 5000) */
	initialDelayMs?: number;
	/** Maximum delay cap in ms (default: 30000) */
	maxDelayMs?: number;
	/** Multiplier applied to the delay after each retry (default: 2) */
	multiplier?: number;
	/** Callback invoked before each retry with the attempt number and upcoming delay */
	onRetry?: (attempt: number, delayMs: number) => void;
}

/**
 * Run an install function and retry on transient resolution errors with
 * exponential backoff. This covers the window (~2 min) where npm CDN nodes
 * have not yet propagated a newly-published version.
 *
 * Total wait with defaults: 5 + 10 + 20 + 30 + 30 + 30 = 125 s ≈ 2 min
 *
 * @param installFn - Async function that performs the install and returns exitCode + stderr
 * @param options - Retry configuration
 * @returns The successful result (exitCode 0)
 * @throws Error if all retries are exhausted or a non-resolution error occurs
 */
export async function installWithRetry(
	installFn: () => Promise<{ exitCode: number; stderr: Buffer }>,
	options: InstallWithRetryOptions = {}
): Promise<{ exitCode: number; stderr: Buffer }> {
	const {
		maxAttempts = 7,
		initialDelayMs = 5000,
		maxDelayMs = 30000,
		multiplier = 2,
		onRetry,
	} = options;

	let delay = initialDelayMs;

	for (let attempt = 1; attempt <= maxAttempts; attempt++) {
		const result = await installFn();

		if (result.exitCode === 0) {
			return result;
		}

		const stderr = result.stderr.toString();

		// Only retry on resolution/propagation errors — bail immediately for anything else
		if (!isResolutionError(stderr)) {
			throw new Error(stderr);
		}

		// Last attempt exhausted — throw
		if (attempt === maxAttempts) {
			throw new Error(stderr);
		}

		onRetry?.(attempt, delay);
		await new Promise((resolve) => setTimeout(resolve, delay));
		delay = Math.min(Math.round(delay * multiplier), maxDelayMs);
	}

	// Unreachable, but satisfies TypeScript
	throw new Error('Install failed after retries');
}
