import { z } from 'zod';
import { APIResponseSchema } from '../api';
import { APIKeyResponseError } from './util';
export const APIKeySchema = z.object({
    id: z.string().describe('the API key id'),
    name: z.string().describe('the API key name'),
    orgId: z.string().describe('the organization id'),
    type: z.string().describe('the API key type'),
    expiresAt: z.string().datetime().nullable().describe('the expiration date'),
    lastUsedAt: z.string().datetime().nullable().optional().describe('the last used date'),
    createdAt: z.string().datetime().describe('the creation date'),
    project: z
        .object({
        id: z.string().describe('the project id'),
        name: z.string().describe('the project name'),
    })
        .nullable()
        .optional()
        .describe('the associated project'),
});
export const APIKeyListResponseArray = z.array(APIKeySchema);
const APIKeyListResponseSchema = APIResponseSchema(APIKeyListResponseArray);
/**
 * List all API keys
 *
 * @param client
 * @param request optional filters for orgId and projectId
 * @returns
 */
export async function apikeyList(client, request) {
    const params = new URLSearchParams();
    if (request?.orgId) {
        params.set('orgId', request.orgId);
    }
    if (request?.projectId) {
        params.set('projectId', request.projectId);
    }
    const queryString = params.toString();
    const endpoint = `/cli/apikey${queryString ? `?${queryString}` : ''}`;
    const resp = await client.get(endpoint, APIKeyListResponseSchema);
    if (resp.success) {
        return resp.data;
    }
    throw new APIKeyResponseError({ message: resp.message });
}
//# sourceMappingURL=list.js.map