import type { Schema } from '../base';
import { SCHEMA_KIND } from '../base';
/**
 * Schema for validating number values.
 * Rejects NaN values.
 *
 * @example
 * ```typescript
 * const schema = s.number();
 * const age = schema.parse(30); // 30
 * schema.parse('30'); // throws ValidationError
 * schema.parse(NaN); // throws ValidationError
 * ```
 */
export declare class NumberSchema implements Schema<number, number> {
    readonly [SCHEMA_KIND] = "NumberSchema";
    description?: string;
    private _finite;
    private _min?;
    private _max?;
    readonly '~standard': {
        version: 1;
        vendor: string;
        validate: (value: unknown) => import("@agentuity/core").StandardSchemaV1.FailureResult | import("@agentuity/core").StandardSchemaV1.SuccessResult<number>;
        types: {
            input: number;
            output: number;
        };
    };
    describe(description: string): this;
    /**
     * Require the number to be finite (not Infinity, -Infinity, or NaN).
     *
     * @example
     * ```typescript
     * const schema = s.number().finite();
     * schema.parse(123); // 123
     * schema.parse(Infinity); // throws ValidationError
     * schema.parse(-Infinity); // throws ValidationError
     * ```
     */
    finite(): NumberSchema;
    /**
     * Set minimum value (inclusive).
     *
     * @example
     * ```typescript
     * const schema = s.number().min(0);
     * schema.parse(5); // 5
     * schema.parse(-1); // throws ValidationError
     * ```
     */
    min(value: number): NumberSchema;
    /**
     * Set maximum value (inclusive).
     *
     * @example
     * ```typescript
     * const schema = s.number().max(100);
     * schema.parse(50); // 50
     * schema.parse(101); // throws ValidationError
     * ```
     */
    max(value: number): NumberSchema;
    optional(): import("..").OptionalSchema<this>;
    nullable(): import("..").NullableSchema<this>;
    private _clone;
    parse: (this: Schema<any, number>, value: unknown) => number;
    safeParse: (this: Schema<any, number>, value: unknown) => import("..").SafeParseResult<number>;
}
/**
 * Create a number schema.
 *
 * @example
 * ```typescript
 * const ageSchema = s.number().describe('User age');
 * const age = ageSchema.parse(30);
 *
 * const finiteSchema = s.number().finite();
 * finiteSchema.parse(123); // OK
 * finiteSchema.parse(Infinity); // throws
 *
 * const rangeSchema = s.number().min(0).max(100);
 * rangeSchema.parse(50); // OK
 * rangeSchema.parse(101); // throws
 * ```
 */
export declare function number(): NumberSchema;
//# sourceMappingURL=number.d.ts.map