import { success, createParseMethods, SCHEMA_KIND } from '../base';
/**
 * Schema for nullable values (T | null).
 * Accepts null or the wrapped schema's type.
 *
 * @template T - The wrapped schema type
 *
 * @example
 * ```typescript
 * const schema = s.nullable(s.string());
 * schema.parse('hello'); // 'hello'
 * schema.parse(null); // null
 * schema.parse(123); // throws ValidationError
 * ```
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any
export class NullableSchema {
    [SCHEMA_KIND] = 'NullableSchema';
    schema;
    description;
    '~standard' = {
        version: 1,
        vendor: 'agentuity',
        validate: (value) => {
            if (value === null) {
                return success(null);
            }
            return this.schema['~standard'].validate(value);
        },
        types: undefined,
    };
    // Type-safe parse methods for this instance
    parseMethods = createParseMethods();
    constructor(schema) {
        this.schema = schema;
    }
    describe(description) {
        this.description = description;
        return this;
    }
    optional() {
        // Import here to avoid circular dependency
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const { optional } = require('./optional.js');
        return optional(this);
    }
    nullable() {
        return this; // Already nullable
    }
    parse = this.parseMethods.parse;
    safeParse = this.parseMethods.safeParse;
}
/**
 * Make a schema nullable (T | null).
 *
 * @param schema - The schema to make nullable
 *
 * @example
 * ```typescript
 * const userSchema = s.object({
 *   name: s.string(),
 *   bio: s.nullable(s.string())
 * });
 * ```
 */
// eslint-disable-next-line @typescript-eslint/no-explicit-any
export function nullable(schema) {
    return new NullableSchema(schema);
}
//# sourceMappingURL=nullable.js.map