/**
 * @agentuity/postgres - Resilient PostgreSQL client with automatic reconnection
 *
 * This package provides a PostgreSQL client that wraps Bun's native SQL driver
 * and adds automatic reconnection with exponential backoff.
 *
 * @example
 * ```typescript
 * import { postgres } from '@agentuity/postgres';
 *
 * // Create a client (uses DATABASE_URL by default)
 * const sql = postgres();
 *
 * // Execute queries using tagged template literals
 * const users = await sql`SELECT * FROM users WHERE active = ${true}`;
 *
 * // Transactions
 * const tx = await sql.begin();
 * try {
 *   await tx`INSERT INTO users (name) VALUES (${name})`;
 *   await tx.commit();
 * } catch (error) {
 *   await tx.rollback();
 *   throw error;
 * }
 *
 * // Close when done
 * await sql.close();
 * ```
 *
 * @packageDocumentation
 */
export { postgres, default } from './postgres';
export { PostgresClient, createCallableClient, type CallablePostgresClient } from './client';
export { Transaction, Savepoint, ReservedConnection } from './transaction';
export { patchBunSQL, isPatched, SQL } from './patch';
export type { PostgresConfig, ReconnectConfig, ConnectionStats, TLSConfig, TransactionOptions, ReserveOptions, } from './types';
export { PostgresError, ConnectionClosedError, ReconnectFailedError, QueryTimeoutError, TransactionError, UnsupportedOperationError, isRetryableError, } from './errors';
export { computeBackoff, sleep, mergeReconnectConfig, DEFAULT_RECONNECT_CONFIG } from './reconnect';
export { shutdownAll, getClientCount, getClients, hasActiveClients } from './registry';
//# sourceMappingURL=index.d.ts.map