import type { Context, MiddlewareHandler } from 'hono';
import type { Env } from '../app';
/**
 * WebSocket connection interface for handling WebSocket events.
 */
export interface WebSocketConnection {
    onOpen: (handler: (event: Event) => void | Promise<void>) => void;
    onMessage: (handler: (event: MessageEvent) => void | Promise<void>) => void;
    onClose: (handler: (event: CloseEvent) => void | Promise<void>) => void;
    send: (data: string | ArrayBuffer | Uint8Array) => void;
}
/**
 * Handler function for WebSocket connections.
 * Receives the Hono context and WebSocket connection with a flattened signature.
 */
export type WebSocketHandler<E extends Env = Env> = (c: Context<E>, ws: WebSocketConnection) => void | Promise<void>;
/**
 * Creates a WebSocket middleware for handling WebSocket connections.
 *
 * Use with router.get() to create a WebSocket endpoint:
 *
 * @example
 * ```typescript
 * import { createRouter, websocket } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * router.get('/ws', websocket((c, ws) => {
 *   ws.onOpen(() => {
 *     c.var.logger.info('WebSocket opened');
 *     ws.send('Welcome!');
 *   });
 *
 *   ws.onMessage((event) => {
 *     c.var.logger.info('Received:', event.data);
 *     ws.send('Echo: ' + event.data);
 *   });
 *
 *   ws.onClose(() => {
 *     c.var.logger.info('WebSocket closed');
 *   });
 * }));
 * ```
 *
 * @param handler - Handler function receiving context and WebSocket connection
 * @returns Hono middleware handler for WebSocket upgrade
 */
export declare function websocket<E extends Env = Env>(handler: WebSocketHandler<E>): MiddlewareHandler<E>;
//# sourceMappingURL=websocket.d.ts.map