import type { PluginInput } from '@opencode-ai/plugin';
import type { TmuxConfig } from './types';
export interface TmuxSessionManagerCallbacks {
    onLog?: (message: string) => void;
}
/**
 * Manages tmux panes for background agents.
 *
 * Architecture:
 * 1. QUERY: Get actual tmux pane state (source of truth)
 * 2. DECIDE: Pure function determines actions based on state
 * 3. EXECUTE: Execute actions with verification
 * 4. UPDATE: Update internal cache only after tmux confirms success
 */
export declare class TmuxSessionManager {
    private ctx;
    private config;
    private callbacks?;
    private sessions;
    private pendingSessions;
    private pollInterval?;
    private sourcePaneId;
    private tmuxSessionId;
    private instanceId;
    private ownerPid;
    private serverKey;
    private statusMissingSince;
    /**
     * Operation queue to serialize tmux mutations.
     * This prevents race conditions when multiple sessions are created rapidly.
     */
    private tmuxOpQueue;
    constructor(ctx: PluginInput, config: TmuxConfig, callbacks?: TmuxSessionManagerCallbacks | undefined);
    /**
     * Enqueue a tmux operation to ensure sequential execution.
     * This prevents race conditions when multiple sessions are created/deleted rapidly.
     */
    private enqueue;
    /**
     * Check if tmux integration is enabled and available
     */
    isEnabled(): boolean;
    /**
     * Handle a new background session being created
     * This is called by BackgroundManager when a background task starts
     *
     * Operations are queued to prevent race conditions when multiple sessions
     * are created rapidly.
     */
    onSessionCreated(event: {
        sessionId: string;
        parentId: string;
        title: string;
    }): Promise<void>;
    /**
     * Internal implementation of session creation (runs within the queue)
     */
    private doSessionCreated;
    /**
     * Get all tracked PIDs for logging
     */
    private getTrackedPids;
    /**
     * Handle a session being deleted
     *
     * Explicitly kills the pane when a background session completes.
     * We can't rely on `opencode attach` exiting because it's an interactive
     * terminal that keeps running even after the session goes idle.
     *
     * Operations are queued to prevent race conditions.
     */
    onSessionDeleted(event: {
        sessionId: string;
    }): Promise<void>;
    /**
     * Internal implementation of session deletion (runs within the queue)
     */
    private doSessionDeleted;
    /**
     * Clean up all panes on shutdown
     *
     * Kills the entire "Agents" window, which closes all agent panes at once.
     * Falls back to pkill if PID-based cleanup fails.
     */
    cleanup(): Promise<void>;
    /**
     * Synchronous cleanup for shutdown (ensures completion before exit)
     *
     * Uses spawnSync to guarantee the tmux commands complete before the
     * process exits, which is necessary for signal handlers.
     */
    cleanupSync(): void;
    /**
     * Start polling for session status
     */
    private startPolling;
    /**
     * Stop polling
     */
    private stopPolling;
    /**
     * Poll active sessions for status changes
     */
    private pollSessions;
    /**
     * Poll active sessions for status changes
     */
    private doPollSessions;
    /**
     * Get session mappings for decision engine
     */
    private getSessionMappings;
    private getServerUrl;
    private getServerKey;
    private applyActionResults;
    private refreshMissingPids;
    private fetchSessionStatuses;
    /**
     * Find and report orphaned processes (does NOT kill them by default).
     * Call this manually if you need to identify orphaned processes after a crash.
     *
     * Note: This method only reports - it does not kill processes because we cannot
     * reliably distinguish between processes we spawned vs user-initiated sessions.
     * The shutdown cleanup (cleanup/cleanupSync) is safe because it only kills PIDs
     * we explicitly tracked during this session.
     */
    reportOrphanedProcesses(): Promise<number[]>;
    /**
     * Kill a process and all its children synchronously.
     *
     * This is necessary because we spawn `bash -c "opencode attach ...; tmux kill-pane"`
     * and #{pane_pid} returns the bash PID, not the opencode attach PID.
     */
    private killProcessByPidSync;
    private log;
    /**
     * Static method to clean up orphaned processes without needing an instance.
     * This is useful for manual cleanup commands.
     *
     * @param serverUrl - Optional server URL to filter processes
     * @param logger - Optional logging function
     * @param instanceId - Ownership instance id to target cleanup
     * @returns Object with cleanup results
     */
    static cleanupOrphans(serverUrl?: string, logger?: (msg: string) => void, instanceId?: string): Promise<{
        killed: number;
        windowClosed: boolean;
    }>;
}
//# sourceMappingURL=manager.d.ts.map