import type { Logger } from '@agentuity/core';
import { projectGet, sandboxGet, deploymentGet, type APIClient } from '@agentuity/server';
import { getResourceRegion, setResourceRegion } from '../../cache';
import { getGlobalCatalystAPIClient } from '../../config';
import type { AuthData, Config } from '../../types';
import * as tui from '../../tui';
import { ErrorCode } from '../../errors';

export type IdentifierType = 'project' | 'deployment' | 'sandbox';

/**
 * Determine the type of identifier based on its prefix
 */
export function getIdentifierType(identifier: string): IdentifierType {
	if (identifier.startsWith('proj_')) {
		return 'project';
	}
	if (identifier.startsWith('deploy_')) {
		return 'deployment';
	}
	if (identifier.startsWith('sbx_')) {
		return 'sandbox';
	}
	// Default to project for unknown prefixes
	return 'project';
}

/**
 * Look up the region for a project, deployment, or sandbox identifier.
 * Uses cache-first strategy with API fallback.
 */
export async function getIdentifierRegion(
	logger: Logger,
	auth: AuthData,
	apiClient: APIClient,
	profileName = 'production',
	identifier: string,
	orgId?: string,
	config?: Config | null
): Promise<string> {
	const identifierType = getIdentifierType(identifier);

	// For project, deployment, and sandbox, check cache first
	const cachedRegion = await getResourceRegion(identifierType, profileName, identifier);
	if (cachedRegion) {
		logger.trace(`[region-lookup] Found cached region for ${identifier}: ${cachedRegion}`);
		return cachedRegion;
	}

	logger.trace(`[region-lookup] Cache miss for ${identifier}, fetching from API`);

	let region: string | null = null;

	if (identifierType === 'project') {
		const project = await projectGet(apiClient, { id: identifier, mask: true, keys: false });
		region = project.cloudRegion ?? null;
	} else if (identifierType === 'deployment') {
		const deployment = await deploymentGet(apiClient, identifier);
		region = deployment.cloudRegion ?? null;
	} else {
		// sandbox - pass config to getGlobalCatalystAPIClient for proper region resolution
		const globalClient = await getGlobalCatalystAPIClient(
			logger,
			auth,
			profileName,
			orgId,
			config
		);
		const sandbox = await sandboxGet(globalClient, { sandboxId: identifier, orgId });
		region = sandbox.region ?? null;
	}

	if (!region) {
		tui.fatal(
			`Could not determine region for ${identifierType} '${identifier}'. Use --region flag to specify.`,
			ErrorCode.RESOURCE_NOT_FOUND
		);
	}

	// Validate region is a non-empty string
	if (typeof region !== 'string' || region.trim() === '') {
		tui.fatal(
			`Invalid region returned for ${identifierType} '${identifier}': '${region}'. Use --region flag to specify a valid region.`,
			ErrorCode.RESOURCE_NOT_FOUND
		);
	}

	// Cache the result
	await setResourceRegion(identifierType, profileName, identifier, region);
	logger.trace(`[region-lookup] Cached region for ${identifier}: ${region}`);

	return region;
}

/**
 * Cache the region for a project after creation
 */
export async function cacheProjectRegion(
	profileName = 'production',
	projectId: string,
	region: string
): Promise<void> {
	await setResourceRegion('project', profileName, projectId, region);
}
