import { FetchAdapter } from './adapter';
/**
 * Minimum TTL value in seconds (1 minute)
 */
export declare const STREAM_MIN_TTL_SECONDS = 60;
/**
 * Maximum TTL value in seconds (90 days)
 */
export declare const STREAM_MAX_TTL_SECONDS = 7776000;
/**
 * Default TTL value in seconds (30 days) - used when no TTL is specified
 */
export declare const STREAM_DEFAULT_TTL_SECONDS = 2592000;
/**
 * Properties for creating a stream
 */
export interface CreateStreamProps {
    /**
     * optional metadata for the stream
     */
    metadata?: Record<string, string>;
    /**
     * optional contentType for the stream data. If not set, defaults to application/octet-stream
     */
    contentType?: string;
    /**
     * optional flag to enable gzip compression of stream data during upload. if true, will also add
     * add Content-Encoding: gzip header to responses. The client MUST be able to accept gzip
     * compression for this to work or must be able to uncompress the raw data it receives.
     */
    compress?: true;
    /**
     * optional time-to-live in seconds for the stream. Controls when the stream expires and is automatically deleted.
     * - `undefined` (not provided): Stream expires after 30 days (default)
     * - `null` or `0`: Stream never expires
     * - positive number (≥60): Stream expires after the specified number of seconds (max 90 days)
     *
     * @remarks
     * TTL values below 60 seconds are clamped to 60 seconds by the server.
     * TTL values above 7,776,000 seconds (90 days) are clamped to 90 days.
     *
     * @default 2592000 (30 days)
     */
    ttl?: number | null;
}
/**
 * Parameters for listing streams
 */
export interface ListStreamsParams {
    /**
     * optional namespace filter to search for streams
     */
    namespace?: string;
    /**
     * optional metadata filters to match streams
     */
    metadata?: Record<string, string>;
    /**
     * maximum number of streams to return (default: 100, max: 1000)
     */
    limit?: number;
    /**
     * number of streams to skip for pagination
     */
    offset?: number;
}
/**
 * Stream information returned by list operation
 */
export interface StreamInfo {
    /**
     * unique stream identifier
     */
    id: string;
    /**
     * the namespace of the stream
     */
    namespace: string;
    /**
     * the stream metadata
     */
    metadata: Record<string, string>;
    /**
     * the public URL to access the stream
     */
    url: string;
    /**
     * the size of the stream in bytes
     */
    sizeBytes: number;
    /**
     * ISO 8601 timestamp when stream expires, or undefined if stream never expires
     */
    expiresAt?: string;
}
/**
 * Response from listing streams
 */
export interface ListStreamsResponse {
    /**
     * whether the request was successful
     */
    success: boolean;
    /**
     * optional error message if not successful
     */
    message?: string;
    /**
     * array of streams matching the filter criteria
     */
    streams: StreamInfo[];
    /**
     * total count of streams matching the filter (useful for pagination)
     */
    total: number;
}
/**
 * A durable and resumable stream that can be written to and read many times.
 * The underlying stream is backed by a durable storage system and the URL
 * returned is public and guaranteed to return the same data every time it is accessed.
 * You can read from this stream internal in the agent using the getReader() method or
 * return the URL to the stream to be used externally.
 *
 * You must write and close the stream before it can be read but if you attempt to read
 * before any data is written, the reader will block until the first write occurs.
 */
export interface Stream extends WritableStream {
    /**
     * unique stream identifier
     */
    id: string;
    /**
     * the unique stream url to consume the stream
     */
    url: string;
    /**
     * the total number of bytes written to the stream
     */
    readonly bytesWritten: number;
    /**
     * whether the stream is using compression
     */
    readonly compressed: boolean;
    /**
     * Write data to the stream.
     *
     * Each write call can send up to 5MB of data. For larger payloads, split the data
     * across multiple write calls.
     *
     * @param chunk - The data to write (string, binary, or object that will be JSON-serialized)
     * @throws ServiceException with status 413 if chunk exceeds 5MB
     */
    write(chunk: string | Uint8Array | ArrayBuffer | object): Promise<void>;
    /**
     * close the stream gracefully, handling already closed streams without error
     */
    close(): Promise<void>;
    /**
     * get a ReadableStream that streams from the internal URL
     *
     * Note: This method will block waiting for data until writes start to the Stream.
     * The returned ReadableStream will remain open until the Stream is closed or an error occurs.
     *
     * @returns a ReadableStream that can be passed to response.stream()
     */
    getReader(): ReadableStream<Uint8Array>;
}
/**
 * Stream API for creating and managing durable, resumable data streams.
 * Streams are backed by durable storage and provide public URLs for access.
 */
export interface StreamStorage {
    /**
     * Create a new stream for writing data that can be read multiple times
     *
     * @param namespace - the namespace of the stream (1-254 characters). Use namespaces to group and organize streams.
     * @param props - optional properties including metadata, content type, compression, and TTL
     * @returns a Promise that resolves to the created Stream
     *
     * @example
     * ```typescript
     * // Create a simple text stream (expires in 30 days by default)
     * const stream = await streams.create('agent-logs');
     * await stream.write('Starting agent execution\n');
     * await stream.write('Processing data...\n');
     * await stream.close();
     * console.log('Stream URL:', stream.url);
     *
     * // Create a stream with custom TTL (expires in 1 hour)
     * const tempStream = await streams.create('temp-data', {
     *   ttl: 3600  // 1 hour in seconds
     * });
     *
     * // Create a stream that never expires
     * const permanentStream = await streams.create('permanent-data', {
     *   ttl: null  // or ttl: 0
     * });
     *
     * // Create a compressed JSON stream with metadata
     * const dataStream = await streams.create('data-export', {
     *   contentType: 'application/json',
     *   compress: true,
     *   metadata: { exportDate: '2024-01-15', version: '1.0' }
     * });
     * await dataStream.write({ records: [...] });
     * await dataStream.close();
     *
     * // Read back from the stream
     * const reader = dataStream.getReader();
     * for await (const chunk of reader) {
     *   console.log('Chunk:', chunk);
     * }
     * ```
     */
    create(namespace: string, props?: CreateStreamProps): Promise<Stream>;
    /**
     * Get stream metadata by ID
     *
     * @param id - the stream ID
     * @returns a Promise that resolves to the stream information
     *
     * @example
     * ```typescript
     * const stream = await streams.get('stream_0199a52b06e3767dbe2f10afabb5e5e4');
     * console.log(`Namespace: ${stream.namespace}, Size: ${stream.sizeBytes} bytes`);
     * ```
     */
    get(id: string): Promise<StreamInfo>;
    /**
     * Download stream content
     *
     * @param id - the stream ID to download
     * @returns a Promise that resolves to a ReadableStream of the content
     *
     * @example
     * ```typescript
     * const readable = await streams.download('stream_0199a52b06e3767dbe2f10afabb5e5e4');
     * // Pipe to a file or process the stream
     * ```
     */
    download(id: string): Promise<ReadableStream<Uint8Array>>;
    /**
     * List streams with optional filtering and pagination
     *
     * @param params - optional parameters for filtering and pagination
     * @returns a Promise that resolves to the list of streams with metadata
     *
     * @example
     * ```typescript
     * // List all streams
     * const all = await streams.list();
     * console.log(`Found ${all.total} streams`);
     *
     * // Filter by namespace
     * const logs = await streams.list({ namespace: 'agent-logs' });
     *
     * // Filter by metadata and paginate
     * const filtered = await streams.list({
     *   metadata: { type: 'export' },
     *   limit: 50,
     *   offset: 100
     * });
     *
     * for (const stream of filtered.streams) {
     *   console.log(`${stream.namespace}: ${stream.sizeBytes} bytes at ${stream.url}`);
     * }
     * ```
     */
    list(params?: ListStreamsParams): Promise<ListStreamsResponse>;
    /**
     * Delete a stream by its ID
     *
     * @param id - the stream ID to delete
     * @returns a Promise that resolves when the stream is deleted
     *
     * @example
     * ```typescript
     * await streams.delete('stream-id-123');
     * ```
     */
    delete(id: string): Promise<void>;
}
export declare class StreamStorageService implements StreamStorage {
    #private;
    constructor(baseUrl: string, adapter: FetchAdapter);
    create(namespace: string, props?: CreateStreamProps): Promise<Stream>;
    list(params?: ListStreamsParams): Promise<ListStreamsResponse>;
    get(id: string): Promise<StreamInfo>;
    download(id: string): Promise<ReadableStream<Uint8Array>>;
    delete(id: string): Promise<void>;
}
//# sourceMappingURL=stream.d.ts.map