import { z } from 'zod';
import { createResources } from '@agentuity/server';
import { createSubcommand as defineSubcommand } from '../../../types';
import * as tui from '../../../tui';
import { getCatalystAPIClient } from '../../../config';
import { getCommand } from '../../../command-prefix';
import { isDryRunMode, outputDryRun } from '../../../explain';
import { addResourceEnvVars } from '../../../env-util';
export const createSubcommand = defineSubcommand({
    name: 'create',
    aliases: ['new'],
    description: 'Create a new storage resource',
    tags: ['mutating', 'creates-resource', 'slow', 'requires-auth', 'requires-deployment'],
    idempotent: false,
    requires: { auth: true, org: true, region: true },
    examples: [
        {
            command: getCommand('cloud storage create'),
            description: 'Create a new cloud storage bucket',
        },
        {
            command: getCommand('cloud storage new'),
            description: 'Alias for "cloud storage create" (shorthand "new")',
        },
        {
            command: getCommand('--dry-run cloud storage create'),
            description: 'Dry-run: display what would be created without making changes',
        },
    ],
    schema: {
        options: z.object({
            description: z.string().optional().describe('Optional description for the bucket'),
        }),
        response: z.object({
            success: z.boolean().describe('Whether creation succeeded'),
            name: z.string().describe('Created storage bucket name'),
        }),
    },
    async handler(ctx) {
        const { logger, orgId, region, auth, options, opts } = ctx;
        // Handle dry-run mode
        if (isDryRunMode(options)) {
            const message = `Would create storage bucket in region: ${region}`;
            outputDryRun(message, options);
            if (!options.json) {
                tui.newline();
                tui.info('[DRY RUN] Storage creation skipped');
            }
            return {
                success: false,
                name: 'dry-run-bucket',
            };
        }
        const catalystClient = getCatalystAPIClient(logger, auth, region);
        const created = await tui.spinner({
            message: `Creating storage in ${region}`,
            clearOnSuccess: true,
            callback: async () => {
                return createResources(catalystClient, orgId, region, [
                    { type: 's3', description: opts.description },
                ]);
            },
        });
        const resource = created[0];
        if (resource) {
            // Write environment variables to .env if running inside a project
            if (ctx.projectDir && resource.env && Object.keys(resource.env).length > 0) {
                await addResourceEnvVars(ctx.projectDir, resource.env);
                if (!options.json) {
                    tui.info('Environment variables written to .env');
                }
            }
            if (!options.json) {
                tui.success(`Created storage: ${tui.bold(resource.name)}`);
            }
            return {
                success: true,
                name: resource.name,
            };
        }
        else {
            tui.fatal('Failed to create storage');
        }
    },
});
//# sourceMappingURL=create.js.map