import { z } from 'zod';
import { type APIClient, APIResponseSchema } from '../api';

const PackageRef = z.object({
	name: z.string(),
	version: z.string(),
});

export type PackageRef = z.infer<typeof PackageRef>;

const MalwareFinding = z.object({
	name: z.string(),
	version: z.string(),
	reason: z.string(),
});

export type MalwareFinding = z.infer<typeof MalwareFinding>;

const MalwareCheckSummary = z.object({
	scanned: z.number(),
	flagged: z.number(),
});

const MalwareCheckListMetadata = z.object({
	fetchedAt: z.string(),
	count: z.number(),
});

const MalwareCheckResult = z.object({
	action: z.enum(['allow', 'block']),
	summary: MalwareCheckSummary,
	findings: z.array(MalwareFinding),
	list: MalwareCheckListMetadata.optional(),
	error: z.string().optional(),
});

const MalwareCheckResponseSchema = APIResponseSchema(MalwareCheckResult);

export type MalwareCheckResult = z.infer<typeof MalwareCheckResult>;

export async function projectDeploymentMalwareCheck(
	client: APIClient,
	deploymentId: string,
	packages: PackageRef[]
): Promise<MalwareCheckResult> {
	const resp = await client.request<z.infer<typeof MalwareCheckResponseSchema>>(
		'POST',
		`/security/2026-01-22/${deploymentId}/malware-check`,
		MalwareCheckResponseSchema,
		{
			ecosystem: 'npm',
			packages,
		}
	);

	if (!resp.success) {
		throw new Error(resp.message || 'Malware check request failed');
	}

	return resp.data;
}
