import { z } from 'zod';
import { APIClient, APIResponseSchema } from '../api';
import { ProjectResponseError } from './util';

const CreateProjectRequestSchema = z.object({
	name: z.string().max(255).min(1).describe('the name of the new project'),
	description: z.string().max(255).min(0).optional().describe('the description of the project'),
	tags: z.array(z.string()).optional().describe('tags for the project'),
	orgId: z.string().max(255).min(1).describe('the organization id to create the project in'),
	cloudRegion: z.string().describe('the cloud region to create the project'),
	domains: z.array(z.string()).optional().describe('the custom domains for this project'),
});

const CreateProjectResponse = z.object({
	id: z.string().describe('the unique id for the project'),
	sdkKey: z.string().describe('the SDK key for the project'),
});

const CreateProjectResponseSchema = APIResponseSchema(CreateProjectResponse);

export type CreateProjectRequest = z.infer<typeof CreateProjectRequestSchema>;
export type CreateProjectResponse = z.infer<typeof CreateProjectResponseSchema>;
export type NewProject = z.infer<typeof CreateProjectResponse>;

/**
 * Create a new Project
 *
 * @param client
 * @param body
 * @returns
 */
export async function projectCreate(
	client: APIClient,
	body: CreateProjectRequest
): Promise<NewProject> {
	const resp = await client.request<CreateProjectResponse, CreateProjectRequest>(
		'POST',
		'/cli/project',
		CreateProjectResponseSchema,
		body,
		CreateProjectRequestSchema
	);
	if (resp.success) {
		return resp.data;
	}
	throw new ProjectResponseError({ message: resp.message });
}
