import { APIClient } from '../api';
import { type OrgAnalytics, type QueueAnalytics, type TimeSeriesData, type SSEStatsEvent, type AnalyticsOptions, type StreamAnalyticsOptions } from './types';
/**
 * Get org-level analytics for all queues.
 *
 * Returns aggregated statistics across all queues in the organization.
 *
 * @param client - The API client instance
 * @param options - Analytics options (time range, filters)
 * @returns Org-level analytics summary
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const analytics = await getOrgAnalytics(client, {
 *   start: '2026-01-14T00:00:00Z',
 *   end: '2026-01-15T00:00:00Z',
 * });
 * console.log(`Total queues: ${analytics.summary.total_queues}`);
 * console.log(`Messages published: ${analytics.summary.total_messages_published}`);
 * ```
 */
export declare function getOrgAnalytics(client: APIClient, options?: AnalyticsOptions): Promise<OrgAnalytics>;
/**
 * Get detailed analytics for a specific queue.
 *
 * Returns current state, period statistics, latency metrics, and destination stats.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @param options - Analytics options (time range, filters)
 * @returns Queue analytics
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const analytics = await getQueueAnalytics(client, 'order-processing', {
 *   start: '2026-01-14T00:00:00Z',
 * });
 * console.log(`Backlog: ${analytics.current.backlog}`);
 * console.log(`P95 latency: ${analytics.latency.p95_ms}ms`);
 * ```
 */
export declare function getQueueAnalytics(client: APIClient, name: string, options?: AnalyticsOptions): Promise<QueueAnalytics>;
/**
 * Get time series analytics data for a queue.
 *
 * Returns time-bucketed metrics for visualization in charts and graphs.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @param options - Analytics options (time range, granularity)
 * @returns Time series data
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const timeseries = await getQueueTimeSeries(client, 'order-processing', {
 *   granularity: 'hour',
 *   start: '2026-01-14T00:00:00Z',
 * });
 * for (const point of timeseries.series) {
 *   console.log(`${point.timestamp}: ${point.throughput} msg/h`);
 * }
 * ```
 */
export declare function getQueueTimeSeries(client: APIClient, name: string, options?: AnalyticsOptions): Promise<TimeSeriesData>;
/**
 * Stream real-time analytics for all queues via SSE.
 *
 * Returns an async iterator that yields stats events at the specified interval.
 * The connection stays open until the iterator is closed or an error occurs.
 *
 * @param client - The API client instance
 * @param options - Stream options (interval, orgId)
 * @returns Async iterator of SSE stats events
 *
 * @example
 * ```typescript
 * const stream = streamOrgAnalytics(client, { interval: 5 });
 * for await (const event of stream) {
 *   console.log(`Backlog: ${event.backlog}, Throughput: ${event.throughput_1m}/min`);
 * }
 * ```
 */
export declare function streamOrgAnalytics(client: APIClient, options?: StreamAnalyticsOptions): AsyncGenerator<SSEStatsEvent, void, unknown>;
/**
 * Stream real-time analytics for a specific queue via SSE.
 *
 * Returns an async iterator that yields stats events at the specified interval.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @param options - Stream options (interval, orgId)
 * @returns Async iterator of SSE stats events
 *
 * @example
 * ```typescript
 * const stream = streamQueueAnalytics(client, 'order-processing', { interval: 5 });
 * for await (const event of stream) {
 *   console.log(`Backlog: ${event.backlog}, In-flight: ${event.messages_in_flight}`);
 * }
 * ```
 */
export declare function streamQueueAnalytics(client: APIClient, name: string, options?: StreamAnalyticsOptions): AsyncGenerator<SSEStatsEvent, void, unknown>;
//# sourceMappingURL=analytics.d.ts.map