import { sandboxCreate, sandboxDestroy, sandboxExecute, sandboxGet, sandboxList, sandboxRun, sandboxWriteFiles, sandboxReadFile, snapshotCreate, snapshotGet, snapshotList, snapshotDelete, snapshotTag, } from '@agentuity/server';
import { context, SpanKind, SpanStatusCode, trace } from '@opentelemetry/api';
const TRACER_NAME = 'agentuity.sandbox';
async function withSpan(name, attributes, fn) {
    const tracer = trace.getTracer(TRACER_NAME);
    const currentContext = context.active();
    const span = tracer.startSpan(name, { attributes, kind: SpanKind.CLIENT }, currentContext);
    const spanContext = trace.setSpan(currentContext, span);
    try {
        const result = await context.with(spanContext, fn);
        span.setStatus({ code: SpanStatusCode.OK });
        return result;
    }
    catch (err) {
        const e = err;
        span.recordException(e);
        span.setStatus({ code: SpanStatusCode.ERROR, message: e?.message ?? String(err) });
        throw err;
    }
    finally {
        span.end();
    }
}
function createStreamReader(id, baseUrl) {
    const streamId = id ?? '';
    const url = streamId ? `${baseUrl}/${streamId}` : '';
    return {
        id: streamId,
        url,
        readonly: true,
        getReader() {
            if (!url) {
                return new ReadableStream({
                    start(controller) {
                        controller.close();
                    },
                });
            }
            return new ReadableStream({
                async start(controller) {
                    try {
                        const response = await fetch(url);
                        if (!response.ok || !response.body) {
                            controller.close();
                            return;
                        }
                        const reader = response.body.getReader();
                        while (true) {
                            const { done, value } = await reader.read();
                            if (done)
                                break;
                            controller.enqueue(value);
                        }
                        controller.close();
                    }
                    catch {
                        controller.close();
                    }
                },
            });
        },
    };
}
function createSandboxInstance(client, sandboxId, status, streamBaseUrl, stdoutStreamId, stderrStreamId) {
    const interleaved = !!(stdoutStreamId && stderrStreamId && stdoutStreamId === stderrStreamId);
    return {
        id: sandboxId,
        status,
        stdout: createStreamReader(stdoutStreamId, streamBaseUrl),
        stderr: createStreamReader(stderrStreamId, streamBaseUrl),
        interleaved,
        async execute(options) {
            return withSpan('agentuity.sandbox.execute', {
                'sandbox.id': sandboxId,
                'sandbox.command': options.command?.join(' ') ?? '',
            }, () => sandboxExecute(client, { sandboxId, options, signal: options.signal }));
        },
        async writeFiles(files) {
            await withSpan('agentuity.sandbox.writeFiles', {
                'sandbox.id': sandboxId,
                'sandbox.files.count': files.length,
            }, () => sandboxWriteFiles(client, { sandboxId, files }));
        },
        async readFile(path) {
            return withSpan('agentuity.sandbox.readFile', {
                'sandbox.id': sandboxId,
                'sandbox.file.path': path,
            }, () => sandboxReadFile(client, { sandboxId, path }));
        },
        async destroy() {
            await withSpan('agentuity.sandbox.destroy', { 'sandbox.id': sandboxId }, () => sandboxDestroy(client, { sandboxId }));
        },
    };
}
/**
 * HTTP implementation of the SnapshotService interface
 */
class HTTPSnapshotService {
    client;
    constructor(client) {
        this.client = client;
    }
    async create(sandboxId, options) {
        return withSpan('agentuity.sandbox.snapshot.create', {
            'sandbox.id': sandboxId,
            'snapshot.name': options?.name ?? '',
            'snapshot.tag': options?.tag ?? '',
        }, () => snapshotCreate(this.client, {
            sandboxId,
            name: options?.name,
            description: options?.description,
            tag: options?.tag,
            public: options?.public,
        }));
    }
    async get(snapshotId) {
        return withSpan('agentuity.sandbox.snapshot.get', { 'snapshot.id': snapshotId }, () => snapshotGet(this.client, { snapshotId }));
    }
    async list(params) {
        return withSpan('agentuity.sandbox.snapshot.list', {
            'snapshot.sandboxId': params?.sandboxId ?? '',
            'snapshot.limit': params?.limit ?? 50,
        }, () => snapshotList(this.client, {
            sandboxId: params?.sandboxId,
            limit: params?.limit,
            offset: params?.offset,
        }));
    }
    async delete(snapshotId) {
        return withSpan('agentuity.sandbox.snapshot.delete', { 'snapshot.id': snapshotId }, () => snapshotDelete(this.client, { snapshotId }));
    }
    async tag(snapshotId, tag) {
        return withSpan('agentuity.sandbox.snapshot.tag', {
            'snapshot.id': snapshotId,
            'snapshot.tag': tag ?? '',
        }, () => snapshotTag(this.client, { snapshotId, tag }));
    }
}
/**
 * HTTP implementation of the SandboxService interface
 */
export class HTTPSandboxService {
    client;
    streamBaseUrl;
    /**
     * Snapshot management operations
     */
    snapshot;
    constructor(client, streamBaseUrl) {
        this.client = client;
        this.streamBaseUrl = streamBaseUrl;
        this.snapshot = new HTTPSnapshotService(client);
    }
    async run(options) {
        return withSpan('agentuity.sandbox.run', {
            'sandbox.command': options.command?.exec?.join(' ') ?? '',
            'sandbox.mode': 'oneshot',
        }, () => sandboxRun(this.client, { options }));
    }
    async create(options) {
        return withSpan('agentuity.sandbox.create', {
            'sandbox.network': options?.network?.enabled ?? false,
            'sandbox.snapshot': options?.snapshot ?? '',
        }, async () => {
            const response = await sandboxCreate(this.client, { options });
            return createSandboxInstance(this.client, response.sandboxId, response.status, this.streamBaseUrl, response.stdoutStreamId, response.stderrStreamId);
        });
    }
    async get(sandboxId) {
        return withSpan('agentuity.sandbox.get', { 'sandbox.id': sandboxId }, () => sandboxGet(this.client, { sandboxId }));
    }
    async list(params) {
        return withSpan('agentuity.sandbox.list', {
            'sandbox.status': params?.status ?? '',
            'sandbox.limit': params?.limit ?? 50,
        }, () => sandboxList(this.client, params));
    }
    async destroy(sandboxId) {
        return withSpan('agentuity.sandbox.destroy', { 'sandbox.id': sandboxId }, () => sandboxDestroy(this.client, { sandboxId }));
    }
}
//# sourceMappingURL=http.js.map