# Agent Guidelines for @agentuity/react

## Package Overview

React hooks and components for building Agentuity web applications. Provides type-safe hooks for calling agents and WebSocket communication.

## Commands

- **Build**: `bun run build` (compiles for browser target)
- **Typecheck**: `bun run typecheck` (runs TypeScript type checking)
- **Clean**: `bun run clean` (removes dist/)

## Architecture

- **Runtime**: Browser only (uses browser APIs like fetch, WebSocket)
- **Build target**: Browser with ESNext
- **Dependencies**: Requires `@agentuity/core` (workspace dependency)
- **Peer dependencies**: React 18+ or 19+

## Structure

```text
src/
├── index.ts        # Main entry point
├── context.tsx     # AgentuityProvider and AgentuityContext
├── run.ts          # useAgent hook for HTTP calls
├── websocket.ts    # useWebsocket hook for WebSocket
├── types.ts        # Type definitions for agent registry
├── url.ts          # URL building utilities
└── env.ts          # Environment helpers
```

## Code Style

- **React hooks** - Follow React hooks conventions
- **TypeScript generics** - Heavy use of generics for type safety
- **Functional components** - All components are functional
- **Context API** - Use React Context for configuration
- **Error boundaries** - Throw errors that can be caught by error boundaries

## Important Conventions

- **Provider required** - All hooks must be used within `AgentuityProvider`
- **Type inference** - Agent types are inferred from generated types (AgentRegistry)
- **Base URL** - Defaults to current origin if not provided
- **WebSocket protocol** - Auto-converts http:// to ws:// and https:// to wss://
- **Serialization** - Automatically handles JSON serialization/deserialization

## Hooks API

### useAuth

```typescript
const { isAuthenticated, authLoading, authHeader } = useAuth();
```

- Provides auth state for conditional rendering
- `authHeader` is automatically injected into agent/API calls
- Works with `@agentuity/auth`'s `AuthProvider`

### useAgent

```typescript
const { data, run } = useAgent('agentName');
```

- Returns last response in `data`
- `run()` function for calling the agent
- Supports custom headers, query params, subpaths
- **Auth tokens auto-injected** when `AuthProvider` is in tree

### useWebsocket

```typescript
const { connected, send, setHandler, close } = useWebsocket('/path');
```

- Auto-reconnection on connection loss
- Message queuing when disconnected
- Type-safe message handlers
- **Auth tokens auto-injected** when `AuthProvider` is in tree

## Generated Types

This package expects a `AgentRegistry` type to be augmented by generated code:

```typescript
// Generated by bundler
export interface AgentRegistry {
  'my-agent': {
    inputSchema: z.ZodObject<...>;
    outputSchema: z.ZodObject<...>;
  };
}
```

## Testing

- Test with React Testing Library
- Mock fetch and WebSocket APIs
- Test error boundaries
- Test with and without provider
- When running tests, prefer using a subagent (Task tool) to avoid context bloat from test output

## Publishing Checklist

1. Run `bun run build` to compile for browser
2. Verify `dist/` contains browser-compatible code (no Node.js APIs)
3. Ensure peer dependencies are correctly specified
4. Must publish **after** @agentuity/core
