/**
 * Dev Lock Manager
 *
 * Manages a lockfile to track the dev server process and its children.
 * On startup, detects and cleans up stale processes from previous sessions.
 * Ensures proper cleanup on all exit paths.
 */
interface LoggerLike {
    debug: (msg: string, ...args: unknown[]) => void;
    warn: (msg: string, ...args: unknown[]) => void;
    error: (msg: string, ...args: unknown[]) => void;
}
/**
 * Lockfile format for tracking dev server processes
 */
export interface DevLockFileV1 {
    version: 1;
    projectRoot: string;
    mainPid: number;
    instanceId: string;
    createdAt: string;
    updatedAt: string;
    ports: {
        bun?: number;
        vite?: number;
        gravity?: number;
    };
    children: Array<{
        pid: number;
        type: 'gravity' | 'vite' | 'other';
        description?: string;
    }>;
}
export interface DevLockManager {
    state: DevLockFileV1;
    registerChild: (info: {
        pid: number;
        type: 'gravity' | 'vite' | 'other';
        description?: string;
    }) => Promise<void>;
    updatePorts: (ports: Partial<DevLockFileV1['ports']>) => Promise<void>;
    release: () => Promise<void>;
}
/**
 * Main entry point for dev lock management
 * Call this early in the dev command to:
 * 1. Clean up any stale processes from previous sessions
 * 2. Create a new lockfile for this session
 */
export declare function prepareDevLock(rootDir: string, port: number, logger: LoggerLike): Promise<DevLockManager>;
/**
 * Utility to kill all processes in a lockfile by path
 * Useful for emergency cleanup without creating a new lock
 */
export declare function cleanupLockfile(rootDir: string, logger: LoggerLike): Promise<void>;
/**
 * Synchronous lockfile removal for use in process.on('exit') handlers
 * Does not kill processes - just removes the file
 */
export declare function releaseLockSync(rootDir: string): void;
export {};
//# sourceMappingURL=dev-lock.d.ts.map