import { exitWithError, createError, getExitCode } from './errors';
/**
 * Enhanced logger wrapper that supports structured errors
 */
export class CLILogger {
    logger;
    options;
    constructor(logger, options) {
        this.logger = logger;
        this.options = options;
    }
    /**
     * Exit with a structured error (supports --error-format=json)
     */
    fatalWithError(error) {
        const errorFormat = this.options.errorFormat ?? 'text';
        return exitWithError(error, this.logger, errorFormat);
    }
    // Delegate all other logger methods
    trace(message, ...args) {
        this.logger.trace(message, ...args);
    }
    debug(message, ...args) {
        this.logger.debug(message, ...args);
    }
    info(message, ...args) {
        this.logger.info(message, ...args);
    }
    warn(message, ...args) {
        this.logger.warn(message, ...args);
    }
    error(message, ...args) {
        this.logger.error(message, ...args);
    }
    fatal(message, errorCode, ...args) {
        if (errorCode) {
            // Use structured error with proper exit code
            const formattedMessage = args.length > 0 ? this.formatMessage(message, ...args) : message;
            const error = createError(errorCode, formattedMessage);
            const exitCode = getExitCode(errorCode);
            if (this.options.errorFormat === 'json') {
                exitWithError(error, this.logger, 'json');
            }
            else {
                this.logger.error(formattedMessage);
                process.exit(exitCode);
            }
        }
        else {
            // Fallback to default behavior (exit code 1)
            this.logger.fatal(message, ...args);
        }
    }
    formatMessage(message, ...args) {
        // Simple sprintf-style formatting
        let formatted = message;
        for (const arg of args) {
            formatted = formatted.replace(/%[sd]/, String(arg));
        }
        return formatted;
    }
    child(opts) {
        return this.logger.child(opts);
    }
}
/**
 * Wrap a logger with CLI error handling capabilities
 */
export function wrapLogger(logger, options) {
    return new CLILogger(logger, options);
}
//# sourceMappingURL=cli-logger.js.map