/* eslint-disable @typescript-eslint/no-explicit-any */
import { Hono } from 'hono';
import { returnResponse } from './_util';
import { loadBuildMetadata } from './_metadata';
/**
 * Creates a Hono router with extended methods for Agentuity-specific routing patterns.
 *
 * Standard HTTP methods (get, post, put, delete, patch) are available, plus middleware
 * functions for specialized protocols:
 *
 * - **websocket()** - WebSocket connections (import { websocket } from '@agentuity/runtime')
 * - **sse()** - Server-Sent Events (import { sse } from '@agentuity/runtime')
 * - **stream()** - Streaming responses (import { stream } from '@agentuity/runtime')
 * - **cron()** - Scheduled tasks (import { cron } from '@agentuity/runtime')
 *
 * @template E - Environment type (Hono Env)
 * @template S - Schema type for route definitions
 *
 * @returns Extended Hono router
 *
 * @example
 * ```typescript
 * import { createRouter, websocket, sse, stream, cron } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * // Standard HTTP routes
 * router.get('/hello', (c) => c.text('Hello!'));
 * router.post('/data', async (c) => {
 *   const body = await c.req.json();
 *   return c.json({ received: body });
 * });
 *
 * // WebSocket connection
 * router.get('/ws', websocket((c, ws) => {
 *   ws.onMessage((event) => {
 *     ws.send('Echo: ' + event.data);
 *   });
 * }));
 *
 * // Server-Sent Events
 * router.get('/events', sse((c, stream) => {
 *   stream.writeSSE({ data: 'Hello', event: 'message' });
 * }));
 *
 * // Streaming response
 * router.post('/stream', stream((c) => {
 *   return new ReadableStream({
 *     start(controller) {
 *       controller.enqueue('data\n');
 *       controller.close();
 *     }
 *   });
 * }));
 *
 * // Cron job
 * router.post('/daily', cron('0 0 * * *', (c) => {
 *   return { status: 'complete' };
 * }));
 * ```
 */
export const createRouter = () => {
    const router = new Hono();
    // tslint:disable-next-line:no-any no-unused-variable
    // biome-ignore lint:no-any
    const _router = router;
    for (const method of ['get', 'put', 'post', 'delete', 'options', 'patch']) {
        const _originalInvoker = _router[method].bind(router);
        _router[method] = (path, ...args) => {
            // Pass through to original Hono - it handles all the complex type inference
            // We'll only wrap the final handler to add our response handling
            if (args.length === 0) {
                return _originalInvoker(path);
            }
            // Find the last function in args - that's the handler (everything else is middleware)
            let handlerIndex = args.length - 1;
            while (handlerIndex >= 0 && typeof args[handlerIndex] !== 'function') {
                handlerIndex--;
            }
            if (handlerIndex < 0) {
                // No handler found, pass through as-is
                return _originalInvoker(path, ...args);
            }
            const handler = args[handlerIndex];
            // Check if this is middleware (2 params: c, next) vs handler (1 param: c)
            if (handler.length === 2) {
                // This is middleware-only, pass through
                return _originalInvoker(path, ...args);
            }
            // Wrap the handler to add our response conversion and set routeId
            const wrapper = async (c) => {
                // Look up the route ID from build metadata by matching method and path
                // Try both the registered path and the actual request path (which may include base path)
                const metadata = loadBuildMetadata();
                const methodUpper = method.toUpperCase();
                const requestPath = c.req.routePath || c.req.path;
                // Try matching by registered path first, then by request path, then by path ending
                let route = metadata?.routes?.find((r) => r.method.toUpperCase() === methodUpper && r.path === path);
                if (!route) {
                    route = metadata?.routes?.find((r) => r.method.toUpperCase() === methodUpper && r.path === requestPath);
                }
                if (!route) {
                    // Try matching by path ending (handles /api/translate matching /translate)
                    route = metadata?.routes?.find((r) => r.method.toUpperCase() === methodUpper && r.path.endsWith(path));
                }
                if (route?.id) {
                    c.set('routeId', route.id);
                }
                let result = handler(c);
                if (result instanceof Promise)
                    result = await result;
                // If handler returns a Response, return it unchanged
                if (result instanceof Response)
                    return result;
                return returnResponse(c, result);
            };
            // Replace the handler with our wrapper
            const newArgs = [...args];
            newArgs[handlerIndex] = wrapper;
            return _originalInvoker(path, ...newArgs);
        };
    }
    // Deprecated stubs that throw errors with migration instructions
    _router.websocket = (path, ..._args) => {
        throw new Error(`router.websocket() is deprecated and has been removed.\n\n` +
            `Migration: Use the websocket middleware instead:\n\n` +
            `  import { createRouter, websocket } from '@agentuity/runtime';\n\n` +
            `  const router = createRouter();\n\n` +
            `  // Before (deprecated):\n` +
            `  // router.websocket('${path}', (c) => (ws) => { ... });\n\n` +
            `  // After:\n` +
            `  router.get('${path}', websocket((c, ws) => {\n` +
            `    ws.onMessage((event) => {\n` +
            `      ws.send('Echo: ' + event.data);\n` +
            `    });\n` +
            `  }));`);
    };
    _router.sse = (path, ..._args) => {
        throw new Error(`router.sse() is deprecated and has been removed.\n\n` +
            `Migration: Use the sse middleware instead:\n\n` +
            `  import { createRouter, sse } from '@agentuity/runtime';\n\n` +
            `  const router = createRouter();\n\n` +
            `  // Before (deprecated):\n` +
            `  // router.sse('${path}', (c) => async (stream) => { ... });\n\n` +
            `  // After:\n` +
            `  router.get('${path}', sse((c, stream) => {\n` +
            `    stream.writeSSE({ data: 'Hello', event: 'message' });\n` +
            `  }));`);
    };
    _router.stream = (path, ..._args) => {
        throw new Error(`router.stream() is deprecated and has been removed.\n\n` +
            `Migration: Use the stream middleware instead:\n\n` +
            `  import { createRouter, stream } from '@agentuity/runtime';\n\n` +
            `  const router = createRouter();\n\n` +
            `  // Before (deprecated):\n` +
            `  // router.stream('${path}', (c) => new ReadableStream({ ... }));\n\n` +
            `  // After:\n` +
            `  router.post('${path}', stream((c) => {\n` +
            `    return new ReadableStream({\n` +
            `      start(controller) {\n` +
            `        controller.enqueue('data\\n');\n` +
            `        controller.close();\n` +
            `      }\n` +
            `    });\n` +
            `  }));`);
    };
    _router.cron = (schedule, ..._args) => {
        throw new Error(`router.cron() is deprecated and has been removed.\n\n` +
            `Migration: Use the cron middleware instead:\n\n` +
            `  import { createRouter, cron } from '@agentuity/runtime';\n\n` +
            `  const router = createRouter();\n\n` +
            `  // Before (deprecated):\n` +
            `  // router.cron('${schedule}', (c) => { ... });\n\n` +
            `  // After:\n` +
            `  router.post('/your-cron-path', cron('${schedule}', (c) => {\n` +
            `    return { status: 'complete' };\n` +
            `  }));`);
    };
    return router;
};
//# sourceMappingURL=router.js.map