import type { InferInput, InferOutput } from '@agentuity/core';
import { type RouteRegistry } from '@agentuity/frontend';
/**
 * Extract route keys from RouteRegistry (e.g., 'GET /users', 'POST /users')
 */
export type RouteKey = keyof RouteRegistry;
/**
 * Extract HTTP method from route key
 */
export type ExtractMethod<TRoute extends RouteKey> = TRoute extends `${infer Method} ${string}` ? Method : never;
/**
 * Extract path from route key given a method
 * E.g., ExtractPath<'GET /users', 'GET'> = '/users'
 */
export type ExtractPath<TRoute extends RouteKey, M extends string> = TRoute extends `${M} ${infer Path}` ? Path : never;
/**
 * Reconstruct the route key from method and path
 * This ensures proper type inference when using {method, path} form
 * E.g., RouteFromMethodPath<'GET', '/users'> = 'GET /users'
 */
export type RouteFromMethodPath<M extends string, P extends string> = Extract<RouteKey, `${M} ${P}`>;
/**
 * Check if a route is a streaming route
 */
export type RouteIsStream<TRoute extends RouteKey> = TRoute extends keyof RouteRegistry ? RouteRegistry[TRoute] extends {
    stream: true;
} ? true : false : false;
/**
 * Extract input type for a specific route
 */
export type RouteInput<TRoute extends RouteKey> = TRoute extends keyof RouteRegistry ? RouteRegistry[TRoute] extends {
    inputSchema: any;
} ? InferInput<RouteRegistry[TRoute]['inputSchema']> : never : never;
/**
 * Extract output type for a specific route
 * Returns void if no outputSchema is defined (e.g., 204 No Content)
 * For streaming routes, returns T[] (accumulated chunks)
 */
export type RouteOutput<TRoute extends RouteKey> = TRoute extends keyof RouteRegistry ? RouteRegistry[TRoute] extends {
    outputSchema: infer TSchema;
    stream: true;
} ? TSchema extends undefined | never ? unknown[] : InferOutput<TSchema>[] : RouteRegistry[TRoute] extends {
    outputSchema: infer TSchema;
} ? TSchema extends undefined | never ? void : InferOutput<TSchema> : void : void;
/**
 * Helper to extract single chunk type from RouteOutput
 */
type RouteChunkType<TRoute extends RouteKey> = TRoute extends keyof RouteRegistry ? RouteRegistry[TRoute] extends {
    outputSchema: infer TSchema;
} ? TSchema extends undefined | never ? unknown : InferOutput<TSchema> : unknown : unknown;
/**
 * Extract path params type for a specific route.
 * Returns the typed params object if the route has path parameters, otherwise never.
 */
export type RoutePathParams<TRoute extends RouteKey> = TRoute extends keyof RouteRegistry ? RouteRegistry[TRoute] extends {
    params: infer P;
} ? P : never : never;
/**
 * Base options shared by all UseAPI configurations (without params handling).
 * This is used as a foundation for both route-form and method/path-form options.
 */
type UseAPIBaseOptions<TRoute extends RouteKey> = {
    /** Additional query parameters */
    query?: URLSearchParams | Record<string, string>;
    /** Additional request headers */
    headers?: Record<string, string>;
    /** Whether to execute the request immediately on mount (default: true for GET, false for others) */
    enabled?: boolean;
    /** How long data stays fresh before refetching (ms, default: 0 - always stale) */
    staleTime?: number;
    /** Refetch interval in milliseconds (default: disabled) */
    refetchInterval?: number;
    /** Callback when request succeeds */
    onSuccess?: (data: RouteOutput<TRoute>) => void;
    /** Callback when request fails */
    onError?: (error: Error) => void;
} & (RouteIsStream<TRoute> extends true ? {
    /** Delimiter for splitting stream chunks (default: \n for JSON lines) */
    delimiter?: string;
    /** Optional transform callback for each chunk */
    onChunk?: (chunk: RouteChunkType<TRoute>) => Promise<RouteChunkType<TRoute>> | RouteChunkType<TRoute>;
} : {}) & (ExtractMethod<TRoute> extends 'GET' ? {
    /** GET requests cannot have input (use query params instead) */
    input?: never;
} : {
    /** Input data for the request (required for POST/PUT/PATCH/DELETE) */
    input?: RouteInput<TRoute>;
});
/**
 * Options for the route form with strict params handling.
 * When TRoute is a single route key (not a union), this correctly requires
 * params only for routes that have path parameters.
 */
type UseAPIRouteFormOptions<TRoute extends RouteKey> = UseAPIBaseOptions<TRoute> & (RoutePathParams<TRoute> extends never ? {} : {
    /** Path parameters for routes with dynamic segments (e.g., { id: '123' } for /users/:id) */
    params: RoutePathParams<TRoute>;
});
/**
 * Options for the method/path form with optional params.
 *
 * When using {method, path}, TRoute is inferred as a union of all matching routes,
 * which causes RoutePathParams<TRoute> to be a union of all params types.
 * This makes params incorrectly required even for routes without path parameters.
 *
 * To fix this (GitHub Issue #417), we make params optional in the method/path form.
 * Users who want strict params enforcement should use the route form instead.
 */
type UseAPIMethodPathFormOptions<TRoute extends RouteKey> = UseAPIBaseOptions<TRoute> & {
    /**
     * Optional path parameters for dynamic segments when using method+path.
     * These are optional because TRoute may be inferred as a union of routes.
     * For strict params typing, use the route form: useAPI({ route: 'GET /users/:id', params: {...} })
     */
    params?: RoutePathParams<TRoute> extends never ? never : RoutePathParams<TRoute>;
};
/**
 * Options with route property (e.g., { route: 'GET /users' })
 * Uses strict params handling - params required only for routes with path parameters.
 */
type UseAPIOptionsWithRoute<TRoute extends RouteKey> = UseAPIRouteFormOptions<TRoute> & {
    /** Route key (e.g., 'GET /users', 'POST /users') */
    route: TRoute;
    /** Method cannot be specified when using route */
    method?: never;
    /** Path cannot be specified when using route */
    path?: never;
};
/**
 * Options with method and path properties (e.g., { method: 'GET', path: '/users' })
 * Uses optional params handling to avoid false positives when TRoute is a union.
 *
 * Note: For full type safety with params, prefer using the route form:
 * useAPI({ route: 'GET /users/:id', params: { id: '123' } })
 */
type UseAPIOptionsWithMethodPath<TRoute extends RouteKey> = UseAPIMethodPathFormOptions<TRoute> & {
    /** HTTP method (e.g., 'GET', 'POST') */
    method: ExtractMethod<TRoute>;
    /** Request path (e.g., '/users') */
    path: string;
    /** Route cannot be specified when using method/path */
    route?: never;
};
/**
 * Options for useAPI hook with smart input typing based on HTTP method
 * Supports either route OR {method, path} but not both
 */
export type UseAPIOptions<TRoute extends RouteKey> = UseAPIOptionsWithRoute<TRoute> | UseAPIOptionsWithMethodPath<TRoute>;
/**
 * Base return value from useAPI hook (without data property or execute)
 */
interface UseAPIResultBase {
    /** Error if request failed */
    error: Error | null;
    /** Whether request is currently in progress */
    isLoading: boolean;
    /** Whether request has completed successfully at least once */
    isSuccess: boolean;
    /** Whether request has failed */
    isError: boolean;
    /** Whether data is currently being fetched (including refetches) */
    isFetching: boolean;
    /** Reset state to initial values */
    reset: () => void;
}
/**
 * Return value for GET requests (auto-executing)
 */
type UseAPIResultQuery<TRoute extends RouteKey> = UseAPIResultBase & (RouteOutput<TRoute> extends void ? {
    /** No data property - route returns no content (204 No Content) */
    data?: never;
    /** Manually trigger a refetch */
    refetch: () => Promise<void>;
} : {
    /** Response data (undefined until loaded) */
    data: RouteOutput<TRoute> | undefined;
    /** Manually trigger a refetch */
    refetch: () => Promise<void>;
});
/**
 * Options that can be passed to invoke() at invocation time.
 * Allows dynamic path parameter substitution when calling mutations.
 */
export type InvokeOptions<TRoute extends RouteKey> = RoutePathParams<TRoute> extends never ? {
    params?: never;
} : {
    params?: RoutePathParams<TRoute>;
};
/**
 * Return value for POST/PUT/PATCH/DELETE requests (manual execution)
 */
type UseAPIResultMutation<TRoute extends RouteKey> = UseAPIResultBase & (RouteOutput<TRoute> extends void ? {
    /** No data property - route returns no content (204 No Content) */
    data?: never;
    /**
     * Invoke the mutation with optional input and options.
     * @param input - Request body (required for routes with inputSchema)
     * @param options - Optional invocation options including dynamic path params
     * @example
     * // Route without input
     * await invoke(undefined, { params: { itemId: '123' } });
     *
     * // Route with input
     * await invoke({ name: 'New Name' }, { params: { itemId: '123' } });
     */
    invoke: RouteInput<TRoute> extends never ? (input?: undefined, options?: InvokeOptions<TRoute>) => Promise<void> : (input: RouteInput<TRoute>, options?: InvokeOptions<TRoute>) => Promise<void>;
} : {
    /** Response data (undefined until invoked) */
    data: RouteOutput<TRoute> | undefined;
    /**
     * Invoke the mutation with optional input and options.
     * @param input - Request body (required for routes with inputSchema)
     * @param options - Optional invocation options including dynamic path params
     * @example
     * // Route without input
     * const result = await invoke(undefined, { params: { itemId: '123' } });
     *
     * // Route with input
     * const result = await invoke({ name: 'New Name' }, { params: { itemId: '123' } });
     */
    invoke: RouteInput<TRoute> extends never ? (input?: undefined, options?: InvokeOptions<TRoute>) => Promise<RouteOutput<TRoute>> : (input: RouteInput<TRoute>, options?: InvokeOptions<TRoute>) => Promise<RouteOutput<TRoute>>;
});
/**
 * Return value from useAPI hook
 * - GET requests: Auto-executes, returns refetch()
 * - POST/PUT/PATCH/DELETE: Manual execution, returns execute(input)
 */
export type UseAPIResult<TRoute extends RouteKey> = ExtractMethod<TRoute> extends 'GET' ? UseAPIResultQuery<TRoute> : UseAPIResultMutation<TRoute>;
/**
 * Type-safe API client hook with TanStack Query-inspired features.
 *
 * Provides automatic type inference for route inputs and outputs based on
 * the RouteRegistry generated from your routes.
 *
 * **Smart input typing:**
 * - GET requests: `input` is `never` (use `query` params instead)
 * - POST/PUT/PATCH/DELETE: `input` is properly typed from route schema
 *
 * @template TRoute - Route key from RouteRegistry (e.g., 'GET /users')
 *
 * @example Simple GET request with string
 * ```typescript
 * const { data, isLoading } = useAPI('GET /users');
 * ```
 *
 * @example GET request with route property
 * ```typescript
 * const { data, isLoading } = useAPI({
 *   route: 'GET /users',
 *   query: { search: 'alice' }, // ✅ Use query params for GET
 *   // input: { ... }, // ❌ TypeScript error - GET cannot have input!
 *   staleTime: 5000,
 * });
 * ```
 *
 * @example GET request with method and path
 * ```typescript
 * const { data, isLoading } = useAPI({
 *   method: 'GET',
 *   path: '/users',
 *   query: { search: 'alice' },
 * });
 * ```
 *
 * @example POST request with manual invocation
 * ```typescript
 * const { invoke, data, isLoading } = useAPI('POST /users');
 *
 * // Invoke manually with input
 * await invoke({ name: 'Alice', email: 'alice@example.com' }); // ✅ Fully typed!
 * // data now contains the created user
 * ```
 *
 * @example GET with query parameters
 * ```typescript
 * const { data } = useAPI({
 *   route: 'GET /search',
 *   query: { q: 'react', limit: '10' },
 * });
 * ```
 *
 * @example DELETE with no response body (204 No Content)
 * ```typescript
 * const { invoke, isSuccess } = useAPI('DELETE /users/:id');
 *
 * // Invoke when needed
 * await invoke({ reason: 'Account closed' });
 *
 * // result.data doesn't exist! ❌ TypeScript error
 * // Use isSuccess instead ✅
 * if (isSuccess) {
 *   console.log('User deleted');
 * }
 * ```
 */
export declare function useAPI<TRoute extends RouteKey>(route: TRoute): UseAPIResult<TRoute>;
export declare function useAPI<TRoute extends RouteKey>(options: UseAPIOptionsWithRoute<TRoute>): UseAPIResult<TRoute>;
export declare function useAPI<M extends string, P extends string>(options: {
    method: M;
    path: P;
} & Omit<UseAPIMethodPathFormOptions<RouteFromMethodPath<M, P>>, 'method' | 'path'>): UseAPIResult<RouteFromMethodPath<M, P>>;
export {};
//# sourceMappingURL=api.d.ts.map