import { agents } from '../agents';
import { ConcurrencyManager } from './concurrency';
const DEFAULT_BACKGROUND_CONFIG = {
    enabled: true,
    defaultConcurrency: 1,
    staleTimeoutMs: 30 * 60 * 1000,
};
export class BackgroundManager {
    ctx;
    config;
    concurrency;
    callbacks;
    tasks = new Map();
    tasksByParent = new Map();
    tasksBySession = new Map();
    notifications = new Map();
    toolCallIds = new Map();
    shuttingDown = false;
    constructor(ctx, config, callbacks) {
        this.ctx = ctx;
        this.config = { ...DEFAULT_BACKGROUND_CONFIG, ...config };
        this.concurrency = new ConcurrencyManager({
            defaultLimit: this.config.defaultConcurrency,
            limits: buildConcurrencyLimits(this.config),
        });
        this.callbacks = callbacks;
    }
    async launch(input) {
        const task = {
            id: createTaskId(),
            parentSessionId: input.parentSessionId,
            parentMessageId: input.parentMessageId,
            description: input.description,
            prompt: input.prompt,
            agent: input.agent,
            status: 'pending',
            queuedAt: new Date(),
            concurrencyGroup: this.getConcurrencyGroup(input.agent),
        };
        this.tasks.set(task.id, task);
        this.indexTask(task);
        if (!this.config.enabled) {
            task.status = 'error';
            task.error = 'Background tasks are disabled.';
            task.completedAt = new Date();
            this.markForNotification(task);
            return task;
        }
        void this.startTask(task);
        return task;
    }
    getTask(id) {
        return this.tasks.get(id);
    }
    getTasksByParent(sessionId) {
        const ids = this.tasksByParent.get(sessionId);
        if (!ids)
            return [];
        return Array.from(ids)
            .map((id) => this.tasks.get(id))
            .filter((task) => Boolean(task));
    }
    findBySession(sessionId) {
        const taskId = this.tasksBySession.get(sessionId);
        if (!taskId)
            return undefined;
        return this.tasks.get(taskId);
    }
    cancel(taskId) {
        const task = this.tasks.get(taskId);
        if (!task || task.status === 'completed' || task.status === 'error') {
            return false;
        }
        task.status = 'cancelled';
        task.completedAt = new Date();
        this.releaseConcurrency(task);
        this.markForNotification(task);
        if (task.sessionId) {
            void this.abortSession(task.sessionId);
            this.callbacks?.onSubagentSessionDeleted?.({ sessionId: task.sessionId });
        }
        return true;
    }
    handleEvent(event) {
        if (!event || typeof event.type !== 'string')
            return;
        this.expireStaleTasks();
        if (event.type === 'message.part.updated') {
            const part = event.properties?.part;
            if (!part)
                return;
            const sessionId = part.sessionID;
            if (!sessionId)
                return;
            const task = this.findBySession(sessionId);
            if (!task)
                return;
            this.updateProgress(task, part);
            return;
        }
        if (event.type === 'session.idle') {
            const sessionId = extractSessionId(event.properties);
            const task = sessionId ? this.findBySession(sessionId) : undefined;
            if (!task)
                return;
            void this.completeTask(task);
            return;
        }
        if (event.type === 'session.error') {
            const sessionId = extractSessionId(event.properties);
            const task = sessionId ? this.findBySession(sessionId) : undefined;
            if (!task)
                return;
            const error = extractError(event.properties);
            this.failTask(task, error ?? 'Session error.');
            return;
        }
    }
    markForNotification(task) {
        const sessionId = task.parentSessionId;
        if (!sessionId)
            return;
        const queue = this.notifications.get(sessionId) ?? new Set();
        queue.add(task.id);
        this.notifications.set(sessionId, queue);
    }
    getPendingNotifications(sessionId) {
        const queue = this.notifications.get(sessionId);
        if (!queue)
            return [];
        return Array.from(queue)
            .map((id) => this.tasks.get(id))
            .filter((task) => Boolean(task));
    }
    clearNotifications(sessionId) {
        this.notifications.delete(sessionId);
    }
    shutdown() {
        this.shuttingDown = true;
        this.concurrency.clear();
        this.notifications.clear();
        try {
            void this.callbacks?.onShutdown?.();
        }
        catch {
            // Ignore shutdown callback errors
        }
    }
    indexTask(task) {
        const parentList = this.tasksByParent.get(task.parentSessionId) ?? new Set();
        parentList.add(task.id);
        this.tasksByParent.set(task.parentSessionId, parentList);
    }
    async startTask(task) {
        if (this.shuttingDown)
            return;
        const concurrencyKey = this.getConcurrencyKey(task.agent);
        task.concurrencyKey = concurrencyKey;
        try {
            await this.concurrency.acquire(concurrencyKey);
        }
        catch (error) {
            if (task.status !== 'cancelled') {
                task.status = 'error';
                task.error = error instanceof Error ? error.message : 'Failed to acquire slot.';
                task.completedAt = new Date();
                this.markForNotification(task);
            }
            return;
        }
        if (task.status === 'cancelled') {
            this.releaseConcurrency(task);
            return;
        }
        try {
            const sessionResult = await this.ctx.client.session.create({
                body: {
                    parentID: task.parentSessionId,
                    title: task.description,
                },
                throwOnError: true,
            });
            const session = unwrapResponse(sessionResult);
            if (!session?.id) {
                throw new Error('Failed to create session.');
            }
            task.sessionId = session.id;
            task.status = 'running';
            task.startedAt = new Date();
            this.tasksBySession.set(session.id, task.id);
            this.callbacks?.onSubagentSessionCreated?.({
                sessionId: session.id,
                parentId: task.parentSessionId,
                title: task.description,
            });
            await this.ctx.client.session.prompt({
                path: { id: session.id },
                body: {
                    agent: task.agent,
                    parts: [{ type: 'text', text: task.prompt }],
                },
                throwOnError: true,
            });
        }
        catch (error) {
            this.failTask(task, error instanceof Error ? error.message : 'Failed to launch background task.');
        }
    }
    updateProgress(task, part) {
        const progress = task.progress ?? this.createProgress();
        progress.lastUpdate = new Date();
        if (part.type === 'tool') {
            const callId = part.callID;
            const toolName = part.tool;
            if (toolName) {
                progress.lastTool = toolName;
            }
            if (callId) {
                const seen = this.toolCallIds.get(task.id) ?? new Set();
                if (!seen.has(callId)) {
                    seen.add(callId);
                    progress.toolCalls += 1;
                    this.toolCallIds.set(task.id, seen);
                }
            }
        }
        if (part.type === 'text' && part.text) {
            progress.lastMessage = part.text;
            progress.lastMessageAt = new Date();
        }
        task.progress = progress;
    }
    createProgress() {
        return {
            toolCalls: 0,
            lastUpdate: new Date(),
        };
    }
    async completeTask(task) {
        if (task.status !== 'running')
            return;
        task.status = 'completed';
        task.completedAt = new Date();
        this.releaseConcurrency(task);
        if (task.sessionId) {
            const result = await this.fetchLatestResult(task.sessionId);
            if (result) {
                task.result = result;
            }
            this.callbacks?.onSubagentSessionDeleted?.({ sessionId: task.sessionId });
        }
        this.markForNotification(task);
        void this.notifyParent(task);
    }
    failTask(task, error) {
        if (task.status === 'completed' || task.status === 'error')
            return;
        task.status = 'error';
        task.error = error;
        task.completedAt = new Date();
        this.releaseConcurrency(task);
        if (task.sessionId) {
            this.callbacks?.onSubagentSessionDeleted?.({ sessionId: task.sessionId });
        }
        this.markForNotification(task);
        void this.notifyParent(task);
    }
    releaseConcurrency(task) {
        if (!task.concurrencyKey)
            return;
        this.concurrency.release(task.concurrencyKey);
        delete task.concurrencyKey;
    }
    async notifyParent(task) {
        if (!task.parentSessionId)
            return;
        const statusLine = task.status === 'completed' ? 'completed' : task.status;
        const message = `[BACKGROUND TASK ${statusLine.toUpperCase()}]

Task: ${task.description}
Agent: ${task.agent}
Status: ${task.status}
Task ID: ${task.id}

Use the agentuity_background_output tool with task_id "${task.id}" to view the result.`;
        try {
            await this.ctx.client.session.prompt({
                path: { id: task.parentSessionId },
                body: {
                    parts: [{ type: 'text', text: message }],
                },
                throwOnError: true,
                responseStyle: 'data',
            });
        }
        catch {
            // Ignore notification errors
        }
    }
    async abortSession(sessionId) {
        try {
            await this.ctx.client.session.abort({
                path: { id: sessionId },
                throwOnError: false,
            });
        }
        catch {
            // Ignore abort errors
        }
    }
    async fetchLatestResult(sessionId) {
        try {
            const messagesResult = await this.ctx.client.session.messages({
                path: { id: sessionId },
                throwOnError: true,
            });
            const messages = unwrapResponse(messagesResult) ?? [];
            const entries = Array.isArray(messages) ? messages : [];
            for (let i = entries.length - 1; i >= 0; i -= 1) {
                const entry = entries[i];
                if (entry?.info?.role !== 'assistant')
                    continue;
                const text = extractTextFromParts(entry.parts ?? []);
                if (text)
                    return text;
            }
        }
        catch {
            return undefined;
        }
        return undefined;
    }
    getConcurrencyGroup(agentName) {
        const model = getAgentModel(agentName);
        if (!model)
            return undefined;
        const provider = model.split('/')[0];
        if (model && this.config.modelConcurrency?.[model] !== undefined) {
            return `model:${model}`;
        }
        if (provider && this.config.providerConcurrency?.[provider] !== undefined) {
            return `provider:${provider}`;
        }
        return undefined;
    }
    getConcurrencyKey(agentName) {
        const group = this.getConcurrencyGroup(agentName);
        return group ?? 'default';
    }
    expireStaleTasks() {
        const now = Date.now();
        for (const task of this.tasks.values()) {
            if (task.status !== 'pending' && task.status !== 'running')
                continue;
            const start = task.startedAt?.getTime() ?? task.queuedAt?.getTime();
            if (!start)
                continue;
            if (now - start > this.config.staleTimeoutMs) {
                this.failTask(task, 'Background task timed out.');
            }
        }
    }
}
function buildConcurrencyLimits(config) {
    const limits = {};
    if (config.providerConcurrency) {
        for (const [provider, limit] of Object.entries(config.providerConcurrency)) {
            limits[`provider:${provider}`] = limit;
        }
    }
    if (config.modelConcurrency) {
        for (const [model, limit] of Object.entries(config.modelConcurrency)) {
            limits[`model:${model}`] = limit;
        }
    }
    return limits;
}
function getAgentModel(agentName) {
    const agent = findAgentDefinition(agentName);
    return agent?.defaultModel;
}
function findAgentDefinition(agentName) {
    return Object.values(agents).find((agent) => agent.displayName === agentName || agent.id === agentName || agent.role === agentName);
}
function createTaskId() {
    return `bg_${Math.random().toString(36).slice(2, 8)}`;
}
function extractSessionId(properties) {
    return (properties?.sessionId ?? properties?.sessionID);
}
function extractError(properties) {
    const error = properties?.error;
    return error?.data?.message ?? (typeof error?.name === 'string' ? error.name : undefined);
}
function extractTextFromParts(parts) {
    const textParts = [];
    for (const part of parts) {
        if (typeof part !== 'object' || part === null)
            continue;
        const typed = part;
        if (typed.type === 'text' && typeof typed.text === 'string') {
            textParts.push(typed.text);
        }
    }
    if (textParts.length === 0)
        return undefined;
    return textParts.join('\n');
}
function unwrapResponse(result) {
    if (typeof result === 'object' && result !== null && 'data' in result) {
        return result.data;
    }
    return result;
}
//# sourceMappingURL=manager.js.map