import type { Logger } from '../types';
export interface PackageRef {
    name: string;
    version: string;
}
/**
 * Mapping from alias key (aliasName@version) to actual package reference.
 * Used to resolve npm aliases (e.g., "tailwind-merge-v2": "npm:tailwind-merge@2.6.0")
 * to their actual package names for accurate malware detection.
 */
export type AliasMap = Map<string, PackageRef>;
export declare function extractDependencies(projectDir: string, logger: Logger): Promise<PackageRef[]>;
export declare function parseBunPmLsOutput(output: string): PackageRef[];
/**
 * Load alias mappings from bun.lock file.
 *
 * The bun.lock file contains a "packages" object where:
 * - Keys are the package names as they appear in node_modules (alias names for aliased packages)
 * - Values are arrays where the first element is "actualPackageName@version"
 *
 * For npm aliases like `"tailwind-merge-v2": "npm:tailwind-merge@2.6.0"`:
 * - Key: "tailwind-merge-v2"
 * - Value[0]: "tailwind-merge@2.6.0"
 *
 * This function builds a map from "aliasName@version" to the actual PackageRef.
 */
export declare function loadAliasMap(projectDir: string, logger: Logger): Promise<AliasMap>;
/**
 * Parse bun.lock file content.
 *
 * bun.lock uses a relaxed JSON format (JSONC) with trailing commas, which
 * standard JSON.parse cannot handle. We need to strip trailing commas before parsing.
 *
 * Structure:
 * {
 *   "lockfileVersion": 1,
 *   "packages": {
 *     "package-name": ["actual-package@version", "", {}, "sha512-..."],
 *     ...
 *   }
 * }
 */
export declare function parseBunLockFile(content: string): BunLockFile | null;
/**
 * Represents the structure of a bun.lock file.
 */
export interface BunLockFile {
    lockfileVersion?: number;
    packages?: Record<string, unknown[]>;
}
/**
 * Resolve npm aliases in the package list to their actual package names.
 *
 * This prevents false positives in malware detection where alias names
 * (e.g., "tailwind-merge-v2") are flagged as suspicious when they're
 * actually legitimate aliases for real packages (e.g., "tailwind-merge").
 */
export declare function resolveAliases(packages: PackageRef[], aliasMap: AliasMap, logger: Logger): PackageRef[];
//# sourceMappingURL=deps.d.ts.map