/**
 * Get the global app instance (stub for backwards compatibility)
 * Returns null in Vite-native architecture
 */
export function getApp() {
    return null;
}
// Re-export event functions from _events
export { fireEvent } from './_events';
import { addEventListener as globalAddEventListener, removeEventListener as globalRemoveEventListener, } from './_events';
import { getLogger, getRouter } from './_server';
/**
 * Create an Agentuity application with lifecycle management.
 *
 * In Vite-native architecture:
 * - This only handles setup/shutdown lifecycle
 * - Router creation and middleware are handled by the generated entry file
 * - Server is managed by Vite (dev) or Bun.serve (prod)
 *
 * @template TAppState - Type of application state from setup()
 *
 * @example
 * ```typescript
 * // app.ts
 * import { createApp } from '@agentuity/runtime';
 *
 * const app = await createApp({
 *   setup: async () => {
 *     const db = await connectDB();
 *     return { db };
 *   },
 *   shutdown: async (state) => {
 *     await state.db.close();
 *   }
 * });
 *
 * // Access state in agents via ctx.app.db
 * ```
 */
export async function createApp(config) {
    // Run setup to get app state
    const state = config?.setup ? await config.setup() : {};
    // Store state and config globally for generated entry file to access
    globalThis.__AGENTUITY_APP_STATE__ = state;
    globalThis.__AGENTUITY_APP_CONFIG__ = config;
    // Store shutdown function for cleanup
    const shutdown = config?.shutdown;
    if (shutdown) {
        globalThis.__AGENTUITY_SHUTDOWN__ = shutdown;
    }
    // Return a logger proxy that lazily resolves to the global logger
    // This is necessary because Vite bundling inlines and reorders module code,
    // causing app.ts to execute before entry file sets the global logger.
    // The proxy ensures logger works correctly when actually used (in handlers/callbacks).
    const logger = {
        trace: (...args) => {
            const gl = getLogger();
            if (gl)
                gl.trace(...args);
        },
        debug: (...args) => {
            const gl = getLogger();
            if (gl)
                gl.debug(...args);
        },
        info: (...args) => {
            const gl = getLogger();
            if (gl)
                gl.info(...args);
            else
                console.log('[INFO]', ...args);
        },
        warn: (...args) => {
            const gl = getLogger();
            if (gl)
                gl.warn(...args);
            else
                console.warn('[WARN]', ...args);
        },
        error: (...args) => {
            const gl = getLogger();
            if (gl)
                gl.error(...args);
            else
                console.error('[ERROR]', ...args);
        },
        fatal: (...args) => {
            const gl = getLogger();
            if (gl)
                return gl.fatal(...args);
            // Fallback: log to console but let the real logger handle exit
            console.error('[FATAL]', ...args);
            throw new Error('Fatal error');
        },
        child: (bindings) => {
            const gl = getLogger();
            return gl ? gl.child(bindings) : logger;
        },
    };
    // Create server info from environment
    const port = process.env.PORT || '3500';
    const server = {
        url: `http://127.0.0.1:${port}`,
    };
    // Get router from global (set by entry file before app.ts import)
    // In dev mode, router may not be available during bundling
    const globalRouter = getRouter();
    if (!globalRouter) {
        throw new Error('Router is not available. Ensure router is initialized before calling createApp(). This typically happens during bundling or when the entry file has not properly set up the router.');
    }
    const router = globalRouter;
    return {
        state,
        shutdown,
        config,
        router,
        server,
        logger,
        addEventListener: globalAddEventListener,
        removeEventListener: globalRemoveEventListener,
    };
}
/**
 * Get the global app state
 * Used by generated entry file and middleware
 */
export function getAppState() {
    return globalThis.__AGENTUITY_APP_STATE__ || {};
}
/**
 * Get the global app config
 * Used by generated entry file for middleware setup
 */
export function getAppConfig() {
    return globalThis.__AGENTUITY_APP_CONFIG__;
}
/**
 * Set the global app config (for testing purposes)
 * @internal
 */
export function setAppConfig(config) {
    if (config === undefined) {
        delete globalThis.__AGENTUITY_APP_CONFIG__;
    }
    else {
        globalThis.__AGENTUITY_APP_CONFIG__ = config;
    }
}
/**
 * Symbol used to store shutdown hooks in globalThis.
 */
const SHUTDOWN_HOOKS_KEY = Symbol.for('@agentuity/runtime:shutdown-hooks');
/**
 * Gets the global shutdown hooks registry.
 */
function getShutdownHooks() {
    const global = globalThis;
    if (!global[SHUTDOWN_HOOKS_KEY]) {
        global[SHUTDOWN_HOOKS_KEY] = [];
    }
    return global[SHUTDOWN_HOOKS_KEY];
}
/**
 * Registers a shutdown hook to be called during graceful shutdown.
 *
 * Hooks are called in reverse order of registration (LIFO) after the
 * app's shutdown callback and agent shutdowns have completed.
 *
 * This is useful for packages like @agentuity/postgres to register
 * their own cleanup logic without requiring explicit wiring in each app.
 *
 * @param hook - The function to call during shutdown
 * @returns A function to unregister the hook
 *
 * @example
 * ```typescript
 * import { registerShutdownHook } from '@agentuity/runtime';
 *
 * // Register a cleanup function
 * const unregister = registerShutdownHook(async () => {
 *   await myResource.close();
 * });
 *
 * // Later, if needed, unregister it
 * unregister();
 * ```
 */
export function registerShutdownHook(hook) {
    const hooks = getShutdownHooks();
    hooks.push(hook);
    return () => {
        const index = hooks.indexOf(hook);
        if (index !== -1) {
            hooks.splice(index, 1);
        }
    };
}
/**
 * Run the global shutdown function and all registered shutdown hooks.
 * Called by generated entry file on cleanup.
 *
 * Shutdown order:
 * 1. App's shutdown callback (if defined)
 * 2. Registered shutdown hooks (in reverse order - LIFO)
 */
export async function runShutdown() {
    // Run app's shutdown callback first
    const shutdown = globalThis.__AGENTUITY_SHUTDOWN__;
    if (shutdown) {
        const state = getAppState();
        await shutdown(state);
    }
    // Run registered shutdown hooks in reverse order (LIFO)
    const hooks = getShutdownHooks();
    for (let i = hooks.length - 1; i >= 0; i--) {
        const hook = hooks[i];
        if (!hook)
            continue;
        try {
            await hook();
        }
        catch {
            // Ignore errors during shutdown hooks
        }
    }
}
//# sourceMappingURL=app.js.map